﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_ContentActionTable.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/fs.h>
#include <nn/fs/fs_SystemData.h>
#include <nn/am/am_Result.h>
#include <mutex>
#include <cstring>
#include <utility>
#include <algorithm>
#include <nn/am/service/am_ServiceDiagnostics.h>
#include <nn/am/service/am_ServiceConfig.h>
#include <nn/fs/fs_SystemDataUpdateEvent.h>
#include <nn/os/os_MultipleWaitApi.h>
#include <nn/os/os_SystemEventApi.h>
#include <nn/os/os_Event.h>

namespace nn { namespace am { namespace service { namespace contentAction {

namespace {

bool IsValidContentActionName(util::string_view s) NN_NOEXCEPT
{
    if (!(s.size() <= 64))
    {
        return false;
    }
    return std::all_of(s.cbegin(), s.cend(), [](char c)
    {
        return false
            || ('0' <= c && c <= '9')
            || ('a' <= c && c <= 'z')
            || ('A' <= c && c <= 'Z')
            || c == '_'
        ;
    });
}

class DataAccessor
{
    NN_DISALLOW_COPY(DataAccessor);
    NN_DISALLOW_MOVE(DataAccessor);
private:

    const char* m_MountName;

    mutable os::SdkMutex m_MutexForFileAccess;
    bool m_Mounted = false;
    char m_MountBuffer[4096];

    mutable os::SdkMutex m_MutexForInvalidate;
    bool m_IsValid = false;

    class UpdateNotifierEvent
    {
    private:

        bool m_IsInitialized = false;
        std::unique_ptr<fs::IEventNotifier> m_pEventNotifier;
        os::SystemEventType m_Event;

    public:

        Result Initialize() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(!m_IsInitialized);
            std::unique_ptr<fs::IEventNotifier> p;
            NN_RESULT_DO(fs::OpenSystemDataUpdateEventNotifier(&p));
            NN_RESULT_DO(p->BindEvent(&m_Event, os::EventClearMode_ManualClear));
            this->m_pEventNotifier = std::move(p);
            this->m_IsInitialized = true;
            NN_RESULT_SUCCESS;
        }

        ~UpdateNotifierEvent() NN_NOEXCEPT
        {
            if (m_IsInitialized)
            {
                os::DestroySystemEvent(&m_Event);
            }
        }

        void InitializeMultiWaitHolder(os::MultiWaitHolderType* p) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_IsInitialized);
            os::InitializeMultiWaitHolder(p, &m_Event);
        }

        void Clear() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_IsInitialized);
            os::ClearSystemEvent(&m_Event);
        }

    };
    UpdateNotifierEvent m_UpdateNotifierEvent;

    auto LockForAccess() NN_NOEXCEPT
    {
        return std::unique_lock<decltype(m_MutexForFileAccess)>{m_MutexForFileAccess};
    }

    auto LockForInvalidate() NN_NOEXCEPT
    {
        return std::unique_lock<decltype(m_MutexForInvalidate)>{m_MutexForInvalidate};
    }

    Result MountImpl() NN_NOEXCEPT
    {
        NN_AM_SERVICE_LOG(seq, "Mount Content Action Table System Data\n");
        NN_SDK_ASSERT(m_MutexForFileAccess.IsLockedByCurrentThread());
        NN_SDK_ASSERT(!m_MutexForInvalidate.IsLockedByCurrentThread());
        size_t size;
        NN_RESULT_DO(fs::QueryMountSystemDataCacheSize(&size, {0x0100000000000827}));
        NN_ABORT_UNLESS(size <= sizeof(m_MountBuffer));
        return fs::MountSystemData(m_MountName, {0x0100000000000827}, m_MountBuffer, sizeof(m_MountBuffer));
    }

    void UnmountImpl() NN_NOEXCEPT
    {
        NN_AM_SERVICE_LOG(seq, "Unmount Content Action Table System Data\n");
        NN_SDK_ASSERT(m_MutexForFileAccess.IsLockedByCurrentThread());
        NN_SDK_ASSERT(!m_MutexForInvalidate.IsLockedByCurrentThread());
        fs::Unmount(m_MountName);
    }

    Result EnsureMountImpl() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_MutexForFileAccess.IsLockedByCurrentThread());
        auto lk = LockForInvalidate();
        while (!(m_Mounted && m_IsValid))
        {
            if (m_Mounted)
            {
                // m_Mounted だった場合、!m_IsValid なので、一旦アンマウントする必要がある
                NN_SDK_ASSERT(!m_IsValid);
                lk.unlock();
                UnmountImpl();
                lk.lock();
                this->m_Mounted = false;
            }
            NN_SDK_ASSERT(!m_Mounted);
            // m_IsValid = true にしてから MountImpl する。
            // MountImpl はそこそこ時間がかかる想定であるため、ロック外で行う。
            // MountImpl 中に Invalidate が呼ばれたら m_IsValid == false になっているため、再度ループが走りリトライされる。
            this->m_IsValid = true;
            lk.unlock();
            NN_RESULT_DO(MountImpl());
            lk.lock();
            this->m_Mounted = true;
        }
        NN_SDK_ASSERT(lk.owns_lock());
        NN_SDK_ASSERT(m_Mounted);
        NN_SDK_ASSERT(m_IsValid);
        NN_RESULT_SUCCESS;
    }

    void InvalidateImpl() NN_NOEXCEPT
    {
        auto lk = LockForInvalidate();
        m_UpdateNotifierEvent.Clear();
        this->m_IsValid = false;
    }

    static Result MakeApplicationIdFilePath(char* buffer, size_t bufferSize, const char* mountName, const char* contentActionName) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidContentActionName(contentActionName));
        auto n = std::snprintf(buffer, bufferSize, "%s:/table/%s/ApplicationId", mountName, contentActionName);
        NN_ABORT_UNLESS(n >= 0);
        NN_RESULT_THROW_UNLESS(static_cast<size_t>(n) < bufferSize, am::ResultInvalidParameter());
        NN_RESULT_SUCCESS;
    }

    static Result ReadHex16String(Bit64* pOut, fs::FileHandle h) NN_NOEXCEPT
    {
        int64_t fileSize;
        NN_RESULT_DO(fs::GetFileSize(&fileSize, h));
        const auto ExpectedSize = 2 + 16;
        NN_RESULT_THROW_UNLESS(fileSize == ExpectedSize, am::ResultInvalidContentActionTableData());
        char buffer[ExpectedSize + 1] = {};
        NN_RESULT_DO(fs::ReadFile(h, 0, buffer, ExpectedSize));
        char* pEnd;
        auto ret = std::strtoull(buffer, &pEnd, 16);
        static_assert(sizeof(Bit64) == sizeof(ret), "");
        NN_RESULT_THROW_UNLESS(pEnd == buffer + ExpectedSize, am::ResultInvalidContentActionTableData());
        *pOut = ret;
        NN_RESULT_SUCCESS;
    }

    Result GetApplicationIdImpl(ncm::ApplicationId* pOut, const char* contentActionName) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_MutexForFileAccess.IsLockedByCurrentThread());
        NN_SDK_ASSERT(m_Mounted);
        char path[128];
        NN_RESULT_DO(MakeApplicationIdFilePath(path, sizeof(path), m_MountName, contentActionName));
        fs::FileHandle h;
        NN_RESULT_TRY(fs::OpenFile(&h, path, fs::OpenMode_Read))
            NN_RESULT_CATCH_CONVERT(fs::ResultPathNotFound, am::ResultContentActionNameNotFound())
        NN_RESULT_END_TRY
        NN_UTIL_SCOPE_EXIT
        {
            fs::CloseFile(h);
        };
        Bit64 value;
        NN_RESULT_DO(ReadHex16String(&value, h));
        *pOut = {value};
        NN_RESULT_SUCCESS;
    }

public:

    explicit DataAccessor(const char* mountName) NN_NOEXCEPT
        : m_MountName(mountName)
    {
        NN_SDK_ASSERT(mountName);
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_UpdateNotifierEvent.Initialize());
    }

    ~DataAccessor() NN_NOEXCEPT
    {
        if (m_Mounted)
        {
            fs::Unmount(m_MountName);
        }
    }

    void Invalidate() NN_NOEXCEPT
    {
        NN_AM_SERVICE_LOG(seq, "Updated Content Action Table System Data\n");
        InvalidateImpl();
    }

    Result GetApplicationId(ncm::ApplicationId* pOut, const char* contentActionName) NN_NOEXCEPT
    {
        auto lkForMount = LockForAccess(); // このシーケンスに入るのは同時に一つだけ
        NN_RESULT_DO(EnsureMountImpl());
        NN_SDK_ASSERT(m_Mounted);
        return GetApplicationIdImpl(pOut, contentActionName);
    }

    void InitializeMultiWaitHolder(os::MultiWaitHolderType* p) NN_NOEXCEPT
    {
        m_UpdateNotifierEvent.InitializeMultiWaitHolder(p);
    }

};

util::optional<DataAccessor> g_pAccessor;

}

void UseSystemData(const char* mountName) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(std::strlen(mountName) <= 16);
    g_pAccessor.emplace(mountName);
}

void NotifySystemDataUpdated() NN_NOEXCEPT
{
    if (g_pAccessor)
    {
        g_pAccessor->Invalidate();
    }
}

Result GetApplicationId(ncm::ApplicationId* pOut, const char* contentActionName) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsValidContentActionName(contentActionName), am::ResultInvalidParameter());
    if (auto pDebugValue = am::service::GetApplicationIdByContentActionNameByDebugConfig(contentActionName))
    {
        *pOut = {*pDebugValue};
        NN_RESULT_SUCCESS;
    }
    if (g_pAccessor)
    {
        return g_pAccessor->GetApplicationId(pOut, contentActionName);
    }
    else
    {
        NN_RESULT_THROW(am::ResultContentActionNameNotFound());
    }
}

void InitializeMultiWaitHolderForSystemDataUpdated(os::MultiWaitHolderType* p) NN_NOEXCEPT
{
    if (g_pAccessor)
    {
        g_pAccessor->InitializeMultiWaitHolder(p);
        return;
    }
    else
    {
        static os::Event g_NeverSignaledEvent{os::EventClearMode_ManualClear};
        os::InitializeMultiWaitHolder(p, g_NeverSignaledEvent.GetBase());
        return;
    }
}

}}}}
