﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/am/am_Shim.h>

#include <nn/applet/applet_Types.h>
#include <nn/applet/applet_Result.h>

#include <nn/ae/ae_AudioControlApi.h>
#include <nn/ae/ae_Result.h>
#include <nn/ae/ae_Types.h>

#include <nn/lr/lr_Result.h>
#include <nn/audio/audio_Applet.h>

namespace nn { namespace ae {

//-----------------------------------------------------------------------------
//  マスターボリューム期待値の設定（メインアプレット向け）
//
void SetExpectedMasterVolume(float saVolume, float laVolume) NN_NOEXCEPT
{
    NN_SDK_ASSERT(audio::ProcessMasterVolumeMin <= saVolume &&
                  saVolume <= audio::ProcessMasterVolumeMax,
        "第 1 引数で指定された volume 値が異常です（volume=%f）", saVolume);
    NN_SDK_ASSERT(audio::ProcessMasterVolumeMin <= laVolume &&
                  laVolume <= audio::ProcessMasterVolumeMax,
        "第 2 引数で指定された volume 値が異常です（volume=%f）", laVolume);

    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetAudioController()->SetExpectedMasterVolume(saVolume, laVolume) );
}


//-----------------------------------------------------------------------------
//  マスターボリューム期待値の取得（メインアプレット向け）
//
void GetExpectedMasterVolume(float* pSaVolume, float* pLaVolume) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetAudioController()->GetMainAppletExpectedMasterVolume(pSaVolume));
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetAudioController()->GetLibraryAppletExpectedMasterVolume(pLaVolume));
}


//-----------------------------------------------------------------------------
//  メインアプレットのマスターボリュームの変更（LA 向け）
//
void ChangeMainAppletMasterVolume(float volume, TimeSpan fadeTime) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetAudioController()->ChangeMainAppletMasterVolume(volume, fadeTime.GetNanoSeconds()));
}


//-----------------------------------------------------------------------------
//  メインアプレット向けのマスターボリューム期待値の取得（LA 向け）
//
float GetMainAppletExpectedMasterVolume() NN_NOEXCEPT
{
    float volume;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetAudioController()->GetMainAppletExpectedMasterVolume(&volume));
    return volume;
}


//-----------------------------------------------------------------------------
//  ライブラリアプレット向けのマスターボリューム期待値の取得（LA 向け）
//
float GetLibraryAppletExpectedMasterVolume() NN_NOEXCEPT
{
    float volume;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetAudioController()->GetLibraryAppletExpectedMasterVolume(&volume));
    return volume;
}


//-----------------------------------------------------------------------------
//  自アプレットより下位の全アプレットの音量を絞る
//
void SetTransparentVolumeRate(float transparentRate) NN_NOEXCEPT
{
    NN_SDK_ASSERT(0.f <= transparentRate && transparentRate <= 1.f);

    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetAudioController()->SetTransparentVolumeRate(transparentRate));
}


//-----------------------------------------------------------------------------

}}  // namespace nn::ae

