﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_NasProblemDetails.h"

#include "../http/account_HttpUtil.h"
#include <nn/account/nas/account_ResultForNas.h>
#include <nn/account/account_ResultPrivate.h>

#include <utility>

#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace account { namespace nas {
namespace
{
#define NN_ACCOUNT_ERRDEF(error, result) \
    {{error}, result}

const struct
{
    NasProblem problem;
    Result result;
} Errors[] = {
    NN_ACCOUNT_ERRDEF("invalid_token",        ResultNasUserResourceStatusInvalidToken()),
    NN_ACCOUNT_ERRDEF("insufficient_scope",   ResultNasUserResourceStatusInsufficientScope()),
    NN_ACCOUNT_ERRDEF("under_maintenance",    ResultNasUserResourceStatusUnderMaintenance()),
};

#undef NN_ACCOUNT_ERRDEF

#define NN_ACCOUNT_ERRDEF(error, detail, result) \
    {{error, detail}, result}

const struct
{
    NasProblemWithDetail problem;
    Result result;
} ErrorsForAuthorizationRequest[] = {
    NN_ACCOUNT_ERRDEF("unauthorized_client", "",                    ResultNasAuthorizationStatusUnauthorizedClient()),
    NN_ACCOUNT_ERRDEF("access_denied", "",                          ResultNasAuthorizationStatusAccessDenied()),
    // NN_ACCOUNT_ERRDEF("access_denied", "user_insufficient_age",     ResultNasAuthorizationStatusAccessDeniedUserInsufficientAge()),
    NN_ACCOUNT_ERRDEF("access_denied", "id_token_hint_invalid",     ResultNasAuthorizationStatusAccessDeniedIdTokenHintInvalid()),
    // NN_ACCOUNT_ERRDEF("access_denied", "user_terms_agreement_required", ResultNasAuthorizationStatusAccessDeniedUserTermsAgreementRequired()),
    NN_ACCOUNT_ERRDEF("access_denied", "user_deleted",              ResultNasAuthorizationStatusAccessDeniedUserDeleted()),
    NN_ACCOUNT_ERRDEF("invalid_scope", "",                          ResultNasAuthorizationStatusInvalidScope()),
    NN_ACCOUNT_ERRDEF("invalid_scope", "scope_token_unknown",       ResultNasAuthorizationStatusInvalidScopeScopeTokenUnknown()),
    NN_ACCOUNT_ERRDEF("invalid_scope", "scope_token_prohibited",    ResultNasAuthorizationStatusInvalidScopeScopeTokenProhibited()),
    NN_ACCOUNT_ERRDEF("server_error", "",                           ResultNasAuthorizationStatusServerError()),
    NN_ACCOUNT_ERRDEF("login_required", "",                         ResultNasAuthorizationStatusLoginRequired()),
    NN_ACCOUNT_ERRDEF("login_required", "user_not_logged_in",       ResultNasAuthorizationStatusLoginRequiredUserNotLoggedIn()),
    NN_ACCOUNT_ERRDEF("login_required", "user_different_from_id_token_hint",    ResultNasAuthorizationStatusLoginRequiredUserDifferentFromIdTokenHint()),
    NN_ACCOUNT_ERRDEF("consent_required", "",                       ResultNasAuthorizationStatusConsentRequired()),
    NN_ACCOUNT_ERRDEF("interaction_required", "",                   ResultNasAuthorizationStatusInteractionRequired()),
    NN_ACCOUNT_ERRDEF("interaction_required", "user_banned",        ResultNasAuthorizationStatusInteractionRequiredUserBanned()),
    NN_ACCOUNT_ERRDEF("interaction_required", "user_suspended",     ResultNasAuthorizationStatusInteractionRequiredUserSuspended()),
    NN_ACCOUNT_ERRDEF("interaction_required", "user_terms_agreement_required",  ResultNasAuthorizationStatusInteractionRequiredUserTermsAgreementRequired()),
    NN_ACCOUNT_ERRDEF("under_maintenance", "",                      ResultNasAuthorizationStatusUnderMaintenance()),
};

const struct
{
    NasProblemWithDetail problem;
    Result result;
} TokenRequestErrors[] = {
    NN_ACCOUNT_ERRDEF("invalid_request", "",                        ResultNasTokenRequestStatusInvalidRequest()),
    NN_ACCOUNT_ERRDEF("invalid_client", "",                         ResultNasTokenRequestStatusInvalidClient()),
    NN_ACCOUNT_ERRDEF("invalid_grant", "",                          ResultNasTokenRequestStatusInvalidGrant()),
    NN_ACCOUNT_ERRDEF("invalid_grant", "user_deleted",              ResultNasTokenRequestStatusInvalidGrantUserDeleted()),
    NN_ACCOUNT_ERRDEF("invalid_grant", "user_banned",               ResultNasTokenRequestStatusInvalidGrantUserBanned()),
    NN_ACCOUNT_ERRDEF("invalid_grant", "user_suspended",            ResultNasTokenRequestStatusInvalidGrantUserSuspended()),
    NN_ACCOUNT_ERRDEF("invalid_grant", "user_withdrawn",            ResultNasTokenRequestStatusInvalidGrantUserWithdrawn()),
    NN_ACCOUNT_ERRDEF("invalid_grant", "user_terms_agreement_required", ResultNasTokenRequestStatusInvalidGrantUserTermsAgreementRequired()),
    NN_ACCOUNT_ERRDEF("invalid_scope", "",                          ResultNasTokenRequestStatusInvalidScope()),
    NN_ACCOUNT_ERRDEF("invalid_scope", "scope_token_unknown",       ResultNasTokenRequestStatusInvalidScopeScopeTokenUnknown()),
    NN_ACCOUNT_ERRDEF("invalid_scope", "scope_token_prohibited",    ResultNasTokenRequestStatusInvalidScopeScopeTokenProhibited()),
    NN_ACCOUNT_ERRDEF("invalid_scope", "scope_token_not_authorized",    ResultNasTokenRequestStatusInvalidScopeScopeTokenNotAuthorized()),
    NN_ACCOUNT_ERRDEF("unauthorized_client", "",                    ResultNasTokenRequestStatusUnauthorizedClient()),
    NN_ACCOUNT_ERRDEF("unsupported_grant_type", "",                 ResultNasTokenRequestStatusUnsupportedGrantType()),
    NN_ACCOUNT_ERRDEF("server_error", "",                           ResultNasTokenRequestStatusServerError()),
    NN_ACCOUNT_ERRDEF("under_maintenance", "",                      ResultNasTokenRequestStatusUnderMaintenance()),
};

#undef NN_ACCOUNT_ERRDEF

template <typename ListType>
std::pair<bool, Result> FindResult(const ListType& list, const NasProblem& problem) NN_NOEXCEPT
{
    for (const auto& e: list)
    {
        if (e.problem == problem)
        {
            return std::pair<bool, Result>(true, e.result);
        }
    }
    return std::pair<bool, Result>(false, ResultNotImplemented());
}

template <typename ListType>
std::pair<bool, Result> FindResult(const ListType& list, const NasProblemWithDetail& problem) NN_NOEXCEPT
{
    for (const auto& e: list)
    {
        if (e.problem == problem)
        {
            return std::pair<bool, Result>(true, e.result);
        }
    }
    for (const auto& e: list)
    {
        if (e.problem.detail[0] == '\0'
            && strncmp(e.problem.error, problem.error, sizeof(e.problem.error)) == 0)
        {
            return std::pair<bool, Result>(true, e.result);
        }
    }
    return std::pair<bool, Result>(false, ResultNotImplemented());
}

} // ~namespace nn::account::baas::<anonymous>

std::pair<bool, Result> FindResult(const NasProblem& problem) NN_NOEXCEPT
{
    return FindResult(Errors, problem);
}
std::pair<bool, Result> FindResultForTokenRequest(const NasProblemWithDetail& problem) NN_NOEXCEPT
{
    return FindResult(TokenRequestErrors, problem);
}

std::pair<bool, Result> FindResultForAuthorizationRequest(const NasProblemWithDetail& problem) NN_NOEXCEPT
{
    return FindResult(ErrorsForAuthorizationRequest, problem);
}

}}} // ~namespace nn::account::baas
