﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_NasLoginAdaptor.h"

#include "../detail/account_CacheUtil.h"
#include "../detail/account_TimeUtil.h"
#include "../http/account_UrlEncoder.h"
#include <nn/account/account_ResultPrivate.h>

#include <cstdlib>
#include <cstring>

namespace nn {
namespace account {
namespace nas {

#define NN_ACCOUNT_PRINT_ENTRY_STATE(e) \
        NN_SDK_LOG("    %s : %s\n", (e)? "+": "-", (e).path)

/* --------------------------------------------------------------------------------------------
    NasPrimaryTokenAdaptor
*/

NasPrimaryTokenAdaptor::NasPrimaryTokenAdaptor(const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    : m_Storage(storage)
    , m_AccessTokenEntry("$.access_token")
    , m_IdTokenEntry("$.id_token")
    , m_RefreshTokenEntry("$.refresh_token")
    , m_ExpirationEntry("$.expires_in")
    , m_AccessTokenCacheId(detail::InvalidUuid)
    , m_IdTokenCacheId(detail::InvalidUuid)
    , m_RefreshTokenCacheId(detail::InvalidUuid)
    , m_ExpirationAbs(0)
    , m_IsAdapted(false)
{
}
NasPrimaryTokenAdaptor::~NasPrimaryTokenAdaptor() NN_NOEXCEPT
{
    if (m_AccessTokenCacheId)
    {
        detail::CacheUtil::DeleteCacheFile(m_AccessTokenCacheId, m_Storage);
    }
    if (m_IdTokenCacheId)
    {
        detail::CacheUtil::DeleteCacheFile(m_IdTokenCacheId, m_Storage);
    }
    if (m_RefreshTokenCacheId)
    {
        detail::CacheUtil::DeleteCacheFile(m_RefreshTokenCacheId, m_Storage);
    }
}
Result NasPrimaryTokenAdaptor::PullPrimaryTokens(NasCredentialCache* pOutCredentialCache, NasAccessTokenCache* pOutAccessTokenCache) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsAdapted);
    NN_SDK_ASSERT(m_AccessTokenCacheId);
    NN_SDK_ASSERT(m_RefreshTokenCacheId);
    NN_SDK_ASSERT(m_IdTokenCacheId);
    pOutCredentialCache->refreshTokenCacheId = m_RefreshTokenCacheId;
    pOutCredentialCache->idTokenCacheId = m_IdTokenCacheId;
    pOutAccessTokenCache->expiration = m_ExpirationAbs;
    pOutAccessTokenCache->cacheId = m_AccessTokenCacheId;
    m_RefreshTokenCacheId = detail::InvalidUuid;
    m_AccessTokenCacheId = detail::InvalidUuid;
    m_IdTokenCacheId = detail::InvalidUuid;
    NN_RESULT_SUCCESS;
}
Result NasPrimaryTokenAdaptor::AdaptImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_IsAdapted);
    if (!(m_AccessTokenEntry && m_IdTokenEntry && m_RefreshTokenEntry && m_ExpirationEntry))
    {
        NN_SDK_LOG(
            "[nn::account] -----------------------------------------------\n"
            "  Error: NAS Token Request Failed\n");
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_AccessTokenEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_IdTokenEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_RefreshTokenEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_ExpirationEntry);
        NN_RESULT_THROW(ResultNasDataBroken());
    }
    m_IsAdapted = true;
    NN_RESULT_SUCCESS;
}
bool NasPrimaryTokenAdaptor::UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
{
    if (m_ExpirationEntry.CanAccept(jsonPath))
    {
        m_ExpirationAbs = value + detail::GetUptimeInSeconds();
        m_ExpirationEntry.MarkAccepted();
        return true;
    }
    return false;
}
bool NasPrimaryTokenAdaptor::UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    if (m_AccessTokenEntry.CanAccept(jsonPath))
    {
        if (valueLength <= static_cast<int>(detail::NasAccessTokenSizeMax))
        {
            detail::Uuid cacheId;
            auto r = detail::CacheUtil::StoreCacheFile(&cacheId, value, valueLength, m_Storage);
            if (SetIoResult(r).IsSuccess())
            {
                m_AccessTokenCacheId = cacheId;
                m_AccessTokenEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_IdTokenEntry.CanAccept(jsonPath))
    {
        if (valueLength <= static_cast<int>(detail::NasIdTokenSizeMax))
        {
            detail::Uuid cacheId;
            auto r = detail::CacheUtil::StoreCacheFile(&cacheId, value, valueLength, m_Storage);
            if (SetIoResult(r).IsSuccess())
            {
                m_IdTokenCacheId = cacheId;
                m_IdTokenEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_RefreshTokenEntry.CanAccept(jsonPath))
    {
        if (valueLength <= static_cast<int>(detail::NasRefreshTokenSizeMax))
        {
            detail::Uuid cacheId;
            auto r = detail::CacheUtil::StoreCacheFile(&cacheId, value, valueLength, m_Storage);
            if (SetIoResult(r).IsSuccess())
            {
                m_RefreshTokenCacheId = cacheId;
                m_RefreshTokenEntry.MarkAccepted();
            }
        }
        return true;
    }
    return false;
}

/* --------------------------------------------------------------------------------------------
    NasAccessTokenAdaptor
*/

NasAccessTokenAdaptor::NasAccessTokenAdaptor(const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    : m_Storage(storage)
    , m_AccessTokenEntry("$.access_token")
    , m_ExpirationEntry("$.expires_in")
    , m_AccessTokenCacheId(detail::InvalidUuid)
    , m_ExpirationAbs(0)
    , m_IsAdapted(false)
{
}
NasAccessTokenAdaptor::~NasAccessTokenAdaptor() NN_NOEXCEPT
{
    if (m_AccessTokenCacheId)
    {
        detail::CacheUtil::DeleteCacheFile(m_AccessTokenCacheId, m_Storage);
    }
}
Result NasAccessTokenAdaptor::PullAccessToken(NasAccessTokenCache* pOutAccessTokenCache) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsAdapted);
    NN_SDK_ASSERT(m_AccessTokenCacheId);
    pOutAccessTokenCache->expiration = m_ExpirationAbs;
    pOutAccessTokenCache->cacheId = m_AccessTokenCacheId;
    m_AccessTokenCacheId = detail::InvalidUuid;
    NN_RESULT_SUCCESS;
}
Result NasAccessTokenAdaptor::AdaptImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_IsAdapted);
    if (!(m_AccessTokenEntry && m_ExpirationEntry))
    {
        NN_SDK_LOG(
            "[nn::account] -----------------------------------------------\n"
            "  Error: NAS Token Request Failed\n");
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_AccessTokenEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_ExpirationEntry);
        NN_RESULT_THROW(ResultNasDataBroken());
    }
    m_IsAdapted = true;
    NN_RESULT_SUCCESS;
}
bool NasAccessTokenAdaptor::UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
{
    if (m_ExpirationEntry.CanAccept(jsonPath))
    {
        m_ExpirationAbs = value + detail::GetUptimeInSeconds();
        m_ExpirationEntry.MarkAccepted();
        return true;
    }
    return false;
}
bool NasAccessTokenAdaptor::UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    if (m_AccessTokenEntry.CanAccept(jsonPath))
    {
        if (valueLength <= static_cast<int>(detail::NasAccessTokenSizeMax))
        {
            detail::Uuid cacheId;
            auto r = detail::CacheUtil::StoreCacheFile(&cacheId, value, valueLength, m_Storage);
            if (SetIoResult(r).IsSuccess())
            {
                m_AccessTokenCacheId = cacheId;
                m_AccessTokenEntry.MarkAccepted();
            }
        }
        return true;
    }
    return false;
}

/* --------------------------------------------------------------------------------------------
    NasAuthorizationAdaptorBase
*/

NasAuthorizationErrorAdaptor::NasAuthorizationErrorAdaptor(const char (&redirectUri)[detail::NasCallbackUriLengthMax + 1], const State& state) NN_NOEXCEPT
    : ParserBase(redirectUri)
    , m_State(state)
    , m_StateEntry("state")
    , m_ErrorEntry("error")
    , m_ErrorDetailEntry("error_detail")
{
    NN_SDK_ASSERT(strnlen(redirectUri, sizeof(redirectUri)) <= sizeof(redirectUri));
    NN_SDK_ASSERT(strnlen(state.data, sizeof(state.data)) <= sizeof(state.data));
    m_Problem.error[0] = m_Problem.detail[0] = '\0';
}
bool NasAuthorizationErrorAdaptor::UpdateError(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT
{
    if (m_StateEntry.CanAccept(key, keyLength))
    {
        if (http::CompareEncodedStringPartially(m_State.data, sizeof(m_State.data), value, valueLength))
        {
            m_StateEntry.MarkAccepted();
        }
        return true;
    }
    else if (m_ErrorEntry.CanAccept(key, keyLength))
    {
        if (valueLength < sizeof(m_Problem.error))
        {
            strncpy(m_Problem.error, value, valueLength);
            m_Problem.error[valueLength] = '\0';
            m_ErrorEntry.MarkAccepted();
        }
        return true;
    }
    else if (m_ErrorDetailEntry.CanAccept(key, keyLength))
    {
        if (valueLength < sizeof(m_Problem.detail))
        {
            strncpy(m_Problem.detail, value, valueLength);
            m_Problem.detail[valueLength] = '\0';
            m_ErrorDetailEntry.MarkAccepted();
        }
        return true;
    }
    return false;
}
Result NasAuthorizationErrorAdaptor::AdaptError() NN_NOEXCEPT
{
    if (!(m_StateEntry && !m_ErrorEntry))
    {
        if (!m_StateEntry)
        {
            // 意図した state のレスポンスが返って来なかった
            NN_SDK_LOG("[nn::account] Error: NAS returned state is invalid\n");
            NN_RESULT_THROW(ResultNasDataBroken());
        }

        NN_SDK_LOG(
            "[nn::account] -----------------------------------------------\n"
            "  Error: NAS Authorization Request Failed: '%s' (status=%s)\n",
            m_Problem.error, m_Problem.detail);
        auto r = FindResultForAuthorizationRequest(m_Problem);
        if (r.first)
        {
            NN_RESULT_THROW(r.second);
        }
        NN_RESULT_THROW(ResultNasDataBroken());
    }
    NN_RESULT_SUCCESS;
}

/* --------------------------------------------------------------------------------------------
    NasServiceAuthorizationAdaptor
*/

NasServiceAuthorizationAdaptor::NasServiceAuthorizationAdaptor(
    const char (&redirectUri)[detail::NasCallbackUriLengthMax + 1], const State& state,
    const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    : NasAuthorizationErrorAdaptor(redirectUri, state)
    , m_Storage(storage)
    , m_CodeEntry("code")
    , m_CodeCacheId(detail::InvalidUuid)
    , m_IoResult(ResultSuccess())
    , m_IsAdapted(false)
{
}
NasServiceAuthorizationAdaptor::~NasServiceAuthorizationAdaptor() NN_NOEXCEPT
{
    if (m_CodeCacheId)
    {
        detail::CacheUtil::DeleteCacheFile(m_CodeCacheId, m_Storage);
    }
}
void NasServiceAuthorizationAdaptor::UpdateImplImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT
{
    if (!CanContinue())
    {
        return;
    }
    else if (m_CodeEntry.CanAccept(key, keyLength))
    {
        if (valueLength <= detail::NasAuthorizationCodeSizeMax)
        {
            detail::Uuid cacheId;
            m_IoResult = detail::CacheUtil::StoreCacheFile(&cacheId, value, valueLength, m_Storage);
            if (m_IoResult.IsSuccess())
            {
                m_CodeCacheId = cacheId;
                m_CodeEntry.MarkAccepted();
            }
        }
    }
}
Result NasServiceAuthorizationAdaptor::AdaptImpl() NN_NOEXCEPT
{
    NN_RESULT_DO(m_IoResult);
    // データ取得時にエラーチェックするため、Adapt は常に成功させる
    m_IsAdapted = true;
    NN_RESULT_SUCCESS;
}
Result NasServiceAuthorizationAdaptor::PullAuthorizationCode(detail::Uuid* pOut) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsAdapted);
    NN_RESULT_THROW_UNLESS(m_CodeEntry, ResultNasDataBroken());
    NN_SDK_ASSERT(m_CodeCacheId);
    *pOut = m_CodeCacheId;
    m_CodeCacheId = detail::InvalidUuid;
    NN_RESULT_SUCCESS;
}

/* --------------------------------------------------------------------------------------------
    NasApplicationAuthorizationAdaptor
*/

NasApplicationAuthorizationAdaptor::NasApplicationAuthorizationAdaptor(
    NasApplicationAuthorization* pOutAuthorization,
    const char (&redirectUri)[detail::NasCallbackUriLengthMax + 1], const State& state) NN_NOEXCEPT
    : NasAuthorizationErrorAdaptor(redirectUri, state)
    , m_CodeEntry("code")
    , m_IdTokenEntry("id_token")
    , m_pOutAuthorization(pOutAuthorization)
{
}
void NasApplicationAuthorizationAdaptor::UpdateImplImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT
{
    if (m_CodeEntry.CanAccept(key, keyLength))
    {
        if (valueLength <= sizeof(m_pOutAuthorization->code))
        {
            std::strncpy(m_pOutAuthorization->code, value, valueLength);
            if (valueLength < sizeof(m_pOutAuthorization->code))
            {
                std::memset(&m_pOutAuthorization->code[valueLength], 0x00, sizeof(m_pOutAuthorization->code) - valueLength);
            }
            m_CodeEntry.MarkAccepted();
        }
    }
    else if (m_IdTokenEntry.CanAccept(key, keyLength))
    {
        if (valueLength <= sizeof(m_pOutAuthorization->idToken))
        {
            std::strncpy(m_pOutAuthorization->idToken, value, valueLength);
            if (valueLength < sizeof(m_pOutAuthorization->idToken))
            {
                std::memset(&m_pOutAuthorization->idToken[valueLength], 0x00, sizeof(m_pOutAuthorization->idToken) - valueLength);
            }
            m_IdTokenEntry.MarkAccepted();
        }
    }
}
Result NasApplicationAuthorizationAdaptor::AdaptImpl() NN_NOEXCEPT
{
    if (!(m_CodeEntry && m_IdTokenEntry))
    {
        NN_SDK_LOG(
            "[nn::account] -----------------------------------------------\n"
            "  Error: NAS Authorization Request Failed\n");
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_CodeEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_IdTokenEntry);

        NN_RESULT_THROW(ResultNasDataBroken());
    }
    NN_RESULT_SUCCESS;
}

/* --------------------------------------------------------------------------------------------
    NasIdTokenAdaptor
*/

NasIdTokenAdaptor::NasIdTokenAdaptor(
    char(&idToken)[detail::NasIdTokenSizeMax],
    const char(&redirectUri)[detail::NasCallbackUriLengthMax + 1], const State& state,
    const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    : NasAuthorizationErrorAdaptor(redirectUri, state)
    , m_Storage(storage)
    , m_IdTokenEntry("id_token")
    , m_IdToken(idToken)
{
}
void NasIdTokenAdaptor::UpdateImplImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT
{
    if (m_IdTokenEntry.CanAccept(key, keyLength))
    {
        if (valueLength <= sizeof(m_IdToken))
        {
            std::strncpy(m_IdToken, value, sizeof(m_IdToken));
            m_IdTokenEntry.MarkAccepted();
        }
    }
}
Result NasIdTokenAdaptor::AdaptImpl() NN_NOEXCEPT
{
    if (!m_IdTokenEntry)
    {
        NN_SDK_LOG(
            "[nn::account] -----------------------------------------------\n"
            "  Error: NAS Guest Login for ID token Failed\n");
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_IdTokenEntry);
        NN_RESULT_THROW(ResultNasDataBroken());
    }
    NN_RESULT_SUCCESS;
}

} // ~namespace nn::account::nas
}
}
