﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "account_UuidUtil.h"
#include <nn/account/account_Types.h>
#include <nn/account/account_NintendoAccountTypes.h>

namespace nn { namespace account { namespace detail {

// 諸々のパスを解決する
class PathUtil
{
public:
    // UserRegistry 用
    static const char* GetUserListPath(char* buffer, size_t bufferSize, const char* rootPath) NN_NOEXCEPT;
    // ProfileStorage 用
    static const char* GetProfileDirectoryPath(char* buffer, size_t bufferSize, const char* rootPath) NN_NOEXCEPT;
    static const char* GetProfileListPath(char* buffer, size_t bufferSize, const char* rootPath) NN_NOEXCEPT;
    static const char* GetProfileImagePath(char* buffer, size_t bufferSize, const Uid& uid, const char* rootPath) NN_NOEXCEPT;
    // BaaS 用
    static const char* GetBaasDirectoryPath(char* buffer, size_t bufferSize, const char* rootPath) NN_NOEXCEPT;
    static const char* GetBaasUserFilePath(char* buffer, size_t bufferSize, const Uid& uid, const char* rootPath) NN_NOEXCEPT;
    // NAS 用
    static const char* GetNasDirectoryPath(char* buffer, size_t bufferSize, const char* rootPath) NN_NOEXCEPT;
    static const char* GetNasUserInfoFilePath(char* buffer, size_t bufferSize, const NintendoAccountId& naId, const char* rootPath) NN_NOEXCEPT;
    static const char* GetNasUserIdTokenFilePath(char* buffer, size_t bufferSize, const NintendoAccountId& naId, const char* rootPath) NN_NOEXCEPT;
    static const char* GetNasUserRefreshTokenFilePath(char* buffer, size_t bufferSize, const NintendoAccountId& naId, const char* rootPath) NN_NOEXCEPT;
    static const char* GetNasUserResourceCacheFilePath(char* buffer, size_t bufferSize, const NintendoAccountId& naId, const char* rootPath) NN_NOEXCEPT;
    // キャッシュ用
    static const char* GetCacheDirectoryPath(char* buffer, size_t bufferSize, const char* rootPath) NN_NOEXCEPT;
    static const char* GetCachePath(char* buffer, size_t bufferSize, const Uuid& cacheId, const char* rootPath) NN_NOEXCEPT;
};

}}} // ~namespace nn::account::detail

#include <nn/nn_SdkAssert.h>
#include <nn/util/util_FormatString.h>

namespace nn { namespace account { namespace detail {

/*  [アカウントシステムのセーブデータボリュームのファイルツリー]

    <volume>:/
    |-- registry.dat    -- 登録済みユーザーの順序付き UID リスト
    |-- avators/        -- ユーザーのプロフィール用ディレクトリ
    | |-- profiles.dat  -- 全ユーザープロフィールの順序なしリスト
    | |-- <uid>.jpg     -- ユーザー <uid> のプロフィール画像
    | `-- :
    |-- baas/           -- BaaS 関係の情報
    | |-- <uid>.dat     -- ユーザーごとのログイン情報
    | `-- :
    |-- nas/                    -- NAS 関係の情報
    | |-- <naid>.dat            -- NA 認証情報
    | |-- <naid>_id.token       -- NA IDトークン
    | |-- <naid>_refresh.token  -- NA リフレッシュトークン
    | |-- <naid>_user.json      -- NA ユーザーリソース
    | `-- :
    `-- cache/          -- UL/DL 用データのキャッシュ用ディレクトリ
      |-- <uuid>        -- キャッシュデータ (拡張子なし)
      `-- :
 */

#define NN_ACCOUNT_USER_LIST_NAME       "registry.dat"
#define NN_ACCOUNT_AVATOR_DIR           "avators/"
#define NN_ACCOUNT_AVATOR_LIST_NAME     "profiles.dat"
#define NN_ACCOUNT_AVATOR_EXT           ".jpg"
#define NN_ACCOUNT_BAAS_DIR             "baas/"
#define NN_ACCOUNT_BAAS_USER_FILE_EXT   ".dat"
#define NN_ACCOUNT_NAS_DIR              "nas/"
#define NN_ACCOUNT_NAS_INFO_FILE_EXT    ".dat"
#define NN_ACCOUNT_NAS_ID_TOKEN_FILE_EXT "_id.token"
#define NN_ACCOUNT_NAS_REFRESH_TOKEN_FILE_EXT "_refresh.token"
#define NN_ACCOUNT_NAS_USER_RESOURCE_CACHE_FILE_EXT "_user.json"
#define NN_ACCOUNT_CACHE_DIR            "cache/"
/* --------------------------------------------------------------------------------------------
    UserRegistry
*/
inline const char* PathUtil::GetUserListPath(char* buffer, size_t bufferSize, const char* rootPath) NN_NOEXCEPT
{
    auto e = util::SNPrintf(
        buffer, bufferSize,
        "%s/" NN_ACCOUNT_USER_LIST_NAME,
        rootPath);
    NN_SDK_ASSERT(e < static_cast<int>(bufferSize));
    NN_UNUSED(e);
    return buffer;
}
/* --------------------------------------------------------------------------------------------
    UserProfile
*/
inline const char* PathUtil::GetProfileDirectoryPath(char* buffer, size_t bufferSize, const char* rootPath) NN_NOEXCEPT
{
    auto e = util::SNPrintf(
        buffer, bufferSize,
        "%s/" NN_ACCOUNT_AVATOR_DIR,
        rootPath);
    NN_SDK_ASSERT(e < static_cast<int>(bufferSize));
    NN_UNUSED(e);
    return buffer;
}
inline const char* PathUtil::GetProfileListPath(char* buffer, size_t bufferSize, const char* rootPath) NN_NOEXCEPT
{
    auto e = util::SNPrintf(
        buffer, bufferSize,
        "%s/" NN_ACCOUNT_AVATOR_DIR NN_ACCOUNT_AVATOR_LIST_NAME,
        rootPath);
    NN_SDK_ASSERT(e < static_cast<int>(bufferSize));
    NN_UNUSED(e);
    return buffer;
}
inline const char* PathUtil::GetProfileImagePath(char* buffer, size_t bufferSize, const Uid& uid, const char* rootPath) NN_NOEXCEPT
{
    auto e = util::SNPrintf(
        buffer, bufferSize,
        "%s/" NN_ACCOUNT_AVATOR_DIR "%s" NN_ACCOUNT_AVATOR_EXT,
        rootPath, ConvertToUuid(uid).ToString().GetInnerValue());
    NN_SDK_ASSERT(e < static_cast<int>(bufferSize));
    NN_UNUSED(e);
    return buffer;
}
/* --------------------------------------------------------------------------------------------
    Baas
*/
inline const char* PathUtil::GetBaasDirectoryPath(char* buffer, size_t bufferSize, const char* rootPath) NN_NOEXCEPT
{
    auto e = util::SNPrintf(
        buffer, bufferSize,
        "%s/" NN_ACCOUNT_BAAS_DIR,
        rootPath);
    NN_SDK_ASSERT(e < static_cast<int>(bufferSize));
    NN_UNUSED(e);
    return buffer;
}
inline const char* PathUtil::GetBaasUserFilePath(char* buffer, size_t bufferSize, const Uid& uid, const char* rootPath) NN_NOEXCEPT
{
    auto e = util::SNPrintf(
        buffer, bufferSize,
        "%s/" NN_ACCOUNT_BAAS_DIR "%s" NN_ACCOUNT_BAAS_USER_FILE_EXT,
        rootPath, ConvertToUuid(uid).ToString().GetInnerValue());
    NN_SDK_ASSERT(e < static_cast<int>(bufferSize));
    NN_UNUSED(e);
    return buffer;
}
/* --------------------------------------------------------------------------------------------
    NAS
*/
inline const char* PathUtil::GetNasDirectoryPath(char* buffer, size_t bufferSize, const char* rootPath) NN_NOEXCEPT
{
    auto e = util::SNPrintf(
        buffer, bufferSize,
        "%s/" NN_ACCOUNT_NAS_DIR,
        rootPath);
    NN_SDK_ASSERT(e < static_cast<int>(bufferSize));
    NN_UNUSED(e);
    return buffer;
}
inline const char* PathUtil::GetNasUserInfoFilePath(char* buffer, size_t bufferSize, const NintendoAccountId& naId, const char* rootPath) NN_NOEXCEPT
{
    auto e = util::SNPrintf(
        buffer, bufferSize,
        "%s/" NN_ACCOUNT_NAS_DIR "%016llx" NN_ACCOUNT_NAS_INFO_FILE_EXT,
        rootPath, naId.id);
    NN_SDK_ASSERT(e < static_cast<int>(bufferSize));
    NN_UNUSED(e);
    return buffer;
}
inline const char* PathUtil::GetNasUserIdTokenFilePath(char* buffer, size_t bufferSize, const NintendoAccountId& naId, const char* rootPath) NN_NOEXCEPT
{
    auto e = util::SNPrintf(
        buffer, bufferSize,
        "%s/" NN_ACCOUNT_NAS_DIR "%016llx" NN_ACCOUNT_NAS_ID_TOKEN_FILE_EXT,
        rootPath, naId.id);
    NN_SDK_ASSERT(e < static_cast<int>(bufferSize));
    NN_UNUSED(e);
    return buffer;
}
inline const char* PathUtil::GetNasUserRefreshTokenFilePath(char* buffer, size_t bufferSize, const NintendoAccountId& naId, const char* rootPath) NN_NOEXCEPT
{
    auto e = util::SNPrintf(
        buffer, bufferSize,
        "%s/" NN_ACCOUNT_NAS_DIR "%016llx" NN_ACCOUNT_NAS_REFRESH_TOKEN_FILE_EXT,
        rootPath, naId.id);
    NN_SDK_ASSERT(e < static_cast<int>(bufferSize));
    NN_UNUSED(e);
    return buffer;
}
inline const char* PathUtil::GetNasUserResourceCacheFilePath(char* buffer, size_t bufferSize, const NintendoAccountId& naId, const char* rootPath) NN_NOEXCEPT
{
    auto e = util::SNPrintf(
        buffer, bufferSize,
        "%s/" NN_ACCOUNT_NAS_DIR "%016llx" NN_ACCOUNT_NAS_USER_RESOURCE_CACHE_FILE_EXT,
        rootPath, naId.id);
    NN_SDK_ASSERT(e < static_cast<int>(bufferSize));
    NN_UNUSED(e);
    return buffer;
}

/* --------------------------------------------------------------------------------------------
    Cache
*/
inline const char* PathUtil::GetCacheDirectoryPath(char* buffer, size_t bufferSize, const char* rootPath) NN_NOEXCEPT
{
    auto e = util::SNPrintf(
        buffer, bufferSize,
        "%s/" NN_ACCOUNT_CACHE_DIR,
        rootPath);
    NN_SDK_ASSERT(e < static_cast<int>(bufferSize), rootPath);
    NN_UNUSED(e);
    return buffer;
}
inline const char* PathUtil::GetCachePath(char* buffer, size_t bufferSize, const Uuid& uuid, const char* rootPath) NN_NOEXCEPT
{
    auto e = util::SNPrintf(
        buffer, bufferSize,
        "%s/" NN_ACCOUNT_CACHE_DIR "%s",
        rootPath, uuid.ToString().GetInnerValue());
    NN_SDK_ASSERT(e < static_cast<int>(bufferSize));
    NN_UNUSED(e);
    return buffer;
}
#undef NN_ACCOUNT_USER_LIST_NAME
#undef NN_ACCOUNT_AVATOR_DIR
#undef NN_ACCOUNT_AVATOR_LIST_NAME
#undef NN_ACCOUNT_AVATOR_EXT
#undef NN_ACCOUNT_CACHE_DIR

}}} // ~namespace nn::account::detail
