﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_HipcProxyClientResource.h"

#include <curl/curl.h>
#include <new>
#include <type_traits>

#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/account/account_RuntimeResource.h>
#include <nn/account/account_RuntimeUtil.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_Thread.h>
#include <nn/time/time_Api.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace account { namespace detail {
namespace {
class Thread
{
private:
    os::Event m_Terminator;
    os::ThreadType m_Thread;
    std::function<void(void)> m_Function;

    static void ThreadFunction(void* p)
    {
        auto& pt = *reinterpret_cast<Thread*>(p);
        pt.m_Function();
    }

public:
    Thread(
        std::function<void(void)> function,
        void* stack, size_t stackSize,
        int priority, const char* name) NN_NOEXCEPT
        : m_Terminator(os::EventClearMode_ManualClear)
        , m_Function(function)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&m_Thread, ThreadFunction, this, stack, stackSize, priority));
        os::SetThreadNamePointer(&m_Thread, name);
    }
    ~Thread() NN_NOEXCEPT
    {
        os::DestroyThread(&m_Thread);
    }
    void Start() NN_NOEXCEPT
    {
        os::StartThread(&m_Thread);
    }
    void Stop() NN_NOEXCEPT
    {
        m_Terminator.Signal();
        os::WaitThread(&m_Thread);
    }
};

class ServiceInitializer
{
private:
    template <size_t Size>
    class Allocator
        : public LargeBufferAllocator
    {
    private:
        typename std::aligned_storage<Size, std::alignment_of<std::max_align_t>::value>::type m_Buffer;
    public:
        Allocator() NN_NOEXCEPT
            : LargeBufferAllocator(&m_Buffer, sizeof(m_Buffer))
        {
        }
    };

    static void ExecutionThreadFunction(ServiceInitializer& service) NN_NOEXCEPT;
    ExecutionResource GetExecutionResource() NN_NOEXCEPT;
    ExecutionManager<account::Executor, ExecutionResource>& GetManagaerForApplicationsRef() NN_NOEXCEPT;
    ExecutionManager<account::Executor, ExecutionResource>& GetManagaerForSystemsRef() NN_NOEXCEPT;

private:
    DefaultServiceResource m_ServiceResource;
    ExecutionManager<account::Executor, ExecutionResource> m_ManagerApp;
    ExecutionManager<account::Executor, ExecutionResource> m_ManagerSys;

    char m_Buffer[RequiredBufferSize];
    Allocator<512 * 1024> m_Allocator;

    bool m_IsStarted {false};
    Thread m_Thread;

public:
    ServiceInitializer(void* threadStack, size_t threadStackSize) NN_NOEXCEPT;
    ~ServiceInitializer() NN_NOEXCEPT;

    void Start() NN_NOEXCEPT;
    DefaultServiceResource& GetServiceResourceRef() NN_NOEXCEPT;
};

void ServiceInitializer::ExecutionThreadFunction(ServiceInitializer& service) NN_NOEXCEPT
{
    auto executionResource = service.GetExecutionResource();
    auto& managerApp = service.GetManagaerForApplicationsRef();
    auto& managerSys = service.GetManagaerForSystemsRef();

    os::MultiWaitType multiWait;
    os::InitializeMultiWait(&multiWait);

    // CURL ハンドルのリフレッシュ
    os::TimerEvent curlTimer(os::EventClearMode_ManualClear);
    os::MultiWaitHolderType curlTimerHolder;
    os::InitializeMultiWaitHolder(&curlTimerHolder, curlTimer.GetBase());
    os::LinkMultiWaitHolder(&multiWait, &curlTimerHolder);

    /* 解放 -------------------------------------------------- */
    NN_UTIL_SCOPE_EXIT {
        os::UnlinkMultiWaitHolder(&curlTimerHolder);
        os::FinalizeMultiWaitHolder(&curlTimerHolder);
        os::FinalizeMultiWait(&multiWait);

        curlTimer.Stop();
        if (executionResource.curlHandle)
        {
            curl_easy_cleanup(executionResource.curlHandle);
        }
    };

    managerApp.Link(&multiWait);
    managerSys.Link(&multiWait);
    int isAlive = 2;
    while (isAlive > 0)
    {
        auto *pHolder = os::WaitAny(&multiWait);

        if (pHolder == &curlTimerHolder)
        {
            curlTimer.Clear();
            curl_easy_cleanup(executionResource.curlHandle);
            executionResource.curlHandle = nullptr;
        }
        else
        {
#if defined(NN_ACCOUNT_NETWORK_ENABLE)
            if (executionResource.curlHandle == nullptr)
            {
                executionResource.curlHandle = curl_easy_init();
                NN_ABORT_UNLESS(executionResource.curlHandle != nullptr);
            }
#endif
            auto userData = os::GetMultiWaitHolderUserData(pHolder);
            if (userData == reinterpret_cast<uintptr_t>(&managerApp))
            {
                if (!managerApp.Execute(pHolder, executionResource))
                {
                    managerApp.Unlink();
                    -- isAlive;
                }
            }
            else if (userData == reinterpret_cast<uintptr_t>(&managerSys))
            {
                if (!managerSys.Execute(pHolder, executionResource))
                {
                    managerSys.Unlink();
                    -- isAlive;
                }
            }

#if defined(NN_ACCOUNT_NETWORK_ENABLE)
            curlTimer.StartOneShot(TimeSpan::FromSeconds(10));
#endif
        }
    }
    NN_SDK_LOG("[nn::account] All workers destroyed\n");
}

ServiceInitializer::ServiceInitializer(void* threadStack, size_t threadStackSize) NN_NOEXCEPT
    : m_ManagerApp(m_ServiceResource.GetExecutorForApplication())
    , m_ManagerSys(m_ServiceResource.GetExecutorForSystem())
    , m_Thread(
        [this]() { ExecutionThreadFunction(*this); },
        threadStack, threadStackSize,
        NN_SYSTEM_THREAD_PRIORITY(account, Executor_Win),
        NN_SYSTEM_THREAD_NAME(account, Executor_Win))
{
}
ServiceInitializer::~ServiceInitializer() NN_NOEXCEPT
{
    if (m_IsStarted)
    {
        NN_SDK_LOG("[nn::account] Destroying worker thread...\n");
        m_ManagerApp.RequestStop();
        m_ManagerSys.RequestStop();
        m_Thread.Stop();
    }
}

void ServiceInitializer::Start() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_IsStarted);
    m_Thread.Start();
    m_IsStarted = true;
}
DefaultServiceResource& ServiceInitializer::GetServiceResourceRef() NN_NOEXCEPT
{
    return m_ServiceResource;
}
ExecutionResource ServiceInitializer::GetExecutionResource() NN_NOEXCEPT
{
    ExecutionResource resource = {
        nullptr,
    {
        m_Buffer, sizeof(m_Buffer)
    },
        &m_Allocator,
    };
    return resource;
}
ExecutionManager<account::Executor, ExecutionResource>& ServiceInitializer::GetManagaerForApplicationsRef() NN_NOEXCEPT
{
    return m_ManagerApp;
}
ExecutionManager<account::Executor, ExecutionResource>& ServiceInitializer::GetManagaerForSystemsRef() NN_NOEXCEPT
{
    return m_ManagerSys;
}

NN_ALIGNAS(4096) char g_ThreadStack[4 * 4096];
} // ~namespace <anonymous>

// この処理は排他で呼ばれる
DefaultServiceResource& GetServiceResourceRef() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(os::SdkMutex, s_Mutex);
    std::lock_guard<decltype(s_Mutex)> lock(s_Mutex);

    NN_FUNCTION_LOCAL_STATIC(util::optional<ServiceInitializer>, s_pServiceInitializer);
    if (!s_pServiceInitializer)
    {
        // 依存モジュールの初期化
        NN_ABORT_UNLESS_RESULT_SUCCESS(time::Initialize());

        s_pServiceInitializer.emplace(g_ThreadStack, sizeof(g_ThreadStack));
        s_pServiceInitializer->Start();
    }
    return s_pServiceInitializer->GetServiceResourceRef();
}
}}} // ~namespace nn::account::detail
