﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/baas/account_BaasUserDriver.h>

#include "account_BaasAuthorizationHeader.h"
#include "account_BaasResourceResolver.h"
#include "account_BaasUserAdaptor.h"
#include "../detail/account_CacheUtil.h"
#include <nn/account/account_Config.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/baas/account_BaasTypes.h>
#include <nn/account/http/account_CurlInputStream.h>
#include <nn/account/http/account_ResultForHttp.h>
#include <nn/account/http/account_SimpleDownloader.h>
#include <nn/account/json/account_RapidJsonApi.h>
#include <nn/account/json/account_RapidJsonStream.h>
#include <nn/account/json/account_ResultForJson.h>

#include <type_traits>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/socket.h>

namespace nn { namespace account { namespace baas {
namespace
{
const size_t StringBufferSizeMin = 256u;
NN_STATIC_ASSERT(StringBufferSizeMin >= UserProfile::ImageUrlBytesForTransfer);

Result CreateImageUploadPostData(
    char* postDataBuffer, size_t postDataBufferSize,
    const NetworkServiceAccountId& ownerId, const detail::Uuid& imageCacheId,
    const detail::AbstractLocalStorage& storage,
    void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    char* inputBuffer = reinterpret_cast<char*>(workBuffer);
    size_t inSize;
    NN_RESULT_DO((detail::CacheUtil::LoadCacheFile<ProfileImageBytesMax * 2, ResultLocalDataBroken>(
        &inSize, inputBuffer, workBufferSize, imageCacheId, storage)));
    NN_ABORT_UNLESS(inSize < workBufferSize);
    inputBuffer[inSize] = '\0';

    char stackBuffer[256];
    json::FixedSizeOneTimeAllocatorForRapidJson allocator(stackBuffer, sizeof(stackBuffer));
    json::StringOutputBufferForRapidJson stringBuffer(postDataBuffer, postDataBufferSize);

    char userId[32];
    auto l = util::SNPrintf(userId, sizeof(userId), "%016llx", ownerId.id);
    NN_SDK_ASSERT(l == 16);
    NN_UNUSED(l);

    json::DefaultRapidJsonMemoryWriter writer(stringBuffer, &allocator, 8);
    NN_RESULT_THROW_UNLESS(writer.StartObject(), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.Key("rawContent"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.String(inputBuffer), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.Key("ownerId"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.String(userId), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.Key("allowTransform"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.Bool(false), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.EndObject(), json::ResultJsonConstructionFailure());

    size_t jsonLength;
    NN_RESULT_DO(stringBuffer.Finalize(&jsonLength));
    NN_SDK_ASSERT(jsonLength < postDataBufferSize);
    NN_UNUSED(jsonLength);
    NN_RESULT_SUCCESS;
}

Result SetupStreamForImageUpload (
    http::CurlInputStream& stream,
    struct curl_slist** headers,
    char* postDataBuffer, size_t postDataBufferSize,
    const UserAccessTokenCache& userAccessTokenCache,
    const NetworkServiceAccountId& userId,
    const detail::Uuid& imageCacheId,
    const detail::AbstractLocalStorage& storage,
    void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    // POST データの作成
    NN_RESULT_DO(CreateImageUploadPostData(
        postDataBuffer, postDataBufferSize, userId, imageCacheId, storage, workBuffer, workBufferSize));

    // Authorization ヘッダの準備
    char* authHeader = reinterpret_cast<char*>(workBuffer);
    NN_RESULT_DO(CreateAuthorizationHeaderWithUserAccessToken(authHeader, workBufferSize, userAccessTokenCache, userId));
    *headers = curl_slist_append(*headers, authHeader);
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());
    *headers = curl_slist_append(*headers, "Content-Type: application/json");
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());

    // リクエスト生成
    NN_RESULT_DO(stream.SetHeaders(*headers));
    NN_RESULT_DO(stream.SetHttpPost(postDataBuffer, false));

    // URL の作成
    char url[256];
    auto l = BaasResourceResolver::GetUrlForImage(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;
}

Result CreateProfileUploadPostData(
    char* postDataBuffer, size_t postDataBufferSize,
    const UserProfile& profile) NN_NOEXCEPT
{
    char stackBuffer[256];
    json::FixedSizeOneTimeAllocatorForRapidJson allocator(stackBuffer, sizeof(stackBuffer));
    json::StringOutputBufferForRapidJson stringBuffer(postDataBuffer, postDataBufferSize);
    json::DefaultRapidJsonMemoryWriter writer(stringBuffer, &allocator, 16);
    NN_RESULT_THROW_UNLESS(writer.StartArray(), json::ResultJsonConstructionFailure());

    NN_RESULT_THROW_UNLESS(writer.StartObject(), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.Key("op"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.String("replace"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.Key("path"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.String("/nickname"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.Key("value"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.String(profile.base.nickname), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.EndObject(), json::ResultJsonConstructionFailure());

    NN_RESULT_THROW_UNLESS(writer.StartObject(), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.Key("op"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.String("replace"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.Key("path"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.String("/thumbnailUrl"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.Key("value"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.String(profile.imageUrl), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.EndObject(), json::ResultJsonConstructionFailure());

    NN_RESULT_THROW_UNLESS(writer.StartObject(), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.Key("op"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.String("add"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.Key("path"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.String("/extras/self/nxAccount"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.Key("value"), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.String(profile.base.extraData), json::ResultJsonConstructionFailure());
    NN_RESULT_THROW_UNLESS(writer.EndObject(), json::ResultJsonConstructionFailure());

    NN_RESULT_THROW_UNLESS(writer.EndArray(), json::ResultJsonConstructionFailure());

    size_t jsonLength;
    NN_RESULT_DO(stringBuffer.Finalize(&jsonLength));
    NN_SDK_ASSERT(jsonLength < postDataBufferSize);
    NN_UNUSED(jsonLength);

    NN_RESULT_SUCCESS;
}
nn::Result SetupStreamForProfileUpload (
    http::CurlInputStream& stream,
    struct curl_slist** headers,
    char* postDataBuffer, size_t postDataBufferSize,
    const UserAccessTokenCache& userAccessTokenCache,
    const NetworkServiceAccountId& userId,
    const UserProfile& profile,
    void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    // POST データの作成
    NN_RESULT_DO(CreateProfileUploadPostData(postDataBuffer, postDataBufferSize, profile));

    // Authorization ヘッダの準備
    char* authHeader = reinterpret_cast<char*>(workBuffer);
    NN_RESULT_DO(CreateAuthorizationHeaderWithUserAccessToken(authHeader, workBufferSize, userAccessTokenCache, userId));
    *headers = curl_slist_append(*headers, authHeader);
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());
    *headers = curl_slist_append(*headers, "Content-Type: application/json-patch+json");
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());

    // リクエスト生成
    NN_RESULT_DO(stream.SetHeaders(*headers));
    NN_RESULT_DO(stream.SetHttpPatch(postDataBuffer, false));

    // URL の作成
    char url[256];
    auto l = BaasResourceResolver::GetUrlForUser(url, sizeof(url), userId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;
}

Result SetupStreamForProfileDownload (
    http::CurlInputStream& stream,
    struct curl_slist** headers,
    const UserAccessTokenCache& userAccessTokenCache,
    const NetworkServiceAccountId& userId,
    void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    // Authorization ヘッダの準備
    char* authHeader = reinterpret_cast<char*>(workBuffer);
    NN_RESULT_DO(CreateAuthorizationHeaderWithUserAccessToken(authHeader, workBufferSize, userAccessTokenCache, userId));
    *headers = curl_slist_append(*headers, authHeader);
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());

    // リクエスト生成
    NN_RESULT_DO(stream.SetHeaders(*headers));

    // URL の作成
    char url[256];
    auto l = BaasResourceResolver::GetUrlForUser(url, sizeof(url), userId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;
}

Result SetupStreamForLink(
    http::CurlInputStream& stream,
    struct curl_slist** headers,
    char* postDataBuffer, size_t postDataBufferSize,
    const UserAccessTokenCache& userAccessTokenCache,
    const NetworkServiceAccountId& userId,
    const detail::Uuid& naIdTokenCacheId,
    const detail::AbstractLocalStorage& storage,
    void* workBuffer, size_t workBufferSize)
{
    // Authorization ヘッダの準備
    char* authHeader = reinterpret_cast<char*>(workBuffer);
    NN_RESULT_DO(CreateAuthorizationHeaderWithUserAccessToken(authHeader, workBufferSize, userAccessTokenCache, userId));
    *headers = curl_slist_append(*headers, authHeader);
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());

    // POST データの作成
    static const char PostDataFormat[] = "idp=nintendoAccount&idToken=";
    NN_SDK_ASSERT(postDataBufferSize > sizeof(PostDataFormat));
    std::strncpy(postDataBuffer, PostDataFormat, postDataBufferSize);

    // NintendoAccount IDトークンの取得
    auto idTokenBuffer = postDataBuffer + (sizeof(PostDataFormat) - 1);
    auto idTokenBufferSize = postDataBufferSize - (sizeof(PostDataFormat) - 1);
    NN_SDK_ASSERT(idTokenBufferSize > detail::NasIdTokenSizeMax);
    size_t sizeActual;
    NN_RESULT_DO((detail::CacheUtil::LoadCacheFile<detail::NasIdTokenSizeMax, ResultNasTokenLengthUnacceptable>(
        &sizeActual, idTokenBuffer, idTokenBufferSize, naIdTokenCacheId, storage)));
    NN_SDK_ASSERT(sizeActual <= detail::NasIdTokenSizeMax);
    NN_SDK_ASSERT(idTokenBufferSize - sizeActual > 0);
    idTokenBuffer[sizeActual] = '\0';

    // リクエスト生成
    NN_RESULT_DO(stream.SetHeaders(*headers));
    NN_RESULT_DO(stream.SetHttpPost(postDataBuffer, false));

    // URL の作成
    char url[256];
    auto l = BaasResourceResolver::GetUrlForLink(url, sizeof(url), userId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;

}

Result SetupStreamForUnlink(
    http::CurlInputStream& stream,
    struct curl_slist** headers,
    const UserAccessTokenCache& userAccessTokenCache,
    const NetworkServiceAccountId& userId,
    void* workBuffer, size_t workBufferSize)
{
    // Authorization ヘッダの準備
    char* authHeader = reinterpret_cast<char*>(workBuffer);
    NN_RESULT_DO(CreateAuthorizationHeaderWithUserAccessToken(authHeader, workBufferSize, userAccessTokenCache, userId));
    *headers = curl_slist_append(*headers, authHeader);
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());

    // POST データの作成
    static const char PostData[] = "idp=nintendoAccount";

    // リクエスト生成
    NN_RESULT_DO(stream.SetHeaders(*headers));
    NN_RESULT_DO(stream.SetHttpPost(PostData, false));

    // URL の作成
    char url[256];
    auto l = BaasResourceResolver::GetUrlForUnlink(url, sizeof(url), userId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;

}
} // ~namespace nn::account::baas::<anonymous>

const size_t BaasUserDriver::RequiredBufferSize = 8 * 1024u;

BaasUserDriver::BaasUserDriver(
    const UserAccessTokenCache& userAccessTokenCache,
    const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    : m_UserAccessTokenCache(userAccessTokenCache)
    , m_Storage(storage)
{
}

Result BaasUserDriver::UploadImage(
    char* url, size_t urlSize,
    const NetworkServiceAccountId& id, const detail::Uuid& imageCacheId,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    void* largeRawBuffer, size_t largeRawBufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    struct Buffer
    {
        union
        {
            struct
            {
                char stringBuffer[StringBufferSizeMin];
                char inputBuffer[detail::IoBufferSizeMin];
            } io;
        } u;
    };
    NN_STATIC_ASSERT(sizeof(Buffer) <= RequiredBufferSize);
    NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

    struct LargeBuffer
    {
        char postDataBuffer[ProfileImageBytesMax * 2];
        char workBuffer[static_cast<size_t>((ProfileImageBytesMax * 7) / 5.0 + 0.5)];
    };
    NN_SDK_ASSERT_NOT_NULL(url);
    NN_SDK_ASSERT(imageCacheId);
    NN_SDK_ASSERT_NOT_NULL(curlHandle);
    NN_SDK_ASSERT_NOT_NULL(rawBuffer);
    NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_ASSERT(bufferSize >= sizeof(Buffer));
    NN_SDK_ASSERT_NOT_NULL(largeRawBuffer);
    NN_SDK_ASSERT(largeRawBufferSize >= sizeof(LargeBuffer));

    Buffer* buffer = reinterpret_cast<Buffer*>(rawBuffer);
    LargeBuffer* largeBuffer = reinterpret_cast<LargeBuffer*>(largeRawBuffer);

    struct curl_slist *headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(Buffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupStreamForImageUpload(
        stream, &headers, largeBuffer->postDataBuffer, sizeof(largeBuffer->postDataBuffer),
        m_UserAccessTokenCache, id, imageCacheId, m_Storage,
        largeBuffer->workBuffer, sizeof(largeBuffer->workBuffer)));

    // 通信の実行と結果の適用
    ImageUploadAdaptor adaptor("jpg", static_cast<size_t>(sizeof("jpg") - 1));
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(json::ImportJsonByRapidJson(adaptor, stream, pCancellable));
    stream.Close();
    NN_RESULT_DO(adaptor.Adapt(stream.GetHttpCode()));

    adaptor.GetUrl(url, urlSize);
    NN_RESULT_SUCCESS;
}

Result BaasUserDriver::DownloadImage(
    size_t* pOutActualSize, void* buffer, size_t bufferSize,
    const char* url, size_t urlSize,
    CURL* curlHandle,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutActualSize);
    NN_SDK_ASSERT_NOT_NULL(buffer);
    NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(buffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_ASSERT_NOT_NULL(url);
    NN_SDK_ASSERT(strlen(url) <= urlSize);
    NN_SDK_ASSERT_NOT_NULL(curlHandle);

    NN_UNUSED(urlSize);

    http::SimpleDownloader downloader(curlHandle, pCancellable);
    NN_RESULT_DO(downloader.Initialize(buffer, bufferSize));
    NN_RESULT_DO(downloader.SetUrl(url));
    NN_RESULT_DO(downloader.SetTimeoutSeconds(300)); // プロフィール画像のダウンロードのタイムアウトは長く取る

    size_t sizeActual;
    NN_RESULT_DO(downloader.Invoke(&sizeActual));
    NN_RESULT_THROW_UNLESS(sizeActual <= ProfileImageBytesMax, ResultBaasDataBroken());
    *pOutActualSize = sizeActual;
    NN_RESULT_SUCCESS;
}

Result BaasUserDriver::UploadProfile(
    const NetworkServiceAccountId& id, const UserProfile& profile,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    struct Buffer
    {
        char postDataBuffer[1024]; // 大体これくらいな雰囲気
        union
        {
            char workBuffer[RequiredBufferSizeForAuthorizationHeader];
            struct
            {
                char stringBuffer[1024]; // max(nickname, url, localData)
                char inputBuffer[detail::IoBufferSizeMin];
            } io;
            UserProfile profile;
        } u;
    };
    NN_STATIC_ASSERT(sizeof(Buffer) <= RequiredBufferSize);
    NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

    NN_SDK_ASSERT_NOT_NULL(curlHandle);
    NN_SDK_ASSERT_NOT_NULL(rawBuffer);
    NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_ASSERT(bufferSize >= sizeof(Buffer));

    Buffer* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    struct curl_slist *headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(Buffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupStreamForProfileUpload(
        stream, &headers, buffer->postDataBuffer, sizeof(buffer->postDataBuffer),
        m_UserAccessTokenCache, id, profile,
        buffer->u.workBuffer, sizeof(buffer->u.workBuffer)));

    // 通信の実行と結果の適用
    UserResourceAdaptor adaptor(m_Storage);
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(json::ImportJsonByRapidJson(adaptor, stream, pCancellable));
    stream.Close();
    NN_RESULT_DO(adaptor.Adapt(stream.GetHttpCode()));

    NN_RESULT_DO(adaptor.LoadUserProfile(&buffer->u.profile));
    NN_RESULT_THROW_UNLESS(strncmp(profile.base.nickname, buffer->u.profile.base.nickname, sizeof(profile.base.nickname)) == 0, ResultBaasDataBroken());
    NN_RESULT_THROW_UNLESS(strncmp(profile.imageUrl, buffer->u.profile.imageUrl, sizeof(profile.imageUrl)) == 0, ResultBaasDataBroken());
    NN_RESULT_THROW_UNLESS(strncmp(profile.base.extraData, buffer->u.profile.base.extraData, sizeof(profile.base.extraData)) == 0, ResultBaasDataBroken());
    NN_RESULT_SUCCESS;
}

Result BaasUserDriver::DownloadProfile(
    UserProfile* pOutProfile, const NetworkServiceAccountId& id,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    struct Buffer
    {
        union
        {
            char workBuffer[RequiredBufferSizeForAuthorizationHeader];
            struct
            {
                char stringBuffer[1024]; // max(nickname, url, localData)
                char inputBuffer[detail::IoBufferSizeMin];
            } io;
        } u;
    };
    NN_STATIC_ASSERT(sizeof(Buffer) <= RequiredBufferSize);
    NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

    NN_SDK_ASSERT_NOT_NULL(pOutProfile);
    NN_SDK_ASSERT_NOT_NULL(curlHandle);
    NN_SDK_ASSERT_NOT_NULL(rawBuffer);
    NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_ASSERT(bufferSize >= sizeof(Buffer));

    Buffer* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    struct curl_slist *headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(Buffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupStreamForProfileDownload(
        stream, &headers,
        m_UserAccessTokenCache, id,
        buffer->u.workBuffer, sizeof(buffer->u.workBuffer)));

    // 通信の実行と結果の適用
    UserResourceAdaptor adaptor(m_Storage);
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(json::ImportJsonByRapidJson(adaptor, stream, pCancellable));
    stream.Close();
    NN_RESULT_DO(adaptor.Adapt(stream.GetHttpCode()));
    NN_RESULT_THROW_UNLESS(adaptor.GetUserId() == id, ResultBaasDataBroken());
    NN_RESULT_DO(adaptor.LoadUserProfile(pOutProfile));
    NN_RESULT_SUCCESS;
}

Result BaasUserDriver::LinkWithNintendoAccount(
    const NetworkServiceAccountId& id, const NintendoAccountId& naId, const detail::Uuid& idTokenCacheId,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    struct Buffer
    {
        char postDataBuffer[detail::NasIdTokenSizeMax + 128];
        union
        {
            char workBuffer[RequiredBufferSizeForAuthorizationHeader];
            struct
            {
                char stringBuffer[128];
                char inputBuffer[detail::IoBufferSizeMin];
            } io;
        } u;
    };
    NN_STATIC_ASSERT(sizeof(Buffer) <= RequiredBufferSize);
    NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

    NN_SDK_ASSERT(idTokenCacheId);
    NN_SDK_ASSERT_NOT_NULL(curlHandle);
    NN_SDK_ASSERT_NOT_NULL(rawBuffer);
    NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_ASSERT(bufferSize >= sizeof(Buffer));

    Buffer* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    struct curl_slist *headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(Buffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupStreamForLink(
        stream, &headers,
        buffer->postDataBuffer, sizeof(buffer->postDataBuffer),
        m_UserAccessTokenCache, id, idTokenCacheId,
        m_Storage,
        buffer->u.workBuffer, sizeof(buffer->u.workBuffer)));

    // 通信の実行と結果の適用
    UserLinkAdaptor adaptor;
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(json::ImportJsonByRapidJson(adaptor, stream, pCancellable));
    stream.Close();
    NN_RESULT_DO(adaptor.Adapt(stream.GetHttpCode()));
    NN_SDK_ASSERT(adaptor.GetUserId() == id);

    // 意図しない NA が link された (通常ない)
    NintendoAccountId linkedNaId;
    NN_RESULT_THROW_UNLESS(adaptor.TryGetNintendoAccountId(&linkedNaId), ResultBaasDataBroken());
    NN_RESULT_THROW_UNLESS(naId == linkedNaId, ResultBaasDataBroken());
    NN_RESULT_SUCCESS;
}

Result BaasUserDriver::UnlinkFromNintendoAccount(
    const NetworkServiceAccountId& id,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    struct Buffer
    {
        union
        {
            char workBuffer[RequiredBufferSizeForAuthorizationHeader];
            struct
            {
                char stringBuffer[128];
                char inputBuffer[detail::IoBufferSizeMin];
            } io;
        } u;
    };
    NN_STATIC_ASSERT(sizeof(Buffer) <= RequiredBufferSize);
    NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

    NN_SDK_ASSERT_NOT_NULL(curlHandle);
    NN_SDK_ASSERT_NOT_NULL(rawBuffer);
    NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_ASSERT(bufferSize >= sizeof(Buffer));

    Buffer* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    struct curl_slist *headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(Buffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupStreamForUnlink(
        stream, &headers,
        m_UserAccessTokenCache, id,
        buffer->u.workBuffer, sizeof(buffer->u.workBuffer)));

    // 通信の実行と結果の適用
    UserLinkAdaptor adaptor;
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(json::ImportJsonByRapidJson(adaptor, stream, pCancellable));
    stream.Close();
    NN_RESULT_DO(adaptor.Adapt(stream.GetHttpCode()));
    NN_SDK_ASSERT(adaptor.GetUserId() == id);

    // unlink しても User ID は変わらないはず
    NN_SDK_ASSERT(adaptor.GetUserId() == id);

    // unlink されてない (通常ない)
    NintendoAccountId linkedNaId;
    NN_RESULT_THROW_UNLESS(!adaptor.TryGetNintendoAccountId(&linkedNaId), ResultBaasDataBroken());
    NN_RESULT_SUCCESS;
}

}}} // ~namespace nn::account::baas
