﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "account_BaasProblemDetails.h"
#include "account_BaasUserAdaptor.h"
#include <nn/account/baas/account_BaasLoginCache.h>
#include <nn/account/baas/account_BaasTypes.h>
#include <nn/account/detail/account_InternalTypes.h>
#include <nn/account/json/account_JsonAdaptor.h>
#include <nn/account/account_NintendoAccountTypes.h>
#include <nn/account/account_Result.h>

#include <cstddef>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace nn { namespace account { namespace baas {

// BaaS の public_client 権限のアクセストークンを取得するアダプタ
class ClientLoginAdaptor
    : public BaasCommonAdaptor<8, 16>
{
    NN_DISALLOW_COPY(ClientLoginAdaptor);
private:
    typedef BaasCommonAdaptor<8, 16> Base;

    ClientAccessTokenCache& m_AccessTokenCache;

    json::LookupEntry m_ExpirationEntry;
    json::LookupEntry m_TokenEntry;

    int64_t m_ExpirationAbs;
    detail::Uuid m_TokenCacheId;

public:
    typedef Base::JsonPathType JsonPathType;

protected:
    virtual Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT NN_OVERRIDE;

public:
    explicit ClientLoginAdaptor(ClientAccessTokenCache& accessTokenCache) NN_NOEXCEPT;
    ~ClientLoginAdaptor() NN_NOEXCEPT;
};

// BaaS の public_user 権限のアクセストークンと ID トークンを取得するアダプタ
class UserLoginAdaptor
    : public BaasCommonAdaptor<16, 64>
{
    NN_DISALLOW_COPY(UserLoginAdaptor);
private:
    typedef BaasCommonAdaptor<16, 64> Base;

    const detail::ApplicationInfo m_AppInfo;
    UserAccessTokenCache& m_AccessTokenCache;
    UserIdTokenCache& m_IdTokenCache;

    json::LookupEntry m_IdEntry;
    json::LookupEntry m_ExpirationEntry;
    json::LookupEntry m_AccessTokenEntry;
    json::LookupEntry m_IdTokenEntry;
    json::LookupEntry m_NaIdEntry;
    bool m_Adapted;

    NetworkServiceAccountId m_Id;
    int64_t m_ExpirationAbs;
    detail::Uuid m_AccessTokenCacheId;
    detail::Uuid m_IdTokenCacheId;
    NintendoAccountId m_NaId;

public:
    typedef Base::JsonPathType JsonPathType;

protected:
    virtual Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT NN_OVERRIDE;

public:
    UserLoginAdaptor(
        UserAccessTokenCache& accessTokenCache,
        UserIdTokenCache& idTokenCache,
        const detail::ApplicationInfo& appInfo) NN_NOEXCEPT;
    ~UserLoginAdaptor() NN_NOEXCEPT;
    NetworkServiceAccountId GetUserId() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_Adapted);
        return m_Id;
    }
    bool TryGetNintendoAccountId(NintendoAccountId* pOut) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_Adapted);
        if (!m_NaId)
        {
            return false;
        }
        *pOut = m_NaId;
        return true;
    }
};

// FederationLogin で BaaS の public_user 権限のアクセストークンと ID トークンを取得するアダプタ
class UserFederationLoginAdaptor
    : public UserResourceAdaptorWithBase<16, 64>
{
    NN_DISALLOW_COPY(UserFederationLoginAdaptor);
private:
    typedef UserResourceAdaptorWithBase<16, 64> Base;

    UserAccessTokenCache& m_AccessTokenCache;

    json::LookupEntry m_ExpirationEntry;
    json::LookupEntry m_AccessTokenEntry;
    json::LookupEntry m_IdTokenEntry;
    bool m_Adapted;

    int64_t m_ExpirationAbs;
    detail::Uuid m_AccessTokenCacheId;

public:
    typedef Base::JsonPathType JsonPathType;

protected:
    virtual Result AdaptImpl() NN_NOEXCEPT final NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT final NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT final NN_OVERRIDE;

public:
    UserFederationLoginAdaptor(
        UserAccessTokenCache& accessTokenCache,
        const detail::AbstractLocalStorage& storage) NN_NOEXCEPT;
    ~UserFederationLoginAdaptor() NN_NOEXCEPT;
};

// BaaS の public_user 権限のアクセストークンと ID トークンを取得するアダプタ (ゲストログイン用)
class UserLoginAdaptorForGuest
    : public BaasCommonAdaptor<16, 64>
{
    NN_DISALLOW_COPY(UserLoginAdaptorForGuest);
private:
    typedef BaasCommonAdaptor<16, 64> Base;

    baas::GuestUserProfile* m_pOutProfile;

    const detail::ApplicationInfo m_AppInfo;
    UserIdTokenCache& m_IdTokenCache;

    json::LookupEntry m_IdEntry;
    json::LookupEntry m_ExpirationEntry;
    json::LookupEntry m_IdTokenEntry;
    json::LookupEntry m_NaIdEntry;
    json::LookupEntry m_NicknameEntry;
    json::LookupEntry m_ImageUrlEntry;
    json::LookupEntry m_LocalDataEntry;

    int64_t m_ExpirationAbs;
    detail::Uuid m_IdTokenCacheId;

public:
    typedef Base::JsonPathType JsonPathType;

protected:
    virtual Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT NN_OVERRIDE;

public:
    UserLoginAdaptorForGuest(
        baas::GuestUserProfile* pOutProfile,
        UserIdTokenCache& idTokenCache,
        const detail::ApplicationInfo& appInfo) NN_NOEXCEPT;
    ~UserLoginAdaptorForGuest() NN_NOEXCEPT;
};

}}} // ~namespace nn::account::baas
