﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiBaasAccessToken.h>
#include <nn/account/account_ApiDebug.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForMigrationService.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiPrivate.h>
#include <nn/account/account_StateRetention.h>

#include "detail/account_ApiUtil.h"
#include "detail/account_HipcProxyClientResource.h"
#include "detail/account_ShimLibraryUtility.h"
#include <nn/account/nas/account_Interface.sfdl.h>
#include <nn/account/account_ExternalNetworkServiceAccountInfo.h>
#include <nn/account/account_IAccountService.sfdl.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/baas/account_Interface.sfdl.h>
#include <nn/account/detail/account_ISessionObject.sfdl.h>
#include <nn/account/detail/account_Log.h>

#include <nn/nn_Abort.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace account {

namespace {

detail::InitializationManager g_Init = NN_ACCOUNT_INITIALIZATION_MANAGER_INITIALIZER;
detail::ObjectHolder g_ObjectHolder;
Uid g_PreselectedUser = InvalidUid;

Uid GetUserIdImpl(const UserHandle& handle) NN_NOEXCEPT
{
    Uid uid = { { handle._data[0], handle._data[1] } };
    return uid;
}

UserHandle MakeUserHandle(const Uid& uid, const baas::IManagerForApplication* ptr) NN_NOEXCEPT
{
    UserHandle handle = {
        {uid._data[0], uid._data[1]},
        reinterpret_cast<uintptr_t>(ptr)
    };
    return handle;
};

class ClientUserHandleStore
{
private:
    static util::optional<UserHandle> s_Handles[UserCountMax];

public:
    static void Clear() NN_NOEXCEPT
    {
        for (auto& h : s_Handles)
        {
            h = util::nullopt;
        }
    }
    static void Add(const UserHandle& handle)
    {
        for (auto& h : s_Handles)
        {
            if (!h)
            {
                h.emplace(handle);
                return;
            }
        }
        NN_SDK_ASSERT(false, "Unreachable");
    }
    static void Delete(const UserHandle& handle) NN_NOEXCEPT
    {
        for (auto& h : s_Handles)
        {
            if (h && GetUserIdImpl(*h) == GetUserIdImpl(handle))
            {
                h = util::nullopt;
                return;
            }
        }
        NN_SDK_ASSERT(false, "Unreachable");
    }
    static void List(int* pOut, UserHandle handles[], int count) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(count >= UserCountMax);
        NN_UNUSED(count);

        int ct = 0;
        for (const auto& h : s_Handles)
        {
            if (h)
            {
                handles[ct++] = *h;
            }
        }
        *pOut = ct;
    }
};
util::optional<UserHandle> ClientUserHandleStore::s_Handles[] = {};
} // ~namespace nn::account::<anonymous>

/** -------------------------------------------------------------------------------------------
    共通 (プライベート)
*/
namespace detail
{
bool IsInitialized() NN_NOEXCEPT
{
    return static_cast<bool>(g_Init);
}
Result TrySelectUserWithoutInteraction(Uid* pOut, bool isNetworkServiceAccountRequired) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOut != nullptr);

    return g_ObjectHolder.GetBase().TrySelectUserWithoutInteraction(pOut, isNetworkServiceAccountRequired);
}
} // ~namespace nn::account::detail

/** -------------------------------------------------------------------------------------------
    初期化と終了
 */

void Initialize() NN_NOEXCEPT
{
    g_Init.Initialize([&]()
    {
        ClientUserHandleStore::Clear();
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.AcquireForApplicatoion());

        auto r = g_ObjectHolder.GetForApplication().InitializeApplicationInfo(0x00ull);
        if (!(r.IsSuccess() || ResultInvalidApplication::Includes(r)))
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(r);
        }
        else if (ResultInvalidApplication::Includes(r))
        {
            NN_DETAIL_ACCOUNT_WARN("[account] Program is not built as an Application, or invalid ApplicationID is specified.\n");
        }

        g_PreselectedUser = detail::TryPopPreselectedUser();
    });
}
void InitializeForSystemService() NN_NOEXCEPT
{
    g_Init.Initialize([&]()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.AcquireForSystemService());
    });
}
void InitializeForAdministrator() NN_NOEXCEPT
{
    g_Init.Initialize([&]()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.AcquireForAdministrator());
    });
}
void InitializeWith(
    sf::SharedPointer<IAccountServiceForAdministrator>&& p) NN_NOEXCEPT
{
    g_Init.Initialize([&]()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.InitializeWith(std::move(p)));
    });
}
void Finalize() NN_NOEXCEPT
{
    g_Init.Finalize([&]
    {
        g_ObjectHolder.Release();
        ClientUserHandleStore::Clear();
    });
}

/** -------------------------------------------------------------------------------------------
    共通
*/
Result GetUserCount(int* pOutCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOutCount != nullptr);

    int32_t count;
    NN_RESULT_DO(g_ObjectHolder.GetBase().GetUserCount(&count));
    *pOutCount = count;
    NN_RESULT_SUCCESS;
}
Result GetUserExistence(bool* pOuExistence, const Uid& user) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(user);
    NN_SDK_REQUIRES(pOuExistence != nullptr);

    return g_ObjectHolder.GetBase().GetUserExistence(pOuExistence, user);
}
Result ListAllUsers(int* pOutActualLength, Uid* outUsers, int arrayLength) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOutActualLength != nullptr);
    NN_SDK_REQUIRES(outUsers != nullptr);
    NN_SDK_REQUIRES(arrayLength > 0);

    Uid users[UserCountMax];
    NN_RESULT_DO(g_ObjectHolder.GetBase().ListAllUsers(sf::OutArray<Uid>(users, sizeof(users) / sizeof(users[0]))));
    *pOutActualLength = detail::CopyList(outUsers, arrayLength, users, sizeof(users) / sizeof(users[0]), InvalidUid);
    NN_RESULT_SUCCESS;
}
Result ListOpenUsers(int* pOutActualLength, Uid* outUsers, int arrayLength) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOutActualLength != nullptr);
    NN_SDK_REQUIRES(outUsers != nullptr);
    NN_SDK_REQUIRES(arrayLength > 0);

    Uid users[UserCountMax];
    NN_RESULT_DO(g_ObjectHolder.GetBase().ListOpenUsers(sf::OutArray<Uid>(users, sizeof(users) / sizeof(users[0]))));
    *pOutActualLength = detail::CopyList(outUsers, arrayLength, users, sizeof(users) / sizeof(users[0]), InvalidUid);
    NN_RESULT_SUCCESS;
}
Result GetLastOpenedUser(Uid* pOutUser) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOutUser != nullptr);

    return g_ObjectHolder.GetBase().GetLastOpenedUser(pOutUser);
}
Result GetProfileDigest(ProfileDigest* pOut, const Uid& user) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOut != nullptr);
    NN_SDK_REQUIRES(user);

    return g_ObjectHolder.GetBase().GetProfileDigest(pOut, user);
}
Result GetNickname(Nickname* pOut, const Uid& user) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOut != nullptr);
    NN_SDK_REQUIRES(user);

    nn::sf::SharedPointer<profile::IProfile> profile;
    NN_RESULT_DO(g_ObjectHolder.GetBase().GetProfile(&profile, user));

    profile::ProfileBase base;
    NN_RESULT_DO(profile->GetBase(&base));
    std::strncpy(pOut->name, base.nickname, NicknameBytesMax);
    pOut->name[NicknameBytesMax] = '\0';
    NN_RESULT_SUCCESS;
}
Result LoadProfileImage(size_t* pOutActualSize, void* outImage, size_t bufferSize, const Uid& user) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOutActualSize != nullptr);
    NN_SDK_REQUIRES(user);

    nn::sf::SharedPointer<profile::IProfile> profile;
    NN_RESULT_DO(g_ObjectHolder.GetBase().GetProfile(&profile, user));

    uint32_t rawSize;
    if (outImage == nullptr)
    {
        NN_RESULT_DO(profile->GetImageSize(&rawSize));
        *pOutActualSize = rawSize;
        NN_RESULT_SUCCESS;
    }
    else
    {
        NN_RESULT_DO(profile->LoadImage(&rawSize, sf::OutBuffer(reinterpret_cast<char*>(outImage), bufferSize)));
        *pOutActualSize = rawSize;
        NN_RESULT_SUCCESS;
    }
}
Result IsUserRegistrationRequestPermitted(bool* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOut != nullptr);
    NN_RESULT_DO(g_ObjectHolder.GetBase().IsUserRegistrationRequestPermitted(pOut, 0x00ull));
    NN_RESULT_SUCCESS;
}

/** -------------------------------------------------------------------------------------------
    一般アプリ開発者向け
     - NetworkServiceAccount 関連は account_ApiNetworkServiceAccount.cpp を見よ
 */
Result ListQualifiedUsers(int* pOutActualLength, Uid outUsers[], int arrayLength) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOutActualLength != nullptr);
    NN_SDK_REQUIRES(outUsers != nullptr);
    NN_SDK_REQUIRES(arrayLength > 0);

    Uid users[UserCountMax];
    NN_RESULT_DO(g_ObjectHolder.GetForApplication().ListQualifiedUsers(sf::OutArray<Uid>(users, sizeof(users) / sizeof(users[0]))));
    *pOutActualLength = detail::CopyList(outUsers, arrayLength, users, sizeof(users) / sizeof(users[0]), InvalidUid);
    NN_RESULT_SUCCESS;
}
Result OpenUser(UserHandle* pOutHandle, const Uid& user) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(user);

    sf::SharedPointer<baas::IManagerForApplication> pBaas;
    NN_RESULT_DO(g_ObjectHolder.GetForApplication().GetBaasAccountManagerForApplication(&pBaas, user));

    auto handle = MakeUserHandle(user, pBaas.Detach());
    ClientUserHandleStore::Add(handle);
    *pOutHandle = handle;
    NN_RESULT_SUCCESS;
}
Result OpenPreselectedUser(UserHandle* pOutHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_ABORT_UNLESS(
        g_PreselectedUser,
        "[nn::account] No user accounts are preselected for current application.\n");
    NN_RESULT_DO(OpenUser(pOutHandle, g_PreselectedUser));
    NN_RESULT_SUCCESS;
}
void CloseUser(const UserHandle& handle) NN_NOEXCEPT
{
    ClientUserHandleStore::Delete(handle);
    sf::ReleaseSharedObject(reinterpret_cast<baas::IManagerForApplication*>(handle._context));
}
Result GetUserId(Uid* pOut, const UserHandle& handle) NN_NOEXCEPT
{
    *pOut = GetUserIdImpl(handle);
    NN_RESULT_SUCCESS;
}
Result CheckNetworkServiceAvailabilityAsync(AsyncContext* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);

    sf::SharedPointer<detail::IAsyncContext> pAsync;
    NN_RESULT_DO(g_ObjectHolder.GetForApplication().CheckNetworkServiceAvailabilityAsync(&pAsync));
    *pOut = AsyncContext(pAsync.Detach());
    NN_RESULT_SUCCESS;
}
Result IntroduceExternalNetworkServiceAccount(ExternalNetworkServiceAccountInfo *pOutInfo, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(workBufferSize <= std::numeric_limits<uint32_t>::max());
    NN_RESULT_DO(detail::CheckInternetAvailability());

    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(workBuffer) % os::MemoryPageSize == 0);
    NN_SDK_REQUIRES(workBufferSize >= RequiredBufferSizeForExternalNetworkServiceAccountInfo);
    NN_SDK_REQUIRES(workBufferSize % os::MemoryPageSize == 0);
    os::TransferMemory memory(workBuffer, workBufferSize, os::MemoryPermission_None);

    sf::SharedPointer<baas::IGuestLoginRequest> p;
    NN_RESULT_DO(g_ObjectHolder.GetForApplication().CreateGuestLoginRequest(&p, sf::NativeHandle(memory.Detach(), true), static_cast<uint32_t>(workBufferSize)));

    detail::Uuid sessionId;
    NN_RESULT_DO(p->GetSessionId(&sessionId));

    // Psel を使用したログイン
    NN_DETAIL_ACCOUNT_INFO("Launching LibraryAppletPlayerSelect to introduce external network service account\n");
    UiSettings ui = {};
    ui.mode = UiMode_IntroduceExternalNsa;
    std::memcpy(&ui.introduceExternalNsa.sessionId, &sessionId, sizeof(ui.introduceExternalNsa.sessionId));
    NN_RESULT_DO(detail::StartPselApplet(ui));

    // 成功時にはすでに利用可能になっている
    *pOutInfo = ExternalNetworkServiceAccountInfo(p.Detach());
    NN_RESULT_SUCCESS;
}
Result DebugIntroduceExternalNetworkServiceAccount(uint64_t* pOutInternalInfo, size_t count, ExternalNetworkServiceAccountInfo *pOutInfo, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(count * sizeof(uint64_t) == sizeof(detail::Uuid));
    NN_UNUSED(count);
    NN_SDK_ASSERT(workBufferSize <= std::numeric_limits<uint32_t>::max());
    NN_RESULT_DO(detail::CheckInternetAvailability());

    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(workBuffer) % os::MemoryPageSize == 0);
    NN_SDK_REQUIRES(workBufferSize >= RequiredBufferSizeForExternalNetworkServiceAccountInfo);
    NN_SDK_REQUIRES(workBufferSize % os::MemoryPageSize == 0);
    os::TransferMemory memory(workBuffer, workBufferSize, os::MemoryPermission_None);

    sf::SharedPointer<baas::IGuestLoginRequest> p;
    NN_RESULT_DO(g_ObjectHolder.GetForApplication().CreateGuestLoginRequest(&p, sf::NativeHandle(memory.Detach(), true), static_cast<uint32_t>(workBufferSize)));

    detail::Uuid sessionId;
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->GetSessionId(&sessionId));
    std::memcpy(pOutInternalInfo, &sessionId, sizeof(sessionId));
    *pOutInfo = ExternalNetworkServiceAccountInfo(p.Detach());
    NN_RESULT_SUCCESS;
}

Result StoreSaveDataThumbnailImage(const Uid& uid, const void* imageBuffer, size_t imageBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);
    NN_SDK_REQUIRES_NOT_NULL(imageBuffer);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(imageBuffer) % os::MemoryPageSize == 0);
    NN_SDK_REQUIRES(imageBufferSize >= detail::SaveDataThumbnailSize);
    NN_SDK_REQUIRES(imageBufferSize % os::MemoryPageSize == 0);

    NN_RESULT_DO(g_ObjectHolder.GetForApplication().StoreSaveDataThumbnail(uid, sf::InBuffer(reinterpret_cast<const char*>(imageBuffer), detail::SaveDataThumbnailSize)));
    NN_RESULT_SUCCESS;
}

Result DeleteSaveDataThumbnailImage(const Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    NN_RESULT_DO(g_ObjectHolder.GetForApplication().ClearSaveDataThumbnail(uid));
    NN_RESULT_SUCCESS;
}

/** -------------------------------------------------------------------------------------------
    マルチプログラムアプリケーション向け
 */
void PushOpenUsers() NN_NOEXCEPT
{
    int count;
    UserHandle handles[UserCountMax];
    ClientUserHandleStore::List(&count, handles, std::extent<decltype(handles)>::value);

    for (auto i = 0; i < count; ++ i)
    {
        auto& handle = handles[i];
        NN_ABORT_UNLESS_RESULT_SUCCESS(reinterpret_cast<baas::IManagerForApplication*>(handle._context)->StoreOpenContext());
        CloseUser(handle);
    }
}
void PopOpenUsers(int* pOut, UserHandle outHandles[], int outHandleCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES_NOT_NULL(outHandles);

    Uid users[UserCountMax];
    auto count = ListPushedOpenUsers(users, std::extent<decltype(users)>::value);
    NN_ABORT_UNLESS(outHandleCount >= count);

    for (auto i = 0; i < count; ++ i)
    {
        sf::SharedPointer<baas::IManagerForApplication> ptr;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForApplication().LoadOpenContext(&ptr, users[i]));

        auto handle = MakeUserHandle(users[i], ptr.Detach());
        ClientUserHandleStore::Add(handle);
        outHandles[i] = handle;
    }
    *pOut = count;
}
int ListPushedOpenUsers(Uid* outUsers, int arrayLength) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(outUsers != nullptr);
    NN_SDK_REQUIRES(arrayLength > 0);

    Uid users[UserCountMax];
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForApplication().ListOpenContextStoredUsers(sf::OutArray<Uid>(users, sizeof(users) / sizeof(users[0]))));
    return detail::CopyList(outUsers, arrayLength, users, sizeof(users) / sizeof(users[0]), InvalidUid);
}

/** -------------------------------------------------------------------------------------------
    本体システム向け
 */
Result ListQualifiedUsers(int* pOutActualLength, Uid outUsers[], int arrayLength, const ApplicationId& applicationId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOutActualLength != nullptr);
    NN_SDK_REQUIRES(outUsers != nullptr);
    NN_SDK_REQUIRES(arrayLength > 0);

    Uid users[UserCountMax];
    NN_RESULT_DO(g_ObjectHolder.GetForSystemService().ListQualifiedUsers(sf::OutArray<Uid>(users, sizeof(users) / sizeof(users[0])), applicationId));
    *pOutActualLength = detail::CopyList(outUsers, arrayLength, users, sizeof(users) / sizeof(users[0]), InvalidUid);
    NN_RESULT_SUCCESS;
}
Result GetUserRegistrationNotifier(Notifier* pOutNotifier) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOutNotifier != nullptr);

    sf::SharedPointer<detail::INotifier> nfr;
    NN_RESULT_DO(g_ObjectHolder.GetForSystemService().GetUserRegistrationNotifier(&nfr));
    *pOutNotifier = Notifier(nfr.Detach());
    NN_RESULT_SUCCESS;
}
Result GetUserStateChangeNotifier(Notifier* pOutNotifier) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOutNotifier != nullptr);

    sf::SharedPointer<detail::INotifier> nfr;
    NN_RESULT_DO(g_ObjectHolder.GetForSystemService().GetUserStateChangeNotifier(&nfr));
    *pOutNotifier = Notifier(nfr.Detach());
    NN_RESULT_SUCCESS;
}
Result GetProfileUpdateNotifier(Notifier* pOutNotifier) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOutNotifier != nullptr);

    sf::SharedPointer<detail::INotifier> nfr;
    NN_RESULT_DO(g_ObjectHolder.GetForSystemService().GetProfileUpdateNotifier(&nfr));
    *pOutNotifier = Notifier(nfr.Detach());
    NN_RESULT_SUCCESS;
}
Result GetNetworkServiceAccountAvailabilityChangeNotifier(Notifier* pOutNotifier) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOutNotifier != nullptr);

    sf::SharedPointer<detail::INotifier> nfr;
    NN_RESULT_DO(g_ObjectHolder.GetForSystemService().GetBaasUserAvailabilityChangeNotifier(&nfr));
    *pOutNotifier = Notifier(nfr.Detach());
    NN_RESULT_SUCCESS;
}
Result GetUserLastOpenedApplication(ApplicationId* pOutApplicationId, uint32_t* pOutApplicationVersion, const Uid& user) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES_NOT_NULL(pOutApplicationId);
    NN_SDK_REQUIRES_NOT_NULL(pOutApplicationVersion);
    NN_SDK_REQUIRES(user);

    NN_RESULT_DO(g_ObjectHolder.GetForSystemService().GetUserLastOpenedApplication(pOutApplicationId, pOutApplicationVersion, user));
    NN_RESULT_SUCCESS;

}
Result GetNetworkServiceAccountManager(NetworkServiceAccountManager* pOut, const Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOut != nullptr);
    NN_SDK_REQUIRES(uid);

    sf::SharedPointer<baas::IManagerForSystemService> ptr;
    NN_RESULT_DO(g_ObjectHolder.GetForSystemService().GetBaasAccountManagerForSystemService(&ptr, uid));
    NetworkServiceAccountManager tmp(uid, ptr.Detach());
    pOut->Swap(tmp);
    NN_RESULT_SUCCESS;
}
Result CheckNetworkServiceAvailabilityAsync(AsyncContext* pOut, const SystemProgramIdentification& identification) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);

    sf::SharedPointer<detail::IAsyncContext> pAsync;
    NN_RESULT_DO(g_ObjectHolder.GetForSystemService().CheckNetworkServiceAvailabilityAsync(&pAsync, identification, 0x00));
    *pOut = AsyncContext(pAsync.Detach());
    NN_RESULT_SUCCESS;
}

Result StoreSaveDataThumbnailImage(const Uid& uid, const ApplicationId& applicationId, const void* imageBuffer, size_t imageBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);
    NN_SDK_REQUIRES(applicationId != ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(imageBuffer);
    NN_SDK_REQUIRES(imageBufferSize >= detail::SaveDataThumbnailSize);

    NN_RESULT_DO(g_ObjectHolder.GetForSystemService().StoreSaveDataThumbnail(uid, applicationId, sf::InBuffer(reinterpret_cast<const char*>(imageBuffer), detail::SaveDataThumbnailSize)));
    NN_RESULT_SUCCESS;
}
Result DeleteSaveDataThumbnailImage(const Uid& uid, const ApplicationId& applicationId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);
    NN_SDK_REQUIRES(applicationId != ApplicationId::GetInvalidId());

    NN_RESULT_DO(g_ObjectHolder.GetForSystemService().ClearSaveDataThumbnail(uid, applicationId));
    NN_RESULT_SUCCESS;
}
Result LoadSaveDataThumbnailImage(void* imageBuffer, size_t imageBufferSize, const Uid& uid, const ApplicationId& applicationId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);
    NN_SDK_REQUIRES(applicationId != ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(imageBuffer);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(imageBuffer) % os::MemoryPageSize == 0);
    NN_SDK_REQUIRES(imageBufferSize >= detail::SaveDataThumbnailSize);
    NN_SDK_REQUIRES(imageBufferSize % os::MemoryPageSize == 0);

    uint32_t sizeActual;
    NN_RESULT_DO(g_ObjectHolder.GetForSystemService().LoadSaveDataThumbnail(
        sf::Out<uint32_t>(&sizeActual), sf::OutBuffer(reinterpret_cast<char*>(imageBuffer), detail::SaveDataThumbnailSize), uid, applicationId));
    NN_SDK_ASSERT(sizeActual == static_cast<uint32_t>(detail::SaveDataThumbnailSize));
    NN_RESULT_SUCCESS;
}
Result GetSaveDataThumbnailImageExistence(bool* pOut, const Uid& uid, const ApplicationId& applicationId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);
    NN_SDK_REQUIRES(applicationId != ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    return g_ObjectHolder.GetForSystemService().GetSaveDataThumbnailExistence(sf::Out<bool>(pOut), uid, applicationId);
}
Declaration EnableInterprogramOpenUserRetention(const ApplicationId& applicationId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);

    sf::SharedPointer<detail::ISessionObject> ptr;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().ActivateOpenContextRetention(&ptr, applicationId));
    return Declaration(ptr.Detach());
}

/** -------------------------------------------------------------------------------------------
    管理者
 */
Result BeginUserRegistration(Uid* pOutUser) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOutUser != nullptr);

    return g_ObjectHolder.GetForAdministrator().BeginUserRegistration(pOutUser);
}
Result CompleteUserRegistration(const Uid& user) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(user);

    return g_ObjectHolder.GetForAdministrator().CompleteUserRegistration(user);
}
Result CompleteUserRegistrationForcibly(const Uid& user) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(user);

    return g_ObjectHolder.GetForAdministrator().CompleteUserRegistrationForcibly(user);
}
Result CancelUserRegistration(const Uid& user) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(user);

    return g_ObjectHolder.GetForAdministrator().CancelUserRegistration(user);
}
Result DeleteUser(const Uid& user) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(user);

    return g_ObjectHolder.GetForAdministrator().DeleteUser(user);
}
Result SetUserPosition(const Uid& user, int position) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(position >= 0);
    NN_SDK_REQUIRES(position < UserCountMax);

    return g_ObjectHolder.GetForAdministrator().SetUserPosition(user, position);
}
Result GetProfileEditor(ProfileEditor* pOutProfile, const Uid& user) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(user);

    sf::SharedPointer<profile::IProfileEditor> editor;
    NN_RESULT_DO(g_ObjectHolder.GetForAdministrator().GetProfileEditor(&editor, user));

    ProfileEditor p(user, editor.Detach());
    NN_RESULT_DO(p.Refresh());
    *pOutProfile = std::move(p);
    NN_RESULT_SUCCESS;
}
Result AuthenticateServiceAsync(AsyncContext* pOutContext) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOutContext != nullptr);

    sf::SharedPointer<detail::IAsyncContext> ptr;
    NN_RESULT_DO(g_ObjectHolder.GetForAdministrator().AuthenticateServiceAsync(&ptr));
    *pOutContext = AsyncContext(ptr.Detach());
    NN_RESULT_SUCCESS;
}
Result GetNetworkServiceAccountAdministrator(NetworkServiceAccountAdministrator* pOut, const Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(pOut != nullptr);
    NN_SDK_REQUIRES(uid);

    sf::SharedPointer<baas::IAdministrator> ptr;
    NN_RESULT_DO(g_ObjectHolder.GetForAdministrator().GetBaasAccountAdministrator(&ptr, uid));
    NetworkServiceAccountAdministrator tmp(uid, ptr.Detach());
    pOut->Swap(tmp);
    NN_RESULT_SUCCESS;
}

Result CreateExternalNetworkServiceAccountRegistrar(ExternalNetworkServiceAccountRegistrar *pOutInfo, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(workBufferSize <= std::numeric_limits<uint32_t>::max());

    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(workBuffer) % os::MemoryPageSize == 0);
    NN_SDK_REQUIRES(workBufferSize >= RequiredBufferSizeForExternalNetworkServiceAccountRegistrar);
    NN_SDK_REQUIRES(workBufferSize % os::MemoryPageSize == 0);
    os::TransferMemory memory(workBuffer, workBufferSize, os::MemoryPermission_None);

    sf::SharedPointer<baas::IFloatingRegistrationRequest> p;
    NN_RESULT_DO(g_ObjectHolder.GetForAdministrator().CreateFloatingRegistrationRequest(&p, sf::NativeHandle(memory.Detach(), true), static_cast<uint32_t>(workBufferSize)));

    *pOutInfo = ExternalNetworkServiceAccountRegistrar(p.Detach());
    NN_RESULT_SUCCESS;
}

Result ProxyProcedureToIntroduceExternalNetworkServiceAccountForRegistration(ExternalNetworkServiceAccountIntroducingProcedure* pOut, const SessionId& sessionId) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(SessionId) == sizeof(detail::Uuid));

    detail::Uuid innerId;
    std::memcpy(&innerId, &sessionId, sizeof(innerId));

    sf::SharedPointer<nas::IOAuthProcedureForExternalNsa> ptr;
    NN_RESULT_DO(g_ObjectHolder.GetForAdministrator().ProxyProcedureForFloatingRegistrationWithNintendoAccount(&ptr, innerId));
    NN_SDK_ASSERT(ptr);
    *pOut = ExternalNetworkServiceAccountIntroducingProcedure(ptr.Detach());
    NN_RESULT_SUCCESS;
}

Result ProxyProcedureToIntroduceExternalNetworkServiceAccount(ExternalNetworkServiceAccountIntroducingProcedure* pOut, const SessionId& sessionId) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(SessionId) == sizeof(detail::Uuid));

    detail::Uuid innerId;
    std::memcpy(&innerId, &sessionId, sizeof(innerId));

    sf::SharedPointer<nas::IOAuthProcedureForExternalNsa> ptr;
    NN_RESULT_DO(g_ObjectHolder.GetForAdministrator().ProxyProcedureForGuestLoginWithNintendoAccount(&ptr, innerId));
    NN_SDK_ASSERT(ptr);
    *pOut = ExternalNetworkServiceAccountIntroducingProcedure(ptr.Detach());
    NN_RESULT_SUCCESS;
}

void Declaration::Swap(Declaration& rhs) NN_NOEXCEPT
{
    std::swap(m_Ptr, rhs.m_Ptr);
}
Declaration::~Declaration() NN_NOEXCEPT
{
    if (m_Ptr)
    {
        sf::ReleaseSharedObject(m_Ptr);
    }
}

/** -------------------------------------------------------------------------------------------
    デバッグ用途 (権限としては、 SystemService 以上)
 */
void DebugInvalidateUserResourceCache(const Uid& user) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(user);

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystemService().DebugInvalidateUserResourceCache(user));
}
Result DebugSetUserStateOpen(const Uid& user) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(user);

    return g_ObjectHolder.GetForSystemService().DebugSetUserStateOpen(user);
}
Result DebugSetUserStateClose(const Uid& user) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);
    NN_SDK_REQUIRES(user);

    return g_ObjectHolder.GetForSystemService().DebugSetUserStateClose(user);
}
Declaration DebugEnableInterprogramOpenUserRetention() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);

    sf::SharedPointer<detail::ISessionObject> ptr;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetBase().DebugActivateOpenContextRetention(&ptr));
    return Declaration(ptr.Detach());
}

/** -------------------------------------------------------------------------------------------
    デバッグ用途 (権限としては、 Administrator 以上)
 */
Declaration SuspendBackgroundActivity() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Init);

    sf::SharedPointer<detail::ISessionObject> ptr;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForAdministrator().SuspendBackgroundDaemon(&ptr));
    return Declaration(ptr.Detach());
}

}} // ~namespace nn::account
