﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/gpio/gpio.h>
#include <nn/i2c/i2c.h>
#include <nn/i2c/i2c_BusDev.h>

#include <nnd/tmp451/tmp451.h>

#include "detail/tmp451_BusConfig.h"
#include "detail/tmp451_Debug.h"

#include "detail/tmp451_InitializeFinalize.h"
#include "detail/tmp451_InternalFunction.h"
#include "detail/tmp451_Measurement.h"
#include "detail/tmp451_SettingsInterrupt.h"
#include "detail/tmp451_SettingsMeasurment.h"
#include "detail/tmp451_SettingsRemote.h"
#include "detail/tmp451_Specification.h"


namespace{
bool                                g_IsInitialized  = false;
nnd::tmp451::detail::BusSessions    g_BusSessions;
}

namespace nnd {
namespace tmp451 {

nn::Result Initialize() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_IsInitialized, "Already initialized.");

    detail::Initialize(&g_BusSessions);

    // デバイスが見つからない場合は上位にエラーをそのまま投げる
    NN_RESULT_TRY(HasDevice(g_BusSessions))
        NN_RESULT_CATCH_ALL
        {
            detail::Finalize(&g_BusSessions);
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    // デバイスの初期化
    NN_RESULT_TRY(detail::ResetDevice(g_BusSessions))
        NN_RESULT_CATCH_ALL
        {
            detail::Finalize(&g_BusSessions);
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    // POR では計測が実行されている状態なので、一旦停止する
    detail::SetMeasurementMode(MeasurementMode::Standby, g_BusSessions);

    g_IsInitialized = true;
    NN_RESULT_SUCCESS;
}

nn::Result Finalize() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);

    // 内部ステータスの変更も兼ねて計測処理の停止
    NN_RESULT_DO(detail::SetMeasurementMode(MeasurementMode::Standby, g_BusSessions));

    // その他設定値初期化のためにリセット
    NN_RESULT_DO(detail::ResetDevice(g_BusSessions));

    detail::Finalize(&g_BusSessions);

    g_IsInitialized = false;
    NN_RESULT_SUCCESS;
}

nn::Result GetMeasurementRange(MeasurementRange* pOutRange) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);

    return detail::GetMeasurementRange(pOutRange, g_BusSessions);
}

nn::Result SetMeasurementRange(MeasurementRange range) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);
    NN_RESULT_THROW_UNLESS(GetMeasurementMode() == MeasurementMode::Standby, ResultInvalidState());

    return detail::SetMeasurementRange(range, g_BusSessions);
}

nn::Result GetMeasurementRateBy2ToThePowerN(int* pOutN) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);

    return detail::GetMeasurementRateBy2ToThePowerN(pOutN, g_BusSessions);
}

nn::Result SetMeasurementRateBy2ToThePowerN(int n) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);
    NN_RESULT_THROW_UNLESS(GetMeasurementMode() == MeasurementMode::Standby, ResultInvalidState());

    return detail::SetMeasurementRateBy2ToThePowerN(n, g_BusSessions);
}

nn::Result GetBusTimeoutFunctionEnabled(bool* pOutEnable) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);

    return detail::GetBusTimeoutFunctionEnabled(pOutEnable, g_BusSessions);
}

nn::Result SetBusTimeoutFunctionEnabled(bool enable) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);
    NN_RESULT_THROW_UNLESS(GetMeasurementMode() == MeasurementMode::Standby, ResultInvalidState());

    return detail::SetBusTimeoutFunctionEnabled(enable, g_BusSessions);
}

nn::Result GetRemoteDigitalFilterControl(DigitalFilter* pOutFilter) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);

    return detail::GetRemoteDigitalFilterControl(pOutFilter, g_BusSessions);
}

nn::Result SetRemoteDigitalFilterControl(DigitalFilter filter) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);
    NN_RESULT_THROW_UNLESS(GetMeasurementMode() == MeasurementMode::Standby, ResultInvalidState());

    return detail::SetRemoteDigitalFilterControl(filter, g_BusSessions);
}

nn::Result GetRemoteTemperatureOffset(Temperature* pOutOffset) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);

    return detail::GetRemoteTemperatureOffset(pOutOffset, g_BusSessions);
}

nn::Result SetRemoteTemperatureOffset(Temperature offset) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);
    NN_RESULT_THROW_UNLESS(GetMeasurementMode() == MeasurementMode::Standby, ResultInvalidState());

    return detail::SetRemoteTemperatureOffset(offset, g_BusSessions);
}

nn::Result GetRemoteEtaFactorCorrectionValueByN(int8_t* pOutN) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);

    return detail::GetRemoteEtaFactorCorrectionValueByN(pOutN, g_BusSessions);
}

nn::Result SetRemoteEtaFactorCorrectionValueByN(int8_t n) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);
    NN_RESULT_THROW_UNLESS(GetMeasurementMode() == MeasurementMode::Standby, ResultInvalidState());

    return detail::SetRemoteEtaFactorCorrectionValueByN(n, g_BusSessions);
}

nn::Result GetPin1DetectionSettings(Temperature* pOutRemoteLimit, Temperature* pOutLocalLimit) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);

    return detail::GetPin1DetectionSettings(pOutRemoteLimit, pOutLocalLimit, g_BusSessions);
}

nn::Result SetPin1DetectionSettings(Temperature remoteLimit, Temperature localLimit) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);
    NN_RESULT_THROW_UNLESS(GetMeasurementMode() == MeasurementMode::Standby, ResultInvalidState());

    return detail::SetPin1DetectionSettings(remoteLimit, localLimit, g_BusSessions);
}

nn::Result GetPin2DetectionSettings(Temperature* pOutRemoteLowLimit, Temperature* pOutRemoteHighLimit,
                                    Temperature* pOutLocalLowLimit, Temperature* pOutLocalHighLimit,
                                    DetectionMode* pOutDetection) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);

    return detail::GetPin2DetectionSettings(pOutRemoteLowLimit, pOutRemoteHighLimit,
                                            pOutLocalLowLimit, pOutLocalHighLimit,
                                            pOutDetection, g_BusSessions);
}

nn::Result SetPin2DetectionSettings(Temperature remoteLowLimit, Temperature remoteHighLimit,
                                    Temperature locaLowlLimit, Temperature localHighLimit,
                                    DetectionMode mode) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);
    NN_RESULT_THROW_UNLESS(GetMeasurementMode() == MeasurementMode::Standby, ResultInvalidState());

    return detail::SetPin2DetectionSettings(remoteLowLimit, remoteHighLimit,
                                            locaLowlLimit, localHighLimit,
                                            mode, g_BusSessions);
}

nn::Result BindInterrupt(nn::os::SystemEventType* pEvent, SignalPin pin) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);
    NN_RESULT_THROW_UNLESS(GetMeasurementMode() == MeasurementMode::Standby, ResultInvalidState());

    return detail::BindInterrupt(pEvent, pin, g_BusSessions);
}

nn::Result UnbindInterrupt(SignalPin pin) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);
    NN_RESULT_THROW_UNLESS(GetMeasurementMode() == MeasurementMode::Standby, ResultInvalidState());

    return detail::UnbindInterrupt(pin, g_BusSessions);
}

nn::Result SetInterruptEnable(bool enable, SignalPin pin) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);
    NN_RESULT_THROW_UNLESS(GetMeasurementMode() == MeasurementMode::Standby, ResultInvalidState());

    detail::SetInterruptEnable(enable, pin, g_BusSessions);
    NN_RESULT_SUCCESS;
}

nn::Result GetThermHysterisis(uint8_t* pOutHysterisis) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);

    return detail::GetThermHysterisis(pOutHysterisis, g_BusSessions);
}

nn::Result SetThermHysterisis(uint8_t hysterisis) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);
    NN_RESULT_THROW_UNLESS(GetMeasurementMode() == MeasurementMode::Standby, ResultInvalidState());

    return detail::SetThermHysterisis(hysterisis, g_BusSessions);
}

nn::Result GetConsecutiveAlert(AlertConsecutive* pOutConsecutive) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);

    return detail::GetConsecutiveAlert(pOutConsecutive, g_BusSessions);
}

nn::Result SetConsecutiveAlert(AlertConsecutive consecutive) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);
    NN_RESULT_THROW_UNLESS(GetMeasurementMode() == MeasurementMode::Standby, ResultInvalidState());

    return detail::SetConsecutiveAlert(consecutive, g_BusSessions);
}

MeasurementMode GetMeasurementMode() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);

    return detail::GetMeasurementMode(g_BusSessions);
}

nn::Result SetMeasurementMode(MeasurementMode mode) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);

    return detail::SetMeasurementMode(mode, g_BusSessions);
}

nn::Result ReadTemperature(Temperature* pOutTemperatute, Location location) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);

    return detail::ReadTemperature(pOutTemperatute, location, g_BusSessions);
}

nn::Result ReadStatus(Status* pOutStatus) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);

    return detail::ReadStatus(pOutStatus, g_BusSessions, true);
}

nn::Result ReadManufacturerId(uint8_t* pOutId) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsInitialized);

    return detail::ReadManufacturerId(pOutId, g_BusSessions);
}


} // tmp451
} // nnd
