﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @file
 * @brief   ドライバーの実装部
 * @details 本体 6軸センサーの種々の処理を担うクラス。
 *          一切の補正処理(キャリブレーション/ゼロ点補正等)を行わず、
 *          生値[count] の入力状態を返すまでを責務とする。
 *
 *          参考文献：
 *            - [Ref1] DocID026899 Rev 6
 *                     iNEMO inertial module: always-on 3D accelerometer and 3D gyroscope, July 2015
 *            - [Ref2] http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=83137097
 */

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/sasbus/sasbus.h>

#include "lsm6ds3_Debug.h"
#include "lsm6ds3_Driver.h"
#include "lsm6ds3_Register.h"
#include "lsm6ds3_Util.h"

namespace {

const nnd::lsm6ds3::AccelerometerFsr DefaultAccelerometerFsr = nnd::lsm6ds3::AccelerometerFsr::AccelerometerFsr_8G;
const nnd::lsm6ds3::AccelerometerOdr DefaultAccelerometerOdr = nnd::lsm6ds3::AccelerometerOdr::AccelerometerOdr_833Hz;
const nnd::lsm6ds3::GyroscopeFsr DefaultGyroscopeFsr = nnd::lsm6ds3::GyroscopeFsr::GyroscopeFsr_1000dps;
const nnd::lsm6ds3::GyroscopeOdr DefaultGyroscopeOdr = nnd::lsm6ds3::GyroscopeOdr::GyroscopeOdr_833Hz;

const uint8_t SasbusReceiveMask = 0x80; // [Ref1] 6.2.1 spi Read

}

namespace nnd {
namespace lsm6ds3 {
namespace detail {

Driver::Driver()  NN_NOEXCEPT
    : m_DeviceId(0x00)
    , m_Initialized(false)
    , m_IsFilterEnabled(false)
{
}

Driver::~Driver() NN_NOEXCEPT
{
}

nn::Result Driver::Initialize(int priority, int idealCoreNumber) NN_NOEXCEPT
{
    nn::sasbus::Initialize();
    nn::sasbus::OpenSession(&m_SasbusSession,
                            ::nn::sasbus::SasbusDevice_Lsm6ds3);

    // 実装の有無を確認
    m_DeviceId = ReadId();
    if(m_DeviceId != WhoAmIValueLsm6ds3 &&
       m_DeviceId != WhoAmIValueLsm6ds3trc)
    {
        m_DeviceId = 0x00;
        m_Initialized = false;

        ::nn::sasbus::CloseSession(&m_SasbusSession);
        ::nn::sasbus::Finalize();
        return ResultNotImplemented();
    }

    m_Initialized = true;

    // デバイスのリセット
    ResetDevice();

    // Initialize 時点でセンサーは ON
    StartAccelerometer();
    StartGyroscope();

    uint8_t ctrl3C = 0x00;
    ::nn::sasbus::Receive(&ctrl3C,
                          &m_SasbusSession,
                          1,
                          static_cast<uint8_t>(Address::Ctrl3C) | SasbusReceiveMask);
    SetBitOn8( &ctrl3C, Register::Ctrl3CIfInc::Mask);    // 多バイトアクセスを ON に
    SetBitOn8( &ctrl3C, Register::Ctrl3CBdu::Mask);      // Block Data Update を ON に
    ::nn::sasbus::Send(&m_SasbusSession,
                       &ctrl3C,
                       1,
                       static_cast<uint8_t>(Address::Ctrl3C));

    // DRDY_MASK を ON に
    uint8_t ctrl4C = 0x00;
    ::nn::sasbus::Receive(&ctrl4C,
                          &m_SasbusSession,
                          1,
                          static_cast<uint8_t>(Address::Ctrl4C) | SasbusReceiveMask);
    SetBitOn8( &ctrl4C, Register::Ctrl4CDrdyMask::Mask);
    ::nn::sasbus::Send(&m_SasbusSession,
                       &ctrl4C,
                       1,
                       static_cast<uint8_t>(Address::Ctrl4C));

    SetNoiseReductionFilter(false);

    NN_RESULT_SUCCESS;
}

void Driver::Finalize() NN_NOEXCEPT
{
    if(m_Initialized == false)
    {
        return;
    }

    // Finalize 時点でセンサーは OFF
    StopAccelerometer();
    StopGyroscope();

    m_Initialized = false;

    ::nn::sasbus::CloseSession(&m_SasbusSession);
    ::nn::sasbus::Finalize();
}

void Driver::ResetDevice() NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        return;
    }

    uint8_t  ctrl3C = 0x00;

    ::nn::sasbus::Receive(&ctrl3C, &m_SasbusSession, 1, static_cast<uint8_t>(Address::Ctrl3C) | SasbusReceiveMask);
    SetBitOn8( &ctrl3C, Register::Ctrl3CSwReset::Mask);
    ::nn::sasbus::Send(&m_SasbusSession, &ctrl3C, 1, static_cast<uint8_t>(Address::Ctrl3C));
}

void Driver::GetState(SixAxisSensorCountState* pOutState) NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        return;
    }

    const size_t  DataBytes = 12;
    uint8_t       buffer[DataBytes];
    std::memset(buffer, 0, sizeof(buffer));

    ::nn::sasbus::Receive(buffer, &m_SasbusSession, DataBytes, static_cast<uint8_t>(Address::OutXLG) | SasbusReceiveMask);

    pOutState->angularVelocity.x = (buffer[1] << 8) | buffer[0];
    pOutState->angularVelocity.y = (buffer[3] << 8) | buffer[2];
    pOutState->angularVelocity.z = (buffer[5] << 8) | buffer[4];

    pOutState->acceleration.x = (buffer[7] << 8)  | buffer[6];
    pOutState->acceleration.y = (buffer[9] << 8)  | buffer[8];
    pOutState->acceleration.z = (buffer[11] << 8) | buffer[10];

    DebugDumpSensorCountState(pOutState);

}

int Driver::GetStatesForPeriodicReceiveMode(SixAxisSensorCountState* pOutState,
                                            int count) NN_NOEXCEPT
{
    NN_UNUSED(count);

    if (m_Initialized == false)
    {
        return 0;
    }

    // TORIAEZU: サンプル数(::nn::sasbus::Lsm6ds3PeriodicReceiveModeDataCountMax より小さい値)
    const size_t                                         StateCountMax = 10;
    nn::sasbus::Lsm6ds3PeriodicReceiveModeDataType       buffer[StateCountMax];
    std::memset(buffer, 0, sizeof(buffer));

    int sampleCount =
        nn::sasbus::GetPeriodicReceiveModeData<nn::sasbus::Lsm6ds3PeriodicReceiveModeDataType, nn::sasbus::Lsm6ds3PeriodicReceiveModeDataCountMax>(
            buffer,
            &m_SasbusSession,
            StateCountMax
        );

    for (int i = 0; i < sampleCount; i++)
    {
        pOutState[i].angularVelocity.x = (buffer[i].data[1] << 8) | buffer[i].data[0];
        pOutState[i].angularVelocity.y = (buffer[i].data[3] << 8) | buffer[i].data[2];
        pOutState[i].angularVelocity.z = (buffer[i].data[5] << 8) | buffer[i].data[4];

        pOutState[i].acceleration.x = (buffer[i].data[7] << 8) |  buffer[i].data[6];
        pOutState[i].acceleration.y = (buffer[i].data[9] << 8) |  buffer[i].data[8];
        pOutState[i].acceleration.z = (buffer[i].data[11] << 8) | buffer[i].data[10];

        pOutState[i].samplingNumber = buffer[i].samplingNumber;
        pOutState[i].tick = buffer[i].tick.GetInt64Value();
    }

    return sampleCount;
}

void Driver::StartPeriodicReceiveMode(nn::TimeSpan interval, char* receiveBuffer, size_t receiveBufferLength) NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        return;
    }

    nn::sasbus::StartPeriodicReceiveMode(&m_SasbusSession, nn::sasbus::Lsm6ds3PeriodicReceiveModeDataSize, static_cast<uint8_t>(Address::OutXLG) | SasbusReceiveMask, interval, receiveBuffer, receiveBufferLength);
}

void Driver::StopPeriodicReceiveMode() NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        return;
    }

    nn::sasbus::StopPeriodicReceiveMode(&m_SasbusSession);
}

void Driver::GetNoiseReductionFilter(bool* pOutEnable) NN_NOEXCEPT
{
    *pOutEnable = m_IsFilterEnabled;
}

void Driver::SetNoiseReductionFilter(bool enable) NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        return;
    }

    // Acc のアンチエイリアシングフィルタ係数を設定
    uint8_t ctrl1Xl = 0x00;
    AccAntiAliasingFilterBandwidthRegValue filterRegValue =
        enable ? AccAntiAliasingFilterBandwidthRegValue::AccAntiAliasingFilterBandwidthRegValue_100Hz:
                 AccAntiAliasingFilterBandwidthRegValue::AccAntiAliasingFilterBandwidthRegValue_400Hz;

    ::nn::sasbus::Receive(&ctrl1Xl, &m_SasbusSession, 1, static_cast<uint8_t>(Address::Ctrl1Xl) | SasbusReceiveMask);
    Modify8(&ctrl1Xl,
            static_cast<uint8_t>(filterRegValue) << Register::Ctrl1XlBw::Pos,
            Register::Ctrl1XlBw::Mask );
    ::nn::sasbus::Send(&m_SasbusSession, &ctrl1Xl, 1, static_cast<uint8_t>(Address::Ctrl1Xl));

    // Acc の LPF 設定
    uint8_t ctrl8Xl = 0x00;
    ::nn::sasbus::Receive(&ctrl8Xl, &m_SasbusSession, 1, static_cast<uint8_t>(Address::Ctrl8Xl) | SasbusReceiveMask);
    if (m_DeviceId == WhoAmIValueLsm6ds3trc)
    {
        enable = !enable; // TR-Cでは反転
    }
    Modify8(&ctrl8Xl,
            enable << Register::Ctrl8XlHpSlopeXlEn::Pos,
            Register::Ctrl8XlHpSlopeXlEn::Mask );
    Modify8(&ctrl8Xl,
            enable << Register::Ctrl8XlLpf2XlEn::Pos,
            Register::Ctrl8XlLpf2XlEn::Mask );
    ::nn::sasbus::Send(&m_SasbusSession, &ctrl8Xl, 1, static_cast<uint8_t>(Address::Ctrl8Xl));

    if (m_DeviceId == WhoAmIValueLsm6ds3)
    {
        // フィルタ設定
        uint8_t ctrl10C = 0x00;
        ::nn::sasbus::Receive(&ctrl10C, &m_SasbusSession, 1, static_cast<uint8_t>(Address::Ctrl10C) | SasbusReceiveMask);
        Modify8(&ctrl10C,
                enable << Register::Ctrl10CFuncEn::Pos,
                Register::Ctrl10CFuncEn::Mask );
        ::nn::sasbus::Send(&m_SasbusSession, &ctrl10C, 1, static_cast<uint8_t>(Address::Ctrl10C));
    }

    m_IsFilterEnabled = enable;
}

// Acceleration
void Driver::StartAccelerometer() NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        return;
    }

    // Accelerometer を Activate する
    uint8_t ctrl9Xl = 0x00;
    ::nn::sasbus::Receive(&ctrl9Xl, &m_SasbusSession, 1, static_cast<uint8_t>(Address::Ctrl9Xl) | SasbusReceiveMask);
    SetBitOn8( &ctrl9Xl, Register::Ctrl9XlXen::Mask);
    SetBitOn8( &ctrl9Xl, Register::Ctrl9XlYen::Mask);
    SetBitOn8( &ctrl9Xl, Register::Ctrl9XlZen::Mask);
    ::nn::sasbus::Send(&m_SasbusSession, &ctrl9Xl, 1, static_cast<uint8_t>(Address::Ctrl9Xl));

    // ODR,FSR を設定する
    SetAccelerometerFsr(DefaultAccelerometerFsr);
    SetAccelerometerOdr(DefaultAccelerometerOdr);
}

void Driver::StopAccelerometer() NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        return;
    }

    // Accelerometer を Deactivate する
    uint8_t ctrl9Xl = 0x00;
    ::nn::sasbus::Receive(&ctrl9Xl, &m_SasbusSession, 1, static_cast<uint8_t>(Address::Ctrl9Xl) | SasbusReceiveMask);
    SetBitOff8( &ctrl9Xl, Register::Ctrl9XlXen::Mask);
    SetBitOff8( &ctrl9Xl, Register::Ctrl9XlYen::Mask);
    SetBitOff8( &ctrl9Xl, Register::Ctrl9XlZen::Mask);
    ::nn::sasbus::Send(&m_SasbusSession, &ctrl9Xl, 1, static_cast<uint8_t>(Address::Ctrl9Xl));

    // ODR を設定する
    SetAccelerometerOdr(AccelerometerOdr::AccelerometerOdr_0Hz);
}

void Driver::GetAccelerometerFsr(AccelerometerFsr* pOutFsr) NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        return;
    }

    uint8_t ctrl1Xl = 0x00;
    ::nn::sasbus::Receive(&ctrl1Xl, &m_SasbusSession, 1, static_cast<uint8_t>(Address::Ctrl1Xl) | SasbusReceiveMask);

    AccFsrRegValue reg = static_cast<AccFsrRegValue>((ctrl1Xl & Register::Ctrl1XlFs::Mask) >> Register::Ctrl1XlFs::Pos);
    *pOutFsr = ConvertToAccelerometerFsr(reg);
}

void Driver::SetAccelerometerFsr(AccelerometerFsr fsr) NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        return;
    }

    uint8_t ctrl1Xl = 0x00;
    ::nn::sasbus::Receive(&ctrl1Xl, &m_SasbusSession, 1, static_cast<uint8_t>(Address::Ctrl1Xl) | SasbusReceiveMask);
    Modify8( &ctrl1Xl,
             static_cast<uint8_t>(ConvertToAccFsrRegValue(fsr)) << Register::Ctrl1XlFs::Pos,
             Register::Ctrl1XlFs::Mask );
    ::nn::sasbus::Send(&m_SasbusSession, &ctrl1Xl, 1, static_cast<uint8_t>(Address::Ctrl1Xl));
}

void Driver::GetAccelerometerOdr(AccelerometerOdr* pOutOdr) NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        return;
    }

    uint8_t ctrl1Xl = 0x00;
    ::nn::sasbus::Receive(&ctrl1Xl, &m_SasbusSession, 1, static_cast<uint8_t>(Address::Ctrl1Xl) | SasbusReceiveMask);
    AccOdrRegValue reg = static_cast<AccOdrRegValue>((ctrl1Xl & Register::Ctrl1XlOdr::Mask) >> Register::Ctrl1XlOdr::Pos);
    *pOutOdr = ConvertToAccelerometerOdr(reg);
}

void Driver::SetAccelerometerOdr(AccelerometerOdr odr) NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        return;
    }

    uint8_t ctrl1Xl = 0x00;
    ::nn::sasbus::Receive(&ctrl1Xl, &m_SasbusSession, 1, static_cast<uint8_t>(Address::Ctrl1Xl) | SasbusReceiveMask);
    Modify8( &ctrl1Xl,
             static_cast<uint8_t>(ConvertToAccOdrRegValue(odr)) << Register::Ctrl1XlOdr::Pos,
             Register::Ctrl1XlOdr::Mask );
    ::nn::sasbus::Send(&m_SasbusSession, &ctrl1Xl, 1, static_cast<uint8_t>(Address::Ctrl1Xl));
}

// Gyroscope
void Driver::StartGyroscope() NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        return;
    }

    if (m_DeviceId == WhoAmIValueLsm6ds3)
    {
        // Gyroscope を Activate する
        uint8_t ctrl10C = 0x00;
        nn::sasbus::Receive(&ctrl10C, &m_SasbusSession, 1, static_cast<uint8_t>(Address::Ctrl10C) | SasbusReceiveMask);
        SetBitOn8(&ctrl10C, Register::Ctrl10CXen::Mask);
        SetBitOn8(&ctrl10C, Register::Ctrl10CYen::Mask);
        SetBitOn8(&ctrl10C, Register::Ctrl10CZen::Mask);
        ::nn::sasbus::Send(&m_SasbusSession, &ctrl10C, 1, static_cast<uint8_t>(Address::Ctrl10C));
    }

    // ODR,FSR を設定する
    SetGyroscopeOdr(DefaultGyroscopeOdr);
    SetGyroscopeFsr(DefaultGyroscopeFsr);
}

void Driver::StopGyroscope() NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        return;
    }

    if (m_DeviceId == WhoAmIValueLsm6ds3)
    {
        // Gyroscope を Deactivate する
        uint8_t ctrl10C = 0x00;
        ::nn::sasbus::Receive(&ctrl10C, &m_SasbusSession, 1, static_cast<uint8_t>(Address::Ctrl10C) | SasbusReceiveMask);
        SetBitOff8(&ctrl10C, Register::Ctrl10CXen::Mask);
        SetBitOff8(&ctrl10C, Register::Ctrl10CYen::Mask);
        SetBitOff8(&ctrl10C, Register::Ctrl10CZen::Mask);
        ::nn::sasbus::Send(&m_SasbusSession, &ctrl10C, 1, static_cast<uint8_t>(Address::Ctrl10C));
    }

    // ODR を設定する
    SetGyroscopeOdr(GyroscopeOdr::GyroscopeOdr_0Hz);
}

void Driver::GetGyroscopeFsr(GyroscopeFsr* pOutFsr) NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        return;
    }

    uint8_t ctrl2G  = 0x00;
    ::nn::sasbus::Receive(&ctrl2G, &m_SasbusSession, 1, static_cast<uint8_t>(Address::Ctrl2G) | SasbusReceiveMask);

    GyroFsrRegValue reg = static_cast<GyroFsrRegValue>((ctrl2G & Register::Ctrl2GFs::Mask) >> Register::Ctrl2GFs::Pos);
    *pOutFsr = ConvertToGyroscopeFsr(reg);
}

void Driver::SetGyroscopeFsr(GyroscopeFsr fsr) NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        return;
    }

    uint8_t ctrl2G  = 0x00;
    ::nn::sasbus::Receive(&ctrl2G, &m_SasbusSession, 1, static_cast<uint8_t>(Address::Ctrl2G) | SasbusReceiveMask);
    Modify8( &ctrl2G,
        static_cast<uint8_t>(ConvertToGyroFsrRegValue(fsr)) << Register::Ctrl2GFs::Pos,
             Register::Ctrl2GFs::Mask );
    ::nn::sasbus::Send(&m_SasbusSession, &ctrl2G, 1, static_cast<uint8_t>(Address::Ctrl2G));
}

void Driver::GetGyroscopeOdr(GyroscopeOdr* pOutOdr) NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        return;
    }

    uint8_t ctrl2G  = 0x00;
    ::nn::sasbus::Receive(&ctrl2G, &m_SasbusSession, 1, static_cast<uint8_t>(Address::Ctrl2G) | SasbusReceiveMask);
    GyroOdrRegValue reg = static_cast<GyroOdrRegValue>((ctrl2G & Register::Ctrl2GOdr::Mask) >> Register::Ctrl2GOdr::Pos);
    *pOutOdr = ConvertToGyroscopeOdr(reg);
}

void Driver::SetGyroscopeOdr(GyroscopeOdr odr) NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        return;
    }

    uint8_t ctrl2G  = 0x00;
    ::nn::sasbus::Receive(&ctrl2G, &m_SasbusSession, 1, static_cast<uint8_t>(Address::Ctrl2G) | SasbusReceiveMask);
    Modify8( &ctrl2G,
             static_cast<uint8_t>(ConvertToGyroOdrRegValue(odr)) << Register::Ctrl2GOdr::Pos,
             Register::Ctrl2GOdr::Mask );
    ::nn::sasbus::Send(&m_SasbusSession, &ctrl2G, 1, static_cast<uint8_t>(Address::Ctrl2G));
}

uint8_t Driver::ReadId() NN_NOEXCEPT
{
    // 初期化済の場合はキャッシュした値を返す。
    if(m_Initialized)
    {
        return m_DeviceId;
    }

    uint8_t  whoAmI = 0x00;
    ::nn::sasbus::Receive(&whoAmI, &m_SasbusSession, 1, static_cast<uint8_t>(Address::WhoAmI) | SasbusReceiveMask);
    NND_LSM6DS3_DETAIL_LOG("ID = 0x%x\n", whoAmI);
    return whoAmI;
}

} // detail
} // lsm6ds3
} // nnd
