﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       タッチコントローラ ftm4cd60d のデバイス固有の定義です。
 */

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/i2c/i2c.h>
#include <nnd/ftm/ftm.h>
#include "ftm_Driver-4cd60d.h"
#include "ftm_Util.h"

namespace nnd { namespace ftm { namespace detail {

namespace {

// Events ID のテーブル
struct EventIdSet
{
    EventId id;
    uint8_t val;
    uint8_t mask;   // タッチ系イベントIDのサイズ違い対応で使用
};
const EventIdSet EventIdTable[] =
{
    {EventId::NoEvents,          0x00, 0xFF},
    {EventId::TouchEnter,        0x03, 0x0F}, // 下位 4 bit のみ
    {EventId::TouchLeave,        0x04, 0x0F}, // 下位 4 bit のみ
    {EventId::TouchMotion,       0x05, 0x0F}, // 下位 4 bit のみ
    {EventId::Error,             0x0F, 0xFF},
    {EventId::ControllerReady,   0x10, 0xFF},
    {EventId::Status,            0x16, 0xFF},
    {EventId::Inform,            0x20, 0xFF},
    // {EventId::Others,},   未定義イベントには数字を割り当てない
};

// 通常時の TouchEnter イベントとの判別が困難なため別テーブルに持つ
const EventIdSet FabEventIdTable[] =
{
    {EventId::CompensationReady, 0x13, 0xFF},
};

// Orientation のテーブル
struct OrientationSet
{
    int32_t     milliDeg;
    uint8_t     val;
};
const OrientationSet OrientationTable[] =
{
    {      0, 0x00},
    {  45000, 0x01},
    {  90000, 0x02},
    { -45000, 0x03},
    {  22500, 0x04},
    {  67500, 0x05},
    { -22500, 0x06},
    { -67500, 0x07},
};

// Status のテーブル
struct StatusSet
{
    StatusType type;
    uint8_t    val;
};
const StatusSet StatusTable[] =
{
    {StatusType::AutoTuneMutual,                      0x01},
    {StatusType::AutoTuneSelf,                        0x02},
    {StatusType::FlashWriteConfig,                    0x03},
    {StatusType::FlashWriteNodeCompensationMemory,    0x04},
    {StatusType::ForceCalSelfAndMutual,               0x05},
    {StatusType::ForceCalSelfOnly,                    0x06},
    //{StatusType::Others,},   未定義イベントには数字を割り当てない
};

// ITO エラータイプのテーブル
struct ItoErrorSet
{
    ItoErrorType type;
    uint8_t      val;
};
const ItoErrorSet ItoErrorTable[] =
{
    {ItoErrorType::NoError,          0x00},
    {ItoErrorType::PanelForceOpen,   0x01},
    {ItoErrorType::PanelSenseOpen,   0x02},
    {ItoErrorType::ForceGroundShort, 0x03},
    {ItoErrorType::SenseGroundShort, 0x04},
    {ItoErrorType::ForceVcmShort,    0x05},
    {ItoErrorType::SenseVcmShort,    0x06},
    {ItoErrorType::ForceForceShort,  0x07},
    {ItoErrorType::SenseSenseShort,  0x08},
    {ItoErrorType::FpcForceOpen,     0x09},
    {ItoErrorType::FpcSenseOpen,     0x0A},
    {ItoErrorType::ItoScan,          0x0F},
    {ItoErrorType::MaxErrorReached,  0x10},
    // {ItoErrorType::Undefined,},   未定義のエラーには数字を割り当てない
};

// FW Commands
enum class FwCommand : uint8_t
{
    ReadInfoId                                             = 0x80,
    ReadId0                                                = 0x81,
    ReadId1                                                = 0x82,
    ReadId2                                                = 0x83,
    ReadStatus                                             = 0x84,
    ReadOneEvent                                           = 0x85,
    ReadAllEvent                                           = 0x86,
    ReadLatestEvent                                        = 0x87,
    SleepIn                                                = 0x90,
    ScreenSenseOff                                         = 0x92,
    ScreenSenseOn                                          = 0x93,
    LowPowerTimerCalibration                               = 0x97,
    SetFastScan                                            = 0x98,
    SetSlowScan                                            = 0x99,
    KeyOff                                                 = 0x9A,
    KeyOn                                                  = 0x9B,
    GloveOff                                               = 0x9E,
    GloveOn                                                = 0x9F,
    SystemReset                                            = 0xA0,
    ClearEventStackOrFlushBuffer                           = 0xA1,
    MutualSenseAutoTune                                    = 0xA3,
    SelfSenseAutoTune                                      = 0xA4,
    FrequencyHopMutualSense                                = 0xA5,
    FrequencyHopSelfSense                                  = 0xA6,
    ItoCheck                                               = 0xA7,
    ChargerConnectedStatusSet                              = 0xA8,
    InternalAndExternalReleaseInformation                  = 0xAA,
    SwRegisterFileWrite                                    = 0xB0,
    SwRegisterFileReadRequests                             = 0xB2,
    SwWriteConfigurationFileToFlash                        = 0xFB,
    SwWriteRegisterFileToFlashInfoBlockForNodeCompensation = 0xFC,
};

// センシングモード
enum class SensingModeServiceId : uint8_t
{
    FingerAndStylus = 0x00,
    FingerOnly      = 0x01,
    StylusOnly      = 0x02,
};

const uint8_t InterruptEnableValue = 0x40;
const uint8_t softwareControlValue = 0x08;

// HW Commands
const uint8_t HwCommandSwitchSensingMode[]  = {0xC3, 0x00};
const uint8_t HwCommandReadInformation[]    = {0xB6, 0x00, 0x04};
const uint8_t HwCommandReadLeftEventCount[] = {0xB6, 0x00, 0x23};
const uint8_t HwCommandReadInterruptState[] = {0xB6, 0x00, 0x2C};
const uint8_t HwCommandSystemReset[]        = {0xB6, 0x00, 0x28, 0x80};
const uint8_t HwCommandInterruptOn[]        = {0xB6, 0x00, 0x2C, (InterruptEnableValue | softwareControlValue)};
const uint8_t HwCommandInterruptOff[]       = {0xB6, 0x00, 0x2C, softwareControlValue};

// Hw Command （検査用）
const uint8_t HwCommandDataReadConstant                = 0xD0;
const uint8_t HwCommandOffsetForMutualRawConstant      = 0x00;
const uint8_t HwCommandOffsetForSelfTxRawConstant      = 0x1A;
const uint8_t HwCommandOffsetForSelfRxRawConstant      = 0x1C;
const uint8_t HwCommandOffsetForCompensation           = 0x50;
const uint8_t HwCommandRequestMutualCompensationData[] = {0xB8, 0x02, 0x00};
const uint8_t HwCommandRequestSelfCompensationData[]   = {0xB8, 0x20, 0x00};
const uint8_t HwCommandRequestGpioState[]              = {0xCF, 0x01};

// 取得可能な最大点数
const uint8_t MaxTouchNumber = 10;

// FIFO の情報
const uint8_t MaxEventReportCount = 64;
const size_t  EventReportByteSize = 8;

// チャンネルの情報
const uint8_t ForceChannelCount = 18;
const uint8_t SenseChannelCount = 32;

// ITO Test のレポートであることを示す値
const char ItoReportConstant = 0x05;

// 選択した Compensation データの種類
const char MsTouchConstant = 0x02;
const char SsTouchConstant = 0x20;

const uint32_t I2cRetryCountMax = 3; // リトライの最大回数
const uint32_t I2cAccessRetryIntervalMs = 5; // リトライ時のアクセス間隔
const uint32_t I2cCommandMaxDataBytes = 255; // Command List で受信可能な最大サイズ [byte]
const uint32_t CommandListMaxReceiveEvents = I2cCommandMaxDataBytes / EventReportByteSize;

// タイムアウト時間

// Ready イベントを待つタイムアウトは 20ms までみる（仕様上は 10ms 以内に完了する）
// https://redbtsred.nintendo.co.jp/redmine/issues/2854
const uint32_t ReadyTimeOutMs = 20;

// Auto Tune のタイムアウトは 2000 ms に設定する
// https://redbtsred.nintendo.co.jp/redmine/issues/3001
const uint32_t AutoTuneTimeOutMs = 2000;

// Auto Tune 結果の Flash に書き込むタイムアウトは 2000 ms に設定する
// https://redbtsred.nintendo.co.jp/redmine/issues/3124
const uint32_t SaveToFlashTimeOutMs = 2000;

// Ito Test のタイムアウトは 2000 ms に設定する
// https://redbtsred.nintendo.co.jp/redmine/issues/2990
const uint32_t ItoTestTimeOutMs = 2000;

// Tune Compensation Data 読み出しのタイムアウトは 2000 ms に設定する
// https://redbtsred.nintendo.co.jp/redmine/issues/2932
const uint32_t TuneCompensationDataTimeOutMs = 2000;

// Gpio ステータス読み出しのタイムアウトは 2000 ms に設定する
// https://redbtsred.nintendo.co.jp/redmine/issues/3123
const uint32_t GpioStateTimeOutMs = 2000;

} // namespace

Ftm4cd60dController::Ftm4cd60dController() NN_NOEXCEPT
    : IController()
    , m_Updater()
{
    m_MaxTouchNumber = static_cast<int>(MaxTouchNumber);
    m_MaxEventReportCount = static_cast<int>(MaxEventReportCount);
    m_EventReportByteSize = static_cast<size_t>(EventReportByteSize);
}

void Ftm4cd60dController::SetI2cSession(::nn::i2c::I2cSession session) NN_NOEXCEPT
{
    m_Session = session;
    m_Updater.SetI2cSession(m_Session);
}

::nn::Result Ftm4cd60dController::ResetDevice() const NN_NOEXCEPT
{
    const uint32_t RetryCountMax = 3;
    ::nn::Result result;
    for (uint32_t retry = 0; retry < RetryCountMax; retry++)
    {
        NN_RESULT_DO(this->WriteRegister(HwCommandSystemReset, sizeof(HwCommandSystemReset)));

        // Controller Ready イベントを待つ
        const ExpectedEvent event = { EventId::ControllerReady };
        result = this->WaitForCompleteEvent(event, ReadyTimeOutMs, ResultTimeoutReady());

        // BusBusy によって Controller Ready イベントを読み落とした可能性があるので
        // 再度 SystemReset コマンドの発行からやり直す
        if (::nn::i2c::ResultBusBusy::Includes(result))
        {
            continue;
        }

        break;
    }

    return result;
}

::nn::Result Ftm4cd60dController::BindInterrupt() const NN_NOEXCEPT
{
    NN_RESULT_DO(this->EnableInterrupt());
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::ActivateSensing() const NN_NOEXCEPT
{
    // ホバリング状態の誤検出を防ぐため Finger モードに入れる
    // FW Version 0x0302 以降でのみ有効
    FirmwareVersion version;
    NN_RESULT_DO(this->ReadFirmwareVersion(&version));
    if (version.program >= 0x0302)
    {
        NN_RESULT_DO(this->SwitchSensingMode(SensingMode::FingerOnly));
    }

    // センシング可能にするためには SenseOn と Flush FIFO が必要
    NN_RESULT_DO(this->ScreenSenseOn());
    NN_RESULT_DO(this->FlushBuffer());

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::DeactivateSensing() const NN_NOEXCEPT
{
    auto cmd = FwCommand::ScreenSenseOff;
    NN_RESULT_DO(this->WriteRegister(&cmd, sizeof(cmd)));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::ReadInformation(char* pOutInformation, size_t size) const NN_NOEXCEPT
{
    const size_t InformationSize = 7;
    char information[InformationSize];

    // 先頭にダミーが返るので、最低 2 バイト読まないと意味のある値は取得できない
    NN_ASSERT_MINMAX(size, static_cast<size_t>(2), InformationSize);

    NN_RESULT_DO(this->ReadRegister(
        information, InformationSize, HwCommandReadInformation, sizeof(HwCommandReadInformation)));

    ::std::memcpy(pOutInformation, information, size);

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::ReadLeftEventCount(uint32_t* pOutLeftCount) const NN_NOEXCEPT
{
    uint8_t leftEvent[2];

    NN_RESULT_DO(this->ReadRegister(
        leftEvent, sizeof(leftEvent), HwCommandReadLeftEventCount, sizeof(HwCommandReadLeftEventCount)));

    // Byte0 : Dummy Byte
    // Byte1 : Events left * 2
    uint8_t leftCount = (leftEvent[1] >> 1);
    *pOutLeftCount = (leftCount <= MaxEventReportCount) ? leftCount : MaxEventReportCount;

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::ReadEventReports(char* pOutReadData, uint32_t* pOutReadCount, bool* pOutIsOverflow, uint32_t readCount) const NN_NOEXCEPT
{
    // 一度に読み出し可能な上限数を超えていないかの確認
    NN_RESULT_THROW_UNLESS(readCount <= static_cast<uint32_t>(MaxEventReportCount), ResultInvalidArgument());

    ::nn::Result result;
    for (uint32_t retry = 0; retry < I2cRetryCountMax; retry++)
    {
        result = this->ReadEventReportsImpl(pOutReadData, readCount);

        if (::nn::i2c::ResultBusBusy::Includes(result))
        {
            ::nn::os::SleepThread(
                ::nn::TimeSpan::FromMilliSeconds(I2cAccessRetryIntervalMs));

            // FIFO に残っているイベントの数を取得
            uint32_t leftCount = 0;
            NN_RESULT_DO(this->ReadLeftEventCount(&leftCount));

            readCount = (readCount < leftCount) ? readCount : leftCount;
            continue;
        }

        break;
    }

    if (result.IsSuccess())
    {
        // ftm4cd60d ではオーバーフローを確認する手段がない
        *pOutIsOverflow = false;

        // 読み出すことのできたイベントの数を返す
        *pOutReadCount = readCount;
    }

    return result;
}

::nn::Result Ftm4cd60dController::ReadEventReports(char* pOutReadData, uint32_t* pOutReadCount, uint32_t* pOutLeftCount, bool* pOutIsOverflow, uint32_t readCount) const NN_NOEXCEPT
{
    // 一度に読み出し可能な上限数を超えていないかの確認
    NN_RESULT_THROW_UNLESS(readCount <= static_cast<uint32_t>(MaxEventReportCount), ResultInvalidArgument());

    ::nn::Result result;
    for (uint32_t retry = 0; retry < I2cRetryCountMax; retry++)
    {
        result = this->ReadEventReportsImpl(pOutReadData, readCount);

        if (::nn::i2c::ResultBusBusy::Includes(result))
        {
            ::nn::os::SleepThread(
                ::nn::TimeSpan::FromMilliSeconds(I2cAccessRetryIntervalMs));

            // FIFO に残っているイベントの数を取得
            uint32_t leftCount = 0;
            NN_RESULT_DO(this->ReadLeftEventCount(&leftCount));

            readCount = (readCount < leftCount) ? readCount : leftCount;
            continue;
        }

        break;
    }

    if (result.IsSuccess())
    {
        // FIFO に残っているイベントの数を取得
        NN_RESULT_DO(this->ReadLeftEventCount(pOutLeftCount));

        // ftm4cd60d ではオーバーフローを確認する手段がない
        *pOutIsOverflow = false;

        // 読み出すことのできたイベントの数を返す
        *pOutReadCount = readCount;
    }

    return result;
}

::nn::Result Ftm4cd60dController::ReadEventReportsImpl(char* pOutReadData, uint32_t readCount) const NN_NOEXCEPT
{
    size_t offset = 0;
    uint32_t leftCount = readCount;
    while (leftCount > 0)
    {
        uint32_t actReadCount = (leftCount < CommandListMaxReceiveEvents) ? leftCount : CommandListMaxReceiveEvents;
        size_t readByteSize = actReadCount * EventReportByteSize;
        auto cmd = static_cast<uint8_t>((actReadCount == 1) ? FwCommand::ReadOneEvent : FwCommand::ReadAllEvent);

        uint8_t commandList[::nn::i2c::CommandListLengthCountMax];
        ::nn::i2c::CommandListFormatter commandListFormatter(commandList, sizeof(commandList));
        ::std::memset(commandList, 0, sizeof(commandList));

        commandListFormatter.EnqueueSendCommand(I2cTransStart, &cmd, sizeof(cmd));
        commandListFormatter.EnqueueReceiveCommand(I2cTransStop, readByteSize);
        NN_RESULT_DO(::nn::i2c::ExecuteCommandList(pOutReadData + reinterpret_cast<uintptr_t>(offset), readByteSize, m_Session,
                                                   commandList, commandListFormatter.GetCurrentLength()));

        offset += actReadCount * EventReportByteSize; // 受信バッファに加えるオフセットをすすめる
        leftCount -= actReadCount; // 残りイベント数更新
    }

    NN_RESULT_SUCCESS;
}

void Ftm4cd60dController::ParseEventReports(EventReport* pOutEventReport, const char* pRawData, uint32_t parseCount) const NN_NOEXCEPT
{
    for (uint32_t event = 0; event < parseCount; event++)
    {
        // Event ID の変換
        uint8_t eId = pRawData[event * EventReportByteSize];
        pOutEventReport[event].eventId = this->ConvertEventValToEnum(eId);

        // Event ID 別コンテンツの変換
        switch (pOutEventReport[event].eventId)
        {
        // タッチ
        case EventId::TouchEnter:
        case EventId::TouchLeave:
        case EventId::TouchMotion:
            {
                pOutEventReport[event].content.touchReport = this->ParseTouchContent(&pRawData[event * EventReportByteSize]);
            }
            break;
        // ステータス
        case EventId::Status:
            {
                pOutEventReport[event].content.statusReport = this->ParseStatusContent(&pRawData[event * EventReportByteSize]);
            }
            break;
        // GPIO ステート (コントローラに繋がるパネルの種類判別用)
        case EventId::Inform:
            {
                pOutEventReport[event].content.gpioReport = this->ParseGpioStateContent(&pRawData[event * EventReportByteSize]);
            }
            break;
        // コンテンツを持たないその他イベント
        case EventId::Error:
        case EventId::NoEvents:
        case EventId::ControllerReady:
        case EventId::SleepOutContollerReady:
        case EventId::Others:
            {
                // EventID 以外に重要なデータを持たないので何もしない
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}

::nn::Result Ftm4cd60dController::UpdateFirmware(
    FirmwareInputFunctionPointer const pFunction, void* const pParameter, size_t fileSize) NN_NOEXCEPT
{
    // 入力用のコールバックを登録
    m_Updater.RegisterFirmwareInputCallback(pFunction, pParameter);

    // 外乱で書き込みミスする可能性を考慮し数回やりなおす
    const uint8_t RetryCountMax = 3;
    ::nn::Result result;
    for (uint8_t retry = 0; retry < RetryCountMax; retry++)
    {
        // アップデートの実行
        result = m_Updater.Proceed(fileSize);
        if (result.IsFailure())
        {
            continue;
        }

        // アップデートが正常に行われなかった場合このリセットに失敗する
        result = this->ResetDevice();
        if (result.IsSuccess())
        {
            break;
        }
    }

    // 入力用コールバックを解除
    m_Updater.UnregisterFirmwareInputCallback();

    return result;
}

::nn::Result Ftm4cd60dController::EraseFirmware() const NN_NOEXCEPT
{
    NN_RESULT_DO(m_Updater.EraseFirmware());
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::RunAutoTune() const NN_NOEXCEPT
{
    // Auto Tune 前に Reset を入れる必要がある
    // https://redbtsred.nintendo.co.jp/redmine/issues/3116
    NN_RESULT_DO(this->ResetDevice());
    NN_RESULT_DO(this->CalibrateLowPowerTimer());
    NN_RESULT_DO(this->RunMutualAutoTune());
    NN_RESULT_DO(this->RunSelfAutoTune());
    NN_RESULT_DO(this->SaveToFlash());

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::CalibrateLowPowerTimer() const NN_NOEXCEPT
{
    auto cmd = FwCommand::LowPowerTimerCalibration;
    NN_RESULT_DO(this->WriteRegister(&cmd, sizeof(cmd)));

    // Low Power Timer Calibration 完了まで 200 ms 固定で待つ必要がある
    // https://redbtsred.nintendo.co.jp/redmine/issues/3125
    ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(200));

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::RunMutualAutoTune() const NN_NOEXCEPT
{
    auto cmd = FwCommand::MutualSenseAutoTune;
    NN_RESULT_DO(this->WriteRegister(&cmd, sizeof(cmd)));

    // Status id (Auto Tune Mutual) を待つ
    const ExpectedEvent event = { EventId::Status, static_cast<char>(StatusType::AutoTuneMutual) };
    NN_RESULT_DO(this->WaitForCompleteEvent(event, AutoTuneTimeOutMs, ResultTimeoutMutualAutoTune()));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::RunSelfAutoTune() const NN_NOEXCEPT
{
    auto cmd = FwCommand::SelfSenseAutoTune;
    NN_RESULT_DO(this->WriteRegister(&cmd, sizeof(cmd)));

    // Status id (Auto Tune Self) を待つ
    const ExpectedEvent event = { EventId::Status, static_cast<char>(StatusType::AutoTuneSelf) };
    NN_RESULT_DO(this->WaitForCompleteEvent(event, AutoTuneTimeOutMs, ResultTimeoutSelfAutoTune()));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::SaveToFlash() const NN_NOEXCEPT
{
    auto cmd = FwCommand::SwWriteRegisterFileToFlashInfoBlockForNodeCompensation;
    NN_RESULT_DO(this->WriteRegister(&cmd, sizeof(cmd)));

    // Status id (Flash Write Node Compensation Memory) を待つ
    const ExpectedEvent event = { EventId::Status, static_cast<char>(StatusType::FlashWriteNodeCompensationMemory) };
    NN_RESULT_DO(this->WaitForCompleteEvent(event, SaveToFlashTimeOutMs, ResultTimeoutSaveToFlash()));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::RunItoTest(ItoEventReport* pOutItoEventReport) const NN_NOEXCEPT
{
    auto cmd = FwCommand::ItoCheck;
    NN_RESULT_DO(this->WriteRegister(&cmd, sizeof(cmd)));

    // Error type (ITO Errors) を待つ
    const ExpectedEvent event = { EventId::Error, static_cast<char>(ItoReportConstant) };
    char report[EventReportByteSize];
    NN_RESULT_DO(this->WaitForCompleteEvent(report, event, ItoTestTimeOutMs, ResultTimeoutItoTest()));
    *pOutItoEventReport = this->ParseItoErrorContent(report);

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::ReadMutualCompensationData(MutualCompensationData* pOutCompensationData) const NN_NOEXCEPT
{
    bool isInterruptEnabled;
    NN_RESULT_DO(this->ReadInterruptState(&isInterruptEnabled));

    // 割り込みが有効になっていれば無効にする
    if (isInterruptEnabled == true)
    {
        NN_RESULT_DO(this->DisableInterrupt());
    }

    // 他のイベントを発生させないためセンシングを止め FIFO をクリアする
    NN_RESULT_DO(this->DeactivateSensing());
    NN_RESULT_DO(this->FlushBuffer());

    // Compensation データの取得要求
    NN_RESULT_DO(this->RequestMutualCompensationData());
    // MS Touch を待つ
    const ExpectedEvent event = { EventId::CompensationReady, static_cast<char>(MsTouchConstant) };
    NN_RESULT_DO(this->WaitForCompleteEvent(event, TuneCompensationDataTimeOutMs, ResultTimeoutCompensation()));

    // Compensation データを読むためのアドレスのオフセット取得
    uint16_t offset;
    NN_RESULT_DO(this->ReadOffsetForInnerData(&offset, HwCommandOffsetForCompensation));

    // Compensation データの取得
    NN_RESULT_DO(this->ReadMutualCompensationDataImpl(pOutCompensationData, offset));

    // 無効にした割り込みを有効に戻す
    if (isInterruptEnabled == true)
    {
        NN_RESULT_DO(this->EnableInterrupt());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::ReadSelfCompensationData(SelfCompensationData* pOutCompensationData) const NN_NOEXCEPT
{
    bool isInterruptEnabled;
    NN_RESULT_DO(this->ReadInterruptState(&isInterruptEnabled));

    // 割り込みが有効になっていれば無効にする
    if (isInterruptEnabled == true)
    {
        NN_RESULT_DO(this->DisableInterrupt());
    }

    // 他のイベントを発生させないためセンシングを止め FIFO をクリアする
    NN_RESULT_DO(this->DeactivateSensing());
    NN_RESULT_DO(this->FlushBuffer());

    // Compensation データの取得要求
    NN_RESULT_DO(this->RequestSelfCompensationData());
    // SS Touch を待つ
    const ExpectedEvent event = { EventId::CompensationReady, static_cast<char>(SsTouchConstant) };
    NN_RESULT_DO(this->WaitForCompleteEvent(event, TuneCompensationDataTimeOutMs, ResultTimeoutCompensation()));

    // Compensation データを読むためのアドレスのオフセット取得
    uint16_t offset;
    NN_RESULT_DO(this->ReadOffsetForInnerData(&offset, HwCommandOffsetForCompensation));

    // Compensation データの取得
    NN_RESULT_DO(this->ReadSelfCompensationDataImpl(pOutCompensationData, offset));

    // 無効にした割り込みを有効に戻す
    if (isInterruptEnabled == true)
    {
        NN_RESULT_DO(this->EnableInterrupt());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::ReadMutualRawData(MutualRawData* pOutRawData) const NN_NOEXCEPT
{
    bool isInterruptEnabled;
    NN_RESULT_DO(this->ReadInterruptState(&isInterruptEnabled));

    // 割り込みが有効になっていれば無効にする
    if (isInterruptEnabled == true)
    {
        NN_RESULT_DO(this->DisableInterrupt());
    }

    // 他のイベントを発生させないためセンシングを止め FIFO をクリアする
    NN_RESULT_DO(this->DeactivateSensing());
    NN_RESULT_DO(this->FlushBuffer());

    // Raw データを読むためのアドレスのオフセット取得
    uint16_t offset;
    NN_RESULT_DO(this->ReadOffsetForInnerData(&offset, HwCommandOffsetForMutualRawConstant));

    // Raw データの取得
    NN_RESULT_DO(this->ReadMutualRawDataImpl(pOutRawData, offset));

    // チャンネルノード数の設定（本処理中にデバイスから取得する方法がないので固定値を返す）
    pOutRawData->forceChannelCount = ForceChannelCount;
    pOutRawData->senseChannelCount = SenseChannelCount;

    // 無効にした割り込みを有効に戻す
    if (isInterruptEnabled == true)
    {
        NN_RESULT_DO(this->EnableInterrupt());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::ReadSelfRawData(SelfRawData* pOutRawData) const NN_NOEXCEPT
{
    bool isInterruptEnabled;
    NN_RESULT_DO(this->ReadInterruptState(&isInterruptEnabled));

    // 割り込みが有効になっていれば無効にする
    if (isInterruptEnabled == true)
    {
        NN_RESULT_DO(this->DisableInterrupt());
    }

    // 他のイベントを発生させないためセンシングを止め FIFO をクリアする
    NN_RESULT_DO(this->DeactivateSensing());
    NN_RESULT_DO(this->FlushBuffer());

    // Tx Raw データを読むためのアドレスのオフセット取得
    uint16_t offset;
    NN_RESULT_DO(this->ReadOffsetForInnerData(&offset, HwCommandOffsetForSelfTxRawConstant));

    // Tx Raw データの取得
    NN_RESULT_DO(this->ReadSelfTxRawDataImpl(pOutRawData, offset));

    // Rx Raw データを読むためのアドレスのオフセット取得
    NN_RESULT_DO(this->ReadOffsetForInnerData(&offset, HwCommandOffsetForSelfRxRawConstant));

    // Rx Raw データの取得
    NN_RESULT_DO(this->ReadSelfRxRawDataImpl(pOutRawData, offset));

    // チャンネルノード数の設定（本処理中にデバイスから取得する方法がないので固定値を返す）
    pOutRawData->forceChannelCount = ForceChannelCount;
    pOutRawData->senseChannelCount = SenseChannelCount;

    // 無効にした割り込みを有効に戻す
    if (isInterruptEnabled == true)
    {
        NN_RESULT_DO(this->EnableInterrupt());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::RequestGpioState(GpioInformEventReport* pOutGpioInformEventReport) const NN_NOEXCEPT
{
    const uint32_t RetryCountMax = 3;
    ::nn::Result result;
    for (uint32_t retry = 0; retry < RetryCountMax; retry++)
    {
        NN_RESULT_DO(this->WriteRegister(HwCommandRequestGpioState, sizeof(HwCommandRequestGpioState)));

        // Inform イベントを待つ
        char report[EventReportByteSize];
        const ExpectedEvent event = { EventId::Inform, static_cast<char>(HwCommandRequestGpioState[1]) };
        result = this->WaitForCompleteEvent(report, event, GpioStateTimeOutMs, ResultTimeoutGpio());

        // BusBusy によって Inform イベントを読み落とした可能性があるので
        // 再度 RequestGpioState コマンドの発行からやり直す
        if (::nn::i2c::ResultBusBusy::Includes(result))
        {
            continue;
        }

        if (result.IsSuccess())
        {
            *pOutGpioInformEventReport = this->ParseGpioStateContent(report);
        }

        break;
    }

    return result;
}

::nn::Result Ftm4cd60dController::ScreenSenseOn() const NN_NOEXCEPT
{
    auto cmd = FwCommand::ScreenSenseOn;
    NN_RESULT_DO(this->WriteRegister(&cmd, sizeof(cmd)));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::FlushBuffer() const NN_NOEXCEPT
{
    auto cmd = FwCommand::ClearEventStackOrFlushBuffer;
    NN_RESULT_DO(this->WriteRegister(&cmd, sizeof(cmd)));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::ReadInterruptState(bool* pOutIsInterruptEnabled) const NN_NOEXCEPT
{
    char interruptStateData[0x2];

    NN_RESULT_DO(
        this->ReadRegister(interruptStateData, sizeof(interruptStateData), HwCommandReadInterruptState, sizeof(HwCommandReadInterruptState)));

    *pOutIsInterruptEnabled = ((interruptStateData[1] & InterruptEnableValue) != 0) ? true : false;

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::EnableInterrupt() const NN_NOEXCEPT
{
    NN_RESULT_DO(this->WriteRegister(HwCommandInterruptOn, sizeof(HwCommandInterruptOn)));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::DisableInterrupt() const NN_NOEXCEPT
{
    NN_RESULT_DO(this->WriteRegister(HwCommandInterruptOff, sizeof(HwCommandInterruptOff)));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::SwitchSensingMode(SensingMode mode) const NN_NOEXCEPT
{
    uint8_t serviceId;
    switch (mode)
    {
    case SensingMode::FingerAndStylus:
        {
            serviceId = static_cast<uint8_t>(SensingModeServiceId::FingerAndStylus);
        }
        break;
    case SensingMode::FingerOnly:
        {
            serviceId = static_cast<uint8_t>(SensingModeServiceId::FingerOnly);
        }
        break;
    case SensingMode::StylusOnly:
        {
            serviceId = static_cast<uint8_t>(SensingModeServiceId::StylusOnly);
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    uint8_t cmd[] =
    {
        HwCommandSwitchSensingMode[0],
        serviceId
    };
    NN_RESULT_DO(this->WriteRegister(cmd, sizeof(cmd)));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::RequestMutualCompensationData() const NN_NOEXCEPT
{
    NN_RESULT_DO(this->WriteRegister(HwCommandRequestMutualCompensationData, sizeof(HwCommandRequestMutualCompensationData)));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::RequestSelfCompensationData() const NN_NOEXCEPT
{
    NN_RESULT_DO(this->WriteRegister(HwCommandRequestSelfCompensationData, sizeof(HwCommandRequestSelfCompensationData)));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::WaitForCompleteEvent(const ExpectedEvent& expectedEvent, uint32_t timeoutMilliSeconds, ::nn::Result result) const NN_NOEXCEPT
{
    char report[EventReportByteSize];
    NN_RESULT_DO(this->WaitForCompleteEvent(report, expectedEvent, timeoutMilliSeconds, result));
    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::WaitForCompleteEvent(char* pOutReport, const ExpectedEvent& expectedEvent, uint32_t timeoutMilliSeconds, ::nn::Result result) const NN_NOEXCEPT
{
    bool     isTimeout = false;
    bool     isOverflow;
    uint32_t readCount;
    uint32_t leftCount;
    char     data[EventReportByteSize];

    // タイムアウトするまでイベントを一つずつ確認する
    ::nn::os::Tick tick = ::nn::os::GetSystemTick();
    while (isTimeout == false)
    {
        ::nn::TimeSpan timeSpan = ::nn::os::ConvertToTimeSpan(::nn::os::GetSystemTick() - tick);
        isTimeout = (timeSpan.GetMilliSeconds() > timeoutMilliSeconds) ? true : false;

        NN_RESULT_DO(this->ReadEventReports(data, &readCount, &leftCount, &isOverflow, 1));
        if (IsExpectedEvent(data, expectedEvent) == true)
        {
            ::std::memcpy(pOutReport, data, EventReportByteSize);
            NN_RESULT_SUCCESS;
        }
    }

    // タイムアウト時点での FIFO 内の残りイベントを確認する
    const uint8_t LeftEventsInFifo = leftCount;
    for (uint8_t event = 0; event < LeftEventsInFifo; event++)
    {
        NN_RESULT_DO(this->ReadEventReports(data, &readCount, &leftCount, &isOverflow, 1));
        if (IsExpectedEvent(data, expectedEvent) == true)
        {
            ::std::memcpy(pOutReport, data, EventReportByteSize);
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(result);
}

::nn::Result Ftm4cd60dController::ReadOffsetForInnerData(uint16_t* pOutOffset, uint8_t offset) const NN_NOEXCEPT
{
    char offsetData[0x4];
    uint8_t cmd[] = {HwCommandDataReadConstant, 0x00, offset};
    NN_RESULT_DO(
        this->ReadRegister(offsetData, sizeof(offsetData), cmd, sizeof(cmd)));

    *pOutOffset = ((offsetData[2] << 8) | offsetData[1]);

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::ReadMutualCompensationDataImpl(MutualCompensationData* pOutCompensationData, uint16_t offset) const NN_NOEXCEPT
{
    // Force Channel ノード数取得、Sense Channel ノード数取得、Mutual Cx1 データの取得
    {
        char buffer[0x10];
        uint16_t address = offset;
        uint8_t cmd[] =
        {
            HwCommandDataReadConstant,
            static_cast<uint8_t>((address & 0xFF00) >> 8),
            static_cast<uint8_t>(address & 0x00FF)
        };

        NN_RESULT_DO(
            this->ReadRegister(buffer, sizeof(buffer), cmd, sizeof(cmd)));

        pOutCompensationData->forceChannelCount = buffer[5];  // Force Channel ノード数
        pOutCompensationData->senseChannelCount = buffer[6];  // Sense Channel ノード数
        pOutCompensationData->cx1               = buffer[10]; // Cx1 データ
    }

    // Mutual Cx2 データの取得
    for (uint8_t force = 0; force < pOutCompensationData->forceChannelCount; force++)
    {
        char buffer[SenseChannelCount + 1]; // 先頭に 1 Byte 余計にダミーデータが返ってくる
        uint16_t address = 0x10 + offset + (force * pOutCompensationData->senseChannelCount);
        uint8_t cmd[] =
        {
            HwCommandDataReadConstant,
            static_cast<uint8_t>((address & 0xFF00) >> 8),
            static_cast<uint8_t>(address & 0x00FF)
        };

        NN_RESULT_DO(
            this->ReadRegister(buffer, sizeof(buffer), cmd, sizeof(cmd)));

        // 先頭 1Byte はダミーが返る
        ::std::memcpy(pOutCompensationData->cx2[force], (buffer + 1), pOutCompensationData->senseChannelCount);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::ReadSelfCompensationDataImpl(SelfCompensationData* pOutCompensationData, uint16_t offset) const NN_NOEXCEPT
{
    // Force Channel ノード数取得、Sense Channel ノード数取得
    // Self Force Ix1 データ、Self Sense Ix1 データ、Self Force Cx1 データ、Self Sense Cx1 データの取得
    {
        char buffer[0x10];
        uint16_t address = offset;
        uint8_t cmd[] =
        {
            HwCommandDataReadConstant,
            static_cast<uint8_t>((address & 0xFF00) >> 8),
            static_cast<uint8_t>(address & 0x00FF)
        };

        NN_RESULT_DO(
            this->ReadRegister(buffer, sizeof(buffer), cmd, sizeof(cmd)));

        pOutCompensationData->forceChannelCount = buffer[5];  // Force Channel ノード数
        pOutCompensationData->senseChannelCount = buffer[6];  // Sense Channel ノード数
        pOutCompensationData->forceIx1          = buffer[10]; // Force Ix1 データ
        pOutCompensationData->senseIx1          = buffer[11]; // Sense Ix1 データ
        pOutCompensationData->forceCx1          = buffer[12]; // Force Cx1 データ
        pOutCompensationData->senseCx1          = buffer[13]; // Sense Cx1 データ
    }

    // Self Ix2 データと Self Cx2 データの取得
    {
        char buffer[2 * (ForceChannelCount + SenseChannelCount) + 1];
        uint16_t address = 0x10 + offset;
        uint8_t cmd[] =
        {
            HwCommandDataReadConstant,
            static_cast<uint8_t>((address & 0xFF00) >> 8),
            static_cast<uint8_t>(address & 0x00FF)
        };

        NN_RESULT_DO(
            this->ReadRegister(buffer, sizeof(buffer), cmd, sizeof(cmd)));

        char* pBuffer = buffer + 1; // 先頭 1 Byte はダミーが返る
        ::std::memcpy(pOutCompensationData->ix2, pBuffer,
                      (pOutCompensationData->forceChannelCount + pOutCompensationData->senseChannelCount));

        pBuffer += (pOutCompensationData->forceChannelCount + pOutCompensationData->senseChannelCount);
        ::std::memcpy(pOutCompensationData->cx2, pBuffer,
                      (pOutCompensationData->forceChannelCount + pOutCompensationData->senseChannelCount));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::ReadMutualRawDataImpl(MutualRawData* pOutRawData, uint16_t offset) const NN_NOEXCEPT
{
    for (uint8_t force = 0; force < ForceChannelCount; force++)
    {
        uint16_t address = offset + 2 * SenseChannelCount + (2 * force * SenseChannelCount);
        uint8_t cmd[] =
        {
            HwCommandDataReadConstant,
            static_cast<uint8_t>((address & 0xFF00) >> 8),
            static_cast<uint8_t>(address & 0x00FF)
        };

        char buffer[2 * SenseChannelCount + 1];
        NN_RESULT_DO(
            this->ReadRegister(buffer, sizeof(buffer), cmd, sizeof(cmd)));

        for (uint8_t sense = 0; sense < SenseChannelCount; sense++)
        {
            // 先頭 1Byte はダミーが返る
            pOutRawData->data[force][sense] = (buffer[2 * sense + 2] << 8) | buffer[2 * sense + 1];
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::ReadSelfTxRawDataImpl(SelfRawData* pOutRawData, uint16_t offset) const NN_NOEXCEPT
{
    uint16_t address = offset;
    uint8_t cmd[] =
    {
        HwCommandDataReadConstant,
        static_cast<uint8_t>((address & 0xFF00) >> 8),
        static_cast<uint8_t>(address & 0x00FF)
    };

    char buffer[2 * ForceChannelCount + 1]; // 先頭に 1 Byte 余計にダミーデータが返ってくる
    NN_RESULT_DO(
        this->ReadRegister(buffer, sizeof(buffer), cmd, sizeof(cmd)));

    for (uint8_t count = 0; count < ForceChannelCount; count++)
    {
        pOutRawData->forceData[count] = (buffer[2 * count + 2] << 8) | buffer[2 * count + 1];
    }

    NN_RESULT_SUCCESS;
}

::nn::Result Ftm4cd60dController::ReadSelfRxRawDataImpl(SelfRawData* pOutRawData, uint16_t offset) const NN_NOEXCEPT
{
    uint16_t address = offset;
    uint8_t cmd[] =
    {
        HwCommandDataReadConstant,
        static_cast<uint8_t>((address & 0xFF00) >> 8),
        static_cast<uint8_t>(address & 0x00FF)
    };

    char buffer[2 * SenseChannelCount + 1]; // 先頭に 1 Byte 余計にダミーデータが返ってくる
    NN_RESULT_DO(
        this->ReadRegister(buffer, sizeof(buffer), cmd, sizeof(cmd)));

    for (uint8_t count = 0; count < SenseChannelCount; count++)
    {
        pOutRawData->senseData[count] = (buffer[2 * count + 2] << 8) | buffer[2 * count + 1];
    }

    NN_RESULT_SUCCESS;
}

TouchEventReport Ftm4cd60dController::ParseTouchContent(const char* pRawData) const NN_NOEXCEPT
{
    TouchEventReport touchContent;

    touchContent.touchId = (pRawData[0] >> 4) & 0x0F;
    touchContent.x = (pRawData[1] << 4) | ((pRawData[3] & 0xF0) >> 4);
    touchContent.y = (pRawData[2] << 4) | (pRawData[3] & 0x0F);

    uint16_t z_coor = pRawData[4] | (pRawData[5] << 8);

    // Finger モードでスタイラスと判定されると正しい楕円軸が取得できない件のワークアラウンド
    // https://redbtsred.nintendo.co.jp/redmine/issues/4709
    uint8_t temporaryMinorAxis = pRawData[6] & 0x3F;
    uint8_t minorAxis = (temporaryMinorAxis == 1 || temporaryMinorAxis == 63) ? 64 : temporaryMinorAxis;

    touchContent.major = (64 * z_coor) / (64 + minorAxis);
    touchContent.minor = (minorAxis * z_coor) / (64 + minorAxis);

    uint8_t ori = ((pRawData[6] & 0xC0) >> 6) | ((pRawData[7] & 0x80) >> 5);
    touchContent.milliDegree = this->ConvertOrientationValToEnum(ori);

    return touchContent;
}

StatusType Ftm4cd60dController::ParseStatusContent(const char* pRawData) const NN_NOEXCEPT
{
    return this->ConvertStatusValToEnum(pRawData[1]);
}

GpioInformEventReport Ftm4cd60dController::ParseGpioStateContent(const char* pRawData) const NN_NOEXCEPT
{
    GpioInformEventReport gpioContent;
    gpioContent.gpio0 = pRawData[2];
    gpioContent.gpio1 = pRawData[3];
    gpioContent.gpio2 = pRawData[4];

    return gpioContent;
}

ItoEventReport Ftm4cd60dController::ParseItoErrorContent(const char* pRawData) const NN_NOEXCEPT
{
    ItoEventReport itoEventRepert;

    itoEventRepert.number = static_cast<int>(pRawData[3]);
    itoEventRepert.type = this->ConvertItoErrorValToEnum(pRawData[2]);

    return itoEventRepert;
}

EventId Ftm4cd60dController::ConvertEventValToEnum(uint8_t idVal) const NN_NOEXCEPT
{
    for (const auto& event : EventIdTable)
    {
        if (event.val == (idVal & event.mask))
        {
            return event.id;
        }
    }
    return EventId::Others;
}

EventId Ftm4cd60dController::ConvertFabEventValToEnum(uint8_t idVal) const NN_NOEXCEPT
{
    for (const auto& event : FabEventIdTable)
    {
        if (event.val == (idVal & event.mask))
        {
            return event.id;
        }
    }
    return EventId::Others;
}

int32_t Ftm4cd60dController::ConvertOrientationValToEnum(uint8_t orientationVal) const NN_NOEXCEPT
{
    for (const auto& orientation : OrientationTable)
    {
        if (orientation.val == orientationVal)
        {
            return orientation.milliDeg;
        }
    }
    return 0; // 未定義の値の場合、0 を返す
}

StatusType Ftm4cd60dController::ConvertStatusValToEnum(uint8_t statusVal) const NN_NOEXCEPT
{
    for (const auto& status : StatusTable)
    {
        if (status.val == statusVal)
        {
            return status.type;
        }
    }
    return StatusType::Others;
}

ItoErrorType Ftm4cd60dController::ConvertItoErrorValToEnum(uint8_t itoErrorVal) const NN_NOEXCEPT
{
    for (const auto& itoError : ItoErrorTable)
    {
        if (itoError.val == itoErrorVal)
        {
            return itoError.type;
        }
    }
    return ItoErrorType::Undefined;
}

bool Ftm4cd60dController::IsExpectedEvent(const char* pReturnedEvent, const ExpectedEvent& expectedEvent) const NN_NOEXCEPT
{
    // 期待する Event ID であるかの確認
    // CompensationReady は TouchEnter との区別が困難なので個別対応
    EventId id = (expectedEvent.id == EventId::CompensationReady) ?
        this->ConvertFabEventValToEnum(pReturnedEvent[0]) : this->ConvertEventValToEnum(pReturnedEvent[0]);
    if (id != expectedEvent.id)
    {
        return false;
    }

    // Event ID に応じてそれ以降のデータの比較方法を決定
    switch (id)
    {
    case EventId::ControllerReady:
        {
            return true;
        }
        break;
    case EventId::Status:
        {
            StatusType type = this->ConvertStatusValToEnum(pReturnedEvent[1]);
            return type == static_cast<StatusType>(expectedEvent.value) ? true : false;
        }
        break;
    case EventId::Error:
    case EventId::Inform:
    case EventId::CompensationReady:
        {
            return pReturnedEvent[1] == expectedEvent.value ? true : false;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

::nn::Result Ftm4cd60dController::WriteRegister(const void* pInData, size_t dataBytes) const NN_NOEXCEPT
{
    ::nn::Result result;
    for (uint32_t retry = 0; retry < I2cRetryCountMax; retry++)
    {
        result = ::nn::i2c::Send(m_Session, pInData, dataBytes, I2cTransStartStop);

        if (::nn::i2c::ResultBusBusy::Includes(result))
        {
            ::nn::os::SleepThread(
                ::nn::TimeSpan::FromMilliSeconds(I2cAccessRetryIntervalMs));
            continue;
        }

        break;
    }

    return result;
}

::nn::Result Ftm4cd60dController::ReadRegister(
    void* pBuffer, size_t receiveDataBytes, const nn::Bit8* pInData, size_t sendDataBytes) const NN_NOEXCEPT
{
    ::nn::Result result;
    for (uint32_t retry = 0; retry < I2cRetryCountMax; retry++)
    {
        uint8_t commandList[::nn::i2c::CommandListLengthCountMax];
        ::nn::i2c::CommandListFormatter commandListFormatter(commandList, sizeof(commandList));
        ::std::memset(commandList, 0, sizeof(commandList));

        commandListFormatter.EnqueueSendCommand(I2cTransStart, pInData, sendDataBytes);
        commandListFormatter.EnqueueReceiveCommand(I2cTransStop, receiveDataBytes);

        result = ::nn::i2c::ExecuteCommandList(
            pBuffer, receiveDataBytes, m_Session, commandList, commandListFormatter.GetCurrentLength());

        if (::nn::i2c::ResultBusBusy::Includes(result))
        {
            ::nn::os::SleepThread(
                ::nn::TimeSpan::FromMilliSeconds(I2cAccessRetryIntervalMs));
            continue;
        }

        break;
    }

    return result;
}

}}} // namespace nnd::ftm::detail
