﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       ftm ライブラリのデータ型や定数の定義です。
 */

#pragma once

#include <nn/nn_Common.h>

namespace nnd {
namespace ftm {

const int       FtmMaxTouchNumber         = 10;   //!< FTM シリーズが検出できる同時タッチの最大数を表します。
const int       FtmMaxEventReportCount    = 64;   //!< FTM シリーズで保持できる EventReport の最大数を表します。
const size_t    FtmMaxEventReportByteSize = 8;    //!< FTM シリーズ内で最大の EventReport のデータサイズを表します。

/**
 * @brief       FTM が発行するイベントのIDです。
 */
enum class EventId: uint8_t
{
    NoEvents,                   //!< イベントはありません。
    TouchEnter,                 //!< タッチの開始イベントを検出しました。タッチに関する情報を取得できます。
    TouchLeave,                 //!< タッチの終了イべントを検出しました。タッチに関する情報を取得できます。
    TouchMotion,                //!< タッチ点が移動したことを検出しました。タッチに関する情報を取得できます。
    Error,                      //!< エラーが検出されました。
    ControllerReady,            //!< コントローラの準備が整いました。
    CompensationReady,          //!< Compensation データの準備が整いました。
    SleepOutContollerReady,     //!< スリープを解除した後にコントローラの準備が整いました。
    Status,                     //!< 任意のステータス変更処理が完了しました。
    Inform,                     //!< GPIO ステートのレポートを受け取りました。
    Others,                     //!< 未定義のイベントが検出されました。
};


/**
 * @brief           ステータス完了イベント EventId_Status で検出可能なステータスの種類です。
 */
enum class StatusType: uint8_t
{
    AutoTuneMutual,                     //!< 相互オートチューン処理が完了しました。
    AutoTuneSelf,                       //!< 自己オートチューン処理が完了しました。
    FlashWriteConfig,                   //!< Flash への Config 書き込みが完了しました。
    FlashWriteNodeCompensationMemory,   //!< Flash への Node Compensation Memory 書き込みが完了しました。
    ForceCalSelfAndMutual,              //!< すべてのフォースキャリブレーションが完了しました。
    ForceCalSelfOnly,                   //!< 自己フォースキャリブレーションのみが完了しました。
    Others,                             //!< 未定義のステータスが変更されました。
};

/**
 * @brief           切り替え可能なセンシングモードの種類です。
 */
enum class SensingMode : uint8_t
{
    FingerAndStylus,                    //!< 指とスタイラスの両検出が可能なモードです。
    FingerOnly,                         //!< 指のみが検出されるモードです。
    StylusOnly,                         //!< スタイラスのみが検出されるモードです。
};

/**
 * @brief           FTM から取得できるタッチに関する情報をまとめた構造体です。
 *
 * @details         タッチに関するイベントが発生した際に、イベントレポートとして取得する情報です。
 */
struct TouchEventReport
{
    uint8_t         touchId;        //!< 識別用 ID
    NN_PADDING1;
    uint16_t        x;              //!< X 座標 [pixel]
    uint16_t        y;              //!< Y 座標 [pixel]
    uint16_t        major;          //!< 長軸のサイズ [pixel]
    uint16_t        minor;          //!< 短軸のサイズ [pixel]
    int32_t         milliDegree;    //!< 回転角 [milli-degree]
};

/**
 * @brief           FTM から取得できるステータスに関する情報をまとめた構造体です。
 */
struct GpioInformEventReport
{
    uint8_t     gpio0;      //!< GPIO0 の状態
    uint8_t     gpio1;      //!< GPIO1 の状態
    uint8_t     gpio2;      //!< GPIO2 の状態
};

/**
 * @brief           ReadEventReport で取得するデータの共用体です。
 */
union EventReportContent
{
    TouchEventReport        touchReport;    //!< タッチイベントとして解析したデータ
    StatusType              statusReport;   //!< ステータスイベントとして解析したデータ
    GpioInformEventReport   gpioReport;     //!< GPIO状態イベントとして解析したデータ
};

/**
 * @brief           FTM から取得できるイベントレポートです。
 *
 * @details         データの内容を解釈するためには、
 *                  パーサ関数を使用してデータの解析を行って下さい。
 */
struct EventReport
{
    EventId             eventId;    //!< 取得したイベントID
    EventReportContent  content;    //!< イベントのデータ本体
};

/**
 * @brief           FTM のファームウェアバージョン用の構造体です。
 *
 * @details         プログラムバージョンもしくはコンフィグバージョンが更新されると
 *                  リリースバージョンも更新されます。
 */
struct FirmwareVersion
{
    int release;        //!< リリースバージョン
    int program;        //!< プログラムバージョン
    int configuration;  //!< コンフィグバージョン
};

/**
 * @brief           FTM のファームウェア更新時に使用するバイナリデータの入力用関数の型です。
 */
typedef nn::Result (*FirmwareInputFunctionPointer)(
    char* buffer, void* pParameter, int64_t offset, size_t size);

} // ftm
} // nnd
