﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       xcd ライブラリの TeraMCU アップデートに関係する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/fs.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Tera.h>
#include <nn/xcd/xcd_FirmwareTypes.h>

namespace nn { namespace xcd {

/**
 * @brief       Tera MCU のファームウェアバージョン情報です。
 */
struct McuVersionData
{
    bool        isCorrupted;        //!< データが壊れているかどうかのフラグ
    bool        isIapCorrupted;     //!< IAP が壊れているかどうかのフラグ
    DeviceType  deviceType;         //!< デバイスの種類
    NN_PADDING1;
    uint16_t    major;              //!< Tera MCU のメジャーバージョン
    uint16_t    minor;              //!< Tera MCU のマイナーバージョン
    char        _reserve[24];       // 予約

    uint32_t ToUint32() const NN_NOEXCEPT
    {
        return (static_cast<uint32_t>(major << 16) & 0xffff0000) |
               (static_cast<uint32_t>(minor) & 0x0000ffff);
    }
};

/**
 * @brief       Tera Firmware Version 030b
 * @details     Joycon preinstalled version
 */
const McuVersionData FirmwareVersionTera_030b = { false, false, DeviceType_Unknown, 0x03, 0x0b };

/**
 * @brief       Tera Firmware Version 0410
 * @details     Add image transfer fast mode
 */
const McuVersionData FirmwareVersionTera_0410 = { false, false, DeviceType_Unknown, 0x04, 0x10 };

/**
 * @brief       Tera Firmware Version 0411
 * @details     Handanalysis brushup and image transfer disable on/off subtraction bug fix
 */
const McuVersionData FirmwareVersionTera_0411 = { false, false, DeviceType_Unknown, 0x04, 0x11 };

/**
 * @brief       Tera Firmware Version 0412
 * @details     Add WriteRegisterAck to ReadData packet
 */
const McuVersionData FirmwareVersionTera_0412 = { false, false, DeviceType_Unknown, 0x04, 0x12 };

/**
 * @brief       Tera Firmware Version 0518
 * @details     Add AdaptiveClustering func, fix Handanalysis bug.
 */
const McuVersionData FirmwareVersionTera_0518 = { false, false, DeviceType_Unknown, 0x05, 0x18 };

/**
 * @brief       Tera MCU のファームウェア更新状況です。
 */
enum McuUpdateState : uint8_t
{
    McuUpdateState_Boot = 0,        //!< ファームウェア更新用のステートに遷移中です。
    McuUpdateState_RomErasing,      //!< Flash ROM を消去しています。
    McuUpdateState_Writing,         //!< ファームウェア書き込み中です。
    McuUpdateState_Reboot,          //!< ファームウェア更新後の再起動中です。
    McuUpdateState_End,             //!< 更新処理が終了しました。更新の成否はファームウェアバージョンで確認する必要があります。
};

/**
 * @brief         Tera MCU の更新状況に関する情報です。
 */
struct McuUpdateStateInfo
{
    McuUpdateState state;       //!< 更新の内部ステート
    NN_PADDING1;
    uint16_t       progress;    //!< 更新進捗状況 (0 - 100)
};

/**
 * @brief       Tera MCU ファームウェアのバージョンを取得します。
 *
 * @details     Tera MCU ファームウェアのバージョンを取得します。
 *              バージョンを取得するには、Tera MCU を起動させておく必要があります。
 *              この関数を呼ぶ前に、 @ref McuState_Nfc または @ref McuState_Ir を指定して @ref SetMcuState を呼び出しておいてください。
 *              (@ref RequestMcuVersion を使用する場合は、 @ref SetMcuState を呼ぶ必要はありません)
 *
 *              正常にバージョンを取得できた後は、デバイスが切断するまで Tera MCU が起動していない状態でもバージョンを取得できます。
 *              ファームウェア壊れと判定された場合は、 @ref SetMcuState 呼び出し時にバージョンの取得が再試行されます。
 *
 * @param[out]  pOutMcuVersionData          バージョンデータ
 * @param[in]   handle                      バージョンを取得するデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                        処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,              デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultMcuVersionNotAvailable,    MCU のバージョン取得が完了していません。}
 * @endresult
 *
 * @pre
 *  - XCD ライブラリは初期化済である必要があります。
 *  - @a pOutMcuVersionData != nullptr
 */
Result GetMcuVersion(McuVersionData* pOutMcuVersionData, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       Tera MCU ファームウェアのバージョン取得を要求します。
 *
 * @details     Tera MCU ファームウェアのバージョン取得要求を発行します。
 *              この関数を呼び出すと、内部で Tera MCU を起動させ、バージョンの取得を行います。
 *              バージョンの取得が完了すると @ref GetMcuVersion が成功する状態になります。
 *
 * @param[in]   handle                      バージョンを取得するデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                        処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,              デバイスが接続されていません。}
 * @endresult
 *
 * @pre
 *  - XCD ライブラリは初期化済である必要があります。
 */
Result RequestMcuVersion(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       Tera MCU の更新を開始します
 *
 * @details     Tera MCU の更新を開始します。
 *              処理が成功すると、データフォーマットが @ref PeriodicDataFormat_McuUpdate に変更されます。
 *              更新が終了すると、データフォーマットが @ref PeriodicDataFormat_Basic に変更されます。
 *              更新が進行している間は、 @ref SetDataFormat でデータフォーマットを変更しないでください。
 *
 * @param[in]   deviceHandle                ファームウェアを更新するデバイスのハンドル
 * @param[in]   image                       ファームウェアイメージ
 * @param[in]   pFinishEvent                ファームウェア更新が終了した際に通知を受けるイベント
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                        処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,              デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultNotSupported,              当該プラットフォームでは、この機能を使用できません。}
 *      @handleresult{nn::xcd::ResultMcuInvalidFirmwareImage,   不正なファームウェアファイルです。}
 * @endresult
 *
 * @pre
 *  - XCD ライブラリは初期化済である必要があります。
 *  - @a pFinishEvent != nullptr
 *  - @a pFinishEvent は作成済みである必要があります。
*/
Result StartMcuUpdate(
    DeviceHandle deviceHandle,
    const FirmwareImage& image,
    nn::os::SystemEventType* pFinishEvent) NN_NOEXCEPT;

/**
 * @brief       Tera MCU 全領域の更新を開始します
 *
 * @details     Tera MCU の IAP を含めた全領域の更新を開始します。
 *              詳細は @ref StartMcuUpdate を参照してください。
 *
 * @param[in]   deviceHandle                ファームウェアを更新するデバイスのハンドル
 * @param[in]   image                       ファームウェアイメージ
 * @param[in]   pFinishEvent                ファームウェア更新が終了した際に通知を受けるイベント
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                        処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,              デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultNotSupported,              当該プラットフォームでは、この機能を使用できません。}
 *      @handleresult{nn::xcd::ResultMcuInvalidFirmwareImage,   不正なファームウェアファイルです。}
 * @endresult
 *
 * @pre
 *  - XCD ライブラリは初期化済である必要があります。
 *  - @a pFinishEvent != nullptr
 *  - @a pFinishEvent は作成済みである必要があります。
*/
Result StartMcuUpdateFull(
    DeviceHandle deviceHandle,
    const FirmwareImage& image,
    nn::os::SystemEventType* pFinishEvent) NN_NOEXCEPT;

/**
 * @brief       Tera MCU の更新を中止します
 *
 * @details     Tera MCU の更新を中止します。
 *              更新を中止した場合、Tera MCU のファームウェアは破損状態になります。
 *
 * @param[in]   handle                      更新を中止するデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,      デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultNotSupported,      当該プラットフォームでは、この機能を使用できません。}
 * @endresult
 *
 * @pre        - XCD ライブラリは初期化済である必要があります。
 */
Result AbortMcuUpdate(DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       Tera MCU の更新状況を取得します。
 *
 * @details     Tera MCU の更新状況を取得します。
 *
 * @param[out]  pOutMcuUpdateStateInfo      更新状況
 * @param[in]   handle                      更新状況を取得するデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,      デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultNotSupported,      当該プラットフォームでは、この機能を使用できません。}
 *      @handleresult{nn::xcd::ResultInvalidMcuState,   MCU のステートが正しくありません。}
 * @endresult
 *
 * @pre
 *  - XCD ライブラリは初期化済である必要があります。
 *  - @a pOutMcuUpdateStateInfo != nullptr
 */
Result GetMcuUpdateState(McuUpdateStateInfo* pOutMcuUpdateStateInfo, DeviceHandle handle) NN_NOEXCEPT;

}} // namespace nn::xcd
