﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       xcd ライブラリの TeraMCU に関係する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/xcd/xcd_Device.h>

namespace nn { namespace xcd {

/**
 * @brief       Tera MCU の内部ステートです。
 */
enum McuState : uint8_t
{
    McuState_Idle,                 //!< MCU を起動した直後に遷移するステートです。
    McuState_Standby,              //!< Tera MCU の機能を使用せずに消費電力を抑えるステートです。
    McuState_Background,           //!< NFC や モーションIRカメラを利用せず Tera MCU の消費電力を抑えるステートです。
    McuState_Nfc,                  //!< NFC を利用するためのステートです。
    McuState_Ir,                   //!< モーションIRカメラを利用するためのステートです。
    McuState_Initializing,         //!< MCU の初期化処理を行うステートです。このステートの間は、MCU を操作できません。
    McuState_FirmwareUpdate,       //!< MCU のファームウェア更新を行うステートです。
    McuState_Unknown               //!< 不明なステートです。
};

/**
 * @brief       Tera MCU の内部ステートを取得します。
 *
 * @details     Tera MCU の内部ステートを取得します。
 *
 * @param[out]  pOutMcuState                ステートの取得先
 * @param[in]   handle                      ステートを取得するデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,            処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,  デバイスが接続されていません。}
 * @endretresult
 *
 * @pre
 *  - XCD ライブラリは初期化済である必要があります。
 *  - pOutMcuState != nullptr
 */
Result GetMcuState(McuState* pOutMcuState, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       Tera MCU の内部ステートを設定します。
 *
 * @details     Tera MCU の内部ステートを @a mcuState で設定したステートに遷移させます。
 *              事前に @ref SetDataFormat を呼び出してデータフォーマットを @ref PeriodicDataFormat_MCU に設定しておく必要があります。
 *
 * @param[in]   mcuState                    設定したいステート
 * @param[in]   handle                      ステートを設定するデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,            処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,  デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultMcuBusy,       状態遷移処理中で MCU がビジー状態です。}
 * @endretresult
 *
 * @pre
 *  - XCD ライブラリは初期化済である必要があります。
 *  - @a mcuState は、 @ref McuState_Standby, @ref McuState_Background, @ref McuState_Nfc, @ref McuState_Ir のいずれかである必要があります。
 */
Result SetMcuState(McuState mcuState, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       Tera MCU の内部ステートを設定します。
 *
 * @details     Tera MCU の内部ステートを @a mcuState で設定したステートに遷移させます。
 *              @ref SetMcuState とは異なり、事前に @ref SetDataFormat を呼び出しておく必要はありません。
 *
 * @param[in]   mcuState                    設定したいステート
 * @param[in]   handle                      ステートを設定するデバイスへのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,            処理に成功しました。}
 *      @handleresult{nn::xcd::ResultNotConnected,  デバイスが接続されていません。}
 *      @handleresult{nn::xcd::ResultMcuBusy,       状態遷移処理中で MCU がビジー状態です。}
 * @endretresult
 *
 * @pre
 *  - XCD ライブラリは初期化済である必要があります。
 *  - @a mcuState は、 @ref McuState_Standby, @ref McuState_Background, @ref McuState_Nfc, @ref McuState_Ir のいずれかである必要があります。
 */
Result SetMcuStateImmediate(McuState mcuState, DeviceHandle handle) NN_NOEXCEPT;

/**
 * @brief       Tera MCU のハードウェア異常エミュレーション機能の有効状態を設定します。
 *
 * @detail      Tera MCU のハードウェア異常エミュレーション機能の有効状態を設定します。
 *              有効にすると、Tera MCU が応答しない状態として振る舞います。
 *
 * @param[in]   isEnabled                   有効状態
 *
 * @pre
 *  - XCD ライブラリは初期化済である必要があります。
 */
void SetMcuHardwareErrorEmulationEnabled(bool isEnabled) NN_NOEXCEPT;

/**
 * @brief       Tera MCU のファームウェア更新失敗エミュレーション機能の有効状態を設定します。
 *
 * @detail      Tera MCU のファームウェア更新失敗エミュレーション機能の有効状態を設定します。
 *              有効にすると、Tera MCU のファームウェア更新が常に Verify で失敗するようになります。
 *
 * @param[in]   isEnabled                   有効状態
 *
 * @pre
 *  - XCD ライブラリは初期化済である必要があります。
 */
void SetMcuUpdateFailureEmulationEnabled(bool isEnabled) NN_NOEXCEPT;

}} // namespace nn::xcd
