﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       xcd ライブラリに関係する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/os/os_MultipleWait.h>

namespace nn { namespace xcd {

/**
 * @brief       xcd ライブラリを初期化します。
 *
 * @details     xcd ライブラリを初期化します。
 *              本関数は任意のタイミングで繰り返し呼び出すことが可能です。
 *              本関数の呼び出し後に抜き差しが起きた場合であっても、本関数の呼出しは不要です。
 *              xcdでは内部でワーカースレッドを保持しないため、何か処理を実行するタイミングでイベントを発行し、Proceed() を呼び出してもらう必要があります。
 *              本関数では、外部のスレッドでイベントを待ち受けるためのMultiWaitType を受け取ります。
 *              本関数はスレッドセーフです。
 *
 * @param[in]   pMultiWait                       xcd内部の状態に更新がありProceed() を実行してもらう必要があることを通知するイベント
 *
 * @pre         - pMultiWait != nullptr
 *              - pMultiWait は初期化状態
 *
 * @post        -  xcd ライブラリは初期化済の状態になります
 *
 *
 */
void Initialize(nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT;

/**
 * @brief       xcd ライブラリを終了します。
 */
void Finalize() NN_NOEXCEPT;

/**
 * @brief       xcdからのイベントかどうかを判定し、該当イベントであれば必要な処理を行う関数
 *
 * @details     指定された同時待ちイベントオブジェクトが Xcd 由来のものかどうかを判定し、
 *              Xcd 由来のものであった場合に必要な処理を行います。
 *              Initialize() で渡したnn::os::MultiWaitType がシグナルされた際に、本関数を実行してください。
 *              本関数は引数に実際にシグナルされたイベントを持つ nn::os::MultiWaitHolderType を受け取ります。
 *
 * @param[in]   pMultiWaitHolder                   シグナルされたオブジェクトを保持するホルダー
 *
 * @pre         - pMultiWaitHolder != nullptr
 *
 * @return      pMultiWaitHolder が Xcd 由来のイベントの場合は true、そうでない場合は false
 *
 */
bool Proceed(const nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT;

/**
 * @brief       xcd の定常的な処理を行います。
 *
 * @details     データの送受信など、xcd 内部で定常的に発生する処理を実行します。
 *              WaitEvent 関数の戻り値が nullptr の場合に呼び出してください。
 */
void ProceedPeriodic() NN_NOEXCEPT;

/**
 * @brief       xcd で定常的に処理するイベントの発生を待機します。
 *
 * @return      Xcd 由来のイベントを表す MultiWaitHolderType もしくは nullptr。
 *              nullptr の場合は ProceedPeriodic 、それ以外の場合は Proceed を呼び出してください。
 */
nn::os::MultiWaitHolderType* WaitEvent() NN_NOEXCEPT;

/**
 * @brief       xcd ライブラリの処理をサスペンドします。
 *
 * @details     スリープに入る前に一時的に Xcd ライブラリの処理を停止する場合に使用します。
 *
 */
void SuspendForSleep() NN_NOEXCEPT;

/**
 * @brief       xcd ライブラリの処理をレジュームします。
 *
 * @details     スリープから復帰した際に Xcd ライブラリの処理を再開する場合に使用します。
 *
 */
void ResumeFromSleep() NN_NOEXCEPT;

}} // namespace nn::xcd
