﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       xcd ライブラリに関係する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/xcd/xcd_BleDevice.h>
#include <nn/xcd/xcd_BleTypes.h>

namespace nn { namespace xcd {

/**
 * @brief       BLE デバイスに対しての各操作に対する実行結果を取得します。
 *
 * @details     BLE デバイスに対して非同期に行われる各操作についての実行結果、レスポンスを取得します。
 *
 *              各 BLE デバイスの操作は本関数が呼ばれるまでビジー状態とします。
 *              次の操作を行う場合に必ず 本関数を呼び結果を取得してから次の操作を行うようにしてください。
 *
 * @param[out]  pOutValue               各操作に対する実行結果
 * @param[in]   handle                  コネクションハンドル
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess                    処理に成功しました。
 * @retval      ResultNotConnected               デバイスが接続されていません。
 * @retval      ResultBluetoothNoOperationResult 操作の実行結果が存在しません。
 */
Result GetBleDeviceOperationResult(BleDeviceOperationResult* pOutValue, const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスの各操作が完了したことを通知するイベントオブジェクトを登録します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 *
 * @param[in]   pSystemEvent  イベントオブジェクトのポインタ
 * @param[in]   handle        コネクションハンドル
 */
Result RegisterBleDeviceOperationEvent(nn::os::SystemEventType* pSystemEventconst, const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスの演出を再生します。操作に対する応答を受信した際にイベントを通知します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultNotSupported          デバイスがこの機能をサポートしていません。
 * @retval      ResultBluetoothLeBusy       ビジー状態です。システム内で操作中だった場合もこのエラーを返すことがあります。
 *
 * @param[in]   number        演出のナンバー
 * @param[in]   handle        コネクションハンドル
 *
 * @details
 *  操作に対する応答は GetOperationResult() で取得してください。
 *  演出はナンバーと紐づけてあらかじめ BLE デバイスに保存されている必要があります。
 *  同じ ID の場合でも、実際の挙動は BLE デバイスに保存された演出の定義によって異なります。
 */
Result PlayDeviceActivity(uint16_t number, const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスの FrMode の動作種別を設定します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultNotSupported          デバイスがこの機能をサポートしていません。
 * @retval      ResultBluetoothLeBusy       ビジー状態です。システム内で操作中だった場合もこのエラーを返すことがあります。
 *
 * @param[in]   frModeType    FrMode の動作種別
 * @param[in]   handle        コネクションハンドル
 *
 * @details
 *  操作に対する応答は GetOperationResult() で取得してください。
 *  FrMode の具体的な動作仕様は、BLE デバイスの仕様を確認してください。
 */
Result SetFrModeType(uint32_t frModeType, const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスの Database Entry のデータを更新します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultNotSupported          デバイスがこの機能をサポートしていません。
 * @retval      ResultBluetoothLeBusy       ビジー状態です。システム内で操作中だった場合もこのエラーを返すことがあります。
 *
 * @param[in]   config        書き込む Database Entry の設定
 * @param[in]   size          書き込むデータのサイズ
 * @param[in]   pEntryBuffer  書き込む Entry のバッファ
 * @param[in]   handle        コネクションハンドル
 *
 * @details
 *  操作に対する応答は GetOperationResult() で取得してください。
 *  Entry の更新が完了するまでほかの Entry の更新はできません。
 */
Result WriteDatabaseEntry(const BleDeviceDatabaseEntryConfig& config, size_t size, const void* pEntryBuffer, const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスのデータベース識別バージョンを設定します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultNotSupported          デバイスがこの機能をサポートしていません。
 * @retval      ResultBluetoothLeBusy       ビジー状態です。システム内で操作中だった場合もこのエラーを返すことがあります。
 *
 * @param[in]   version       データベース識別バージョン
 * @param[in]   handle        コネクションハンドル
 *
 * @details
 *  操作に対する応答は GetOperationResult() で取得してください。
 */
Result SetDatabaseIdentificationVersion(int32_t version, const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスのデータベース識別バージョンを取得します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultNotSupported          デバイスがこの機能をサポートしていません。
 * @retval      ResultBluetoothLeBusy       ビジー状態です。システム内で操作中だった場合もこのエラーを返すことがあります。
 *
 * @param[in]   handle        コネクションハンドル
 *
 * @details
 *  操作に対する応答は GetOperationResult() で取得してください。
 */
Result GetDatabaseIdentificationVersion(const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスの StepCount を取得します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultNotSupported          デバイスがこの機能をサポートしていません。
 * @retval      ResultBluetoothLeBusy       ビジー状態です。システム内で操作中だった場合もこのエラーを返すことがあります。
 *
 * @param[in]   handle        コネクションハンドル
 *
 * @details
 *  操作に対する応答は GetOperationResult() で取得してください。
 */
Result GetStepCount(const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスの StepCount を初期化します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultNotSupported          デバイスがこの機能をサポートしていません。
 * @retval      ResultBluetoothLeBusy       ビジー状態です。システム内で操作中だった場合もこのエラーを返すことがあります。
 *
 * @param[in]   handle        コネクションハンドル
 *
 * @details
 *  操作に対する応答は GetOperationResult() で取得してください。
 */
Result ResetStepCount(const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスの Step 機能を設定します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultNotSupported          デバイスがこの機能をサポートしていません。
 * @retval      ResultBluetoothLeBusy       ビジー状態です。システム内で操作中だった場合もこのエラーを返すことがあります。
 *
 * @param[in]   isEnabled     有効/無効の設定
 * @param[in]   handle        コネクションハンドル
 *
 * @details
 *  操作に対する応答は GetOperationResult() で取得してください。
 */
Result EnableStepCounter(bool isEnabled, const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスの Application セクションを読み出します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultNotSupported          デバイスがこの機能をサポートしていません。
 * @retval      ResultBluetoothLeBusy       ビジー状態です。システム内で操作中だった場合もこのエラーを返すことがあります。
 *
 * @param[in]   address       読み込み先のアドレス
 * @param[in]   readSize      読み込むデータのサイズ
 * @param[in]   handle        コネクションハンドル
 *
 * @details
 *  操作に対する応答は GetOperationResult() で取得してください。
 */
Result ReadApplicationSection(int32_t address, size_t readSize, const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスの Application セクションへ書き込みを行います。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultNotSupported          デバイスがこの機能をサポートしていません。
 * @retval      ResultBluetoothLeBusy       ビジー状態です。システム内で操作中だった場合もこのエラーを返すことがあります。
 *
 * @param[in]   address       書き込み先のアドレス
 * @param[in]   pBuffer       書き込むデータ
 * @param[in]   size          書き込むデータのサイズ
 * @param[in]   handle        コネクションハンドル
 *
 * @details
 *  操作に対する応答は GetOperationResult() で取得してください。
 */
Result WriteApplicationSection(int32_t address, const void* pBuffer, size_t size, const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスに格納された UniqueCode を取得します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultNotSupported          デバイスがこの機能をサポートしていません。
 * @retval      ResultBluetoothLeBusy       ビジー状態です。システム内で操作中だった場合もこのエラーを返すことがあります。
 *
 * @param[in]   handle        コネクションハンドル
 *
 * @details
 *  操作に対する応答は GetOperationResult() で取得してください。
 */
Result ReadContentUniqueCode(const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスに格納された UniqueCode を無効にします。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultNotSupported          デバイスがこの機能をサポートしていません。
 * @retval      ResultBluetoothLeBusy       ビジー状態です。システム内で操作中だった場合もこのエラーを返すことがあります。
 *
 * @param[in]   handle        コネクションハンドル
 *
 * @details
 *  操作に対する応答は GetOperationResult() で取得してください。
 */
Result SetContentUniqueCodeInvalid(const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスの機能を一時停止/再開します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultNotSupported          デバイスがこの機能をサポートしていません。
 * @retval      ResultBluetoothLeBusy       ビジー状態です。システム内で操作中だった場合もこのエラーを返すことがあります。
 *
 * @param[in]   featureFlagSet     機能の一時停止/再開のフラグ
 * @param[in]   handle             コネクションハンドル
 *
 * @details
 *  操作に対する応答は GetOperationResult() で取得してください。
 */
Result SuspendFeature(uint32_t featureFlagSet, const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスのサポートを無効化します。
 *
 * @details
 *  Initialize より前に呼ばれる必要があります。
 */
void DisableBle() NN_NOEXCEPT;

/**
 * @brief       BLE デバイスの Play Log を取得します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultNotSupported          デバイスがこの機能をサポートしていません。
 * @retval      ResultBluetoothLeBusy       ビジー状態です。システム内で操作中だった場合もこのエラーを返すことがあります。
 *
 * @param[in]   index              取得する Play Log のフィールドを示すインデックス
 * @param[in]   handle             コネクションハンドル
 *
 * @details
 *  操作に対する応答は GetOperationResult() で取得してください。
 */
Result ReadDevicePlayLog(uint16_t index, const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスの Play Log を初期化します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultNotSupported          デバイスがこの機能をサポートしていません。
 * @retval      ResultBluetoothLeBusy       ビジー状態です。システム内で操作中だった場合もこのエラーを返すことがあります。
 *
 * @param[in]   index              初期化する Play Log のフィールドを示すインデックス
 * @param[in]   handle             コネクションハンドル
 *
 * @details
 *  操作に対する応答は GetOperationResult() で取得してください。
 */
Result ResetDevicePlayLog(uint16_t index, const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       登録状態の有無にかかわらず全ての Palma の接続待ち受けを開始します
 *
 * @details
 *  多重で呼んでも問題ありません。処理は非同期に行われ実際のスキャンが開始するまで時間がかかる場合があります。
 */
void SetIsPalmaAllConnectable(bool connectable) NN_NOEXCEPT;


/**
 * @brief       登録済みの Palma の接続待ち受けを開始します
 *
 * @details
 *  多重で呼んでも問題ありません。処理は非同期に行われ実際のスキャンが開始するまで時間がかかる場合があります。
 */
void SetIsPalmaPairedConnectable(bool connectable) NN_NOEXCEPT;

/**
 * @brief       接続状態にある Palma の登録を行います
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @details
 *  接続状態の場合処理が成功します。登録処理中のエラー状態をハンドリングすることはできません。
 */
Result PairPalma(const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスの Database Entry のデータ更新を中断します。
 *
 * @param[in]   handle        コネクションハンドル
 *
 * @details
 *  BLE デバイスが接続されていない場合、Database Entry のデータ更新中でない場合は何も行われません。
 */
void CancelWriteDatabaseEntry(const BleConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスのシステム用の演出を再生します。
 *
 * @return      実行結果を返します。
 * @retval      ResultSuccess               処理に成功しました。
 * @retval      ResultNotConnected          デバイスが接続されていません。
 * @retval      ResultNotSupported          デバイスがこの機能をサポートしていません。
 * @retval      ResultBluetoothLeBusy       ビジー状態です。システム内で操作中だった場合もこのエラーを返すことがあります。
 *
 * @param[in]   number        演出のナンバー
 * @param[in]   handle        コネクションハンドル
 *
 * @details
 *  演出はナンバーと紐づけてあらかじめ BLE デバイスに保存されている必要があります。
 *  同じ ID の場合でも、実際の挙動は BLE デバイスに保存された演出の定義によって異なります。
 */
Result PlayDeviceSystemActivity(uint16_t number, const BleConnectionHandle& handle) NN_NOEXCEPT;

}} // namespace nn::xcd
