﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_SystemEvent.h>
#include <nn/mbuf/mbuf_Definitions.h>

#include "wlan_Common.h"
#include "wlan_MacAddress.h"
#include "wlan_Ssid.h"
#include "wlan_StateDefinition.h"
#include "wlan_Types.h"

namespace nn {
namespace wlan {

/*!
   @name       すれちがい通信 関連
   @{
 */


//--------------------------------------------------------------------------
/**
 *    @brief        WLANライブラリのすれちがい通信のサービスを開始します。この初期化を行っていないと、すれちがい通信に関する API は使用できません。本関数は必ず成功します。@n
 *    既に本関数を呼んでいた場合、何も行わず成功が返ります。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess    初期化に成功。
 */
nn::Result InitializeDetectManager() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 *    @brief        WLANライブラリのすれちがい通信のサービスを終了します。本関数は必ず成功します。
 *    @ref InitializeDetectManger を呼んでいない状態で呼んだ場合、何もせず成功が返ります。(developビルドの場合アサートします)
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess    終了処理に成功。
 */
nn::Result FinalizeDetectManager() NN_NOEXCEPT;


namespace Detect {

//--------------------------------------------------------------------------
/**
 *    @brief        すれちがい通信モードをオープンします。1chをHomeチャンネルとしてオープンします。@n
 *    本関数を呼んだ時点では実際の通信は開始されません。
 *    @details      実行可能ステート：WlanState_Ready @n
 *                                    WlanState_DetectIdle (既に目的ステートにいるため何も起こりません) @n
 *
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess               状態遷移に成功。
 *    @retval       ResultInvalidState          不正なステート時に呼び出されました。
 *    @retval       ResultWlanDeviceAbnormal    無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result OpenMode() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        すれちがい通信モードをオープンします。モード開始と同時にHomeチャンネルの指定が出来ます。（デバッグ用途） @n
 *    本関数を呼んだ時点では実際の通信は開始されません。
 *    @details      実行可能ステート：WlanState_Ready @n
 *                                    WlanState_DetectIdle (既に目的ステートにいるため何も起こりません)
 *
 *    @param[in]    channel     Homeチャンネル。開発用途のため5GHzも指定可能ですが使用には注意して下さい。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 *    @retval       ResultWlanDeviceAbnormal    無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result OpenMode(uint16_t channel) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        すれちがい通信モードをクローズします。
 *    @details      実行可能ステート：WlanState_Ready (既に目的ステートにいるため何も起こりません)@n
 *                                    WlanState_DetectIdle
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 *    @retval       ResultWlanDeviceAbnormal    無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result CloseMode() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        すれちがい通信を開始します。
 *    @details      実行可能ステート：WlanState_DetectIdle @n
 *                                    WlanState_Detect (既に目的ステートにいるため何も起こりません)
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 *    @retval       ResultWlanDeviceAbnormal    無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result StartCommunication() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        すれちがい通信を停止します。
 *    @details      実行可能ステート：WlanState_DetectIdle (既に目的ステートにいるため何も起こりません)@n
 *                                    WlanState_Detect
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         状態遷移に成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 *    @retval       ResultWlanDeviceAbnormal    無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result StopCommunication() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        無線プロセスのステートを取得します。各ステートは、@ref State を参照してください。
 *    @details      実行可能ステート： 全ステート
 *
 *    @param[out]   pOutState    現在のステート(@ref State)を返します。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultSuccess         成功。
 */
nn::Result GetState(WlanState* pOutState) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        無線デバイスで使用の許可されたチャンネルリストを返します。
 *    @details      実行可能ステート： 全ステート
 *
 *    @param[out]   outChannels    無線デバイスで使用の許可されたチャンネルリストが返ります。例えば、1~11chと40chが許可チャンネルの場合、
 *                  outChannels[0] = 1;
 *                  outChannels[1] = 2;
 *                  ...
 *                  outChannels[10] = 11;
 *                  outChannels[11] = 40;
 *                  と言う値が入り、以降は0で埋められます。@n
 *                  チャンネルは必ず先頭から連続で昇順に格納されます。（0は有効なチャンネルではないので、昇順リストには含まれません。また、負数が含まれることはありません。）
 *    @param[out]   pOutCount 許可チャンネルの数が返ります。@ref channels の要素数として利用してください。上記の例ですと、12という数値が返ります。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultSuccess         成功。
 */
nn::Result GetAllowedChannels(int16_t (&outChannels)[WirelessChannelsCountMax], uint32_t* pOutCount) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        無線チップのMACアドレスを取得します。
 *    @details      実行可能ステート： 全ステート
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultSuccess         成功。
 */
nn::Result GetMacAddress(MacAddress* pOutMac) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        Action Frameを単発で送信します。@n
 *    すれちがい用ActionFrameは以下のデータフォーマットで送信されます。「Category」、「OUI」および @n
 *    「@ref DetectHeader のhash以外のフィールド」はwlanプロセス側で自動的に埋められます。@n
 *    本関数を使用する際は、Subtype、@ref DetectHeader に入れるhash、dataを指定して下さい。
 *    +--------------------------------------------------------------------------------+ @n
 *    | Category |      OUI      | Subtype | Reserved |  Detect Header  |      data    | @n
 *    |  1 byte  |    3 bytes    |  1 byte |  1 byte  |    12 bytes     |  ~1300 bytes | @n
 *    +--------------------------------------------------------------------------------+ @n
 *    @details      実行可能ステート：WlanState_Detect @n
 *
 *    @param[in]    dstMac    宛先MACアドレス。ZeroMacAddressは無効です。@n
 *    @param[in]    subtype   Subtype。ActionFrameType_Detect を指定して下さい。@n
 *    @param[in]    hash      すれちがいデータのハッシュ値 @n
 *    @param[in]    pData     Action Frameに付加するデータ。上記説明参照。@n
 *    @param[in]    size      Action Frameに付加するデータのサイズ。1300Byte以下を指定してください。@n
 *    @param[in]    dwellTime 送信後、送信先チャンネルに留まっている時間。留まっている間は電波受信状態となります。@n
 *    例えば、相手からの返事を待ちたい場合はこの値を調整して下さい。単位はmsecです。最大100msec待てます。@n
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultCommandFailure  送信に失敗しました。送信完了バッファの解放が追い付かず詰まっている可能性があります。少し間を空けてから再度試して下さい。
 *    @retval       ResultWlanDeviceAbnormal 無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result PutActionFrameOneShot(const MacAddress& dstMac, ActionFrameType subtype, DetectHash hash,
        const uint8_t* pData, size_t size, uint32_t dwellTime) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        定期的に送信するAction Frameの設定を行います。@n
 *    指定したintervalでActionFrameが定期的に送信されます。@n
 *    このActionFrameの宛先は指定できません。Broadcastアドレス宛てとなっています。@n
 *    @ref CancelPeriodicActionFrame を行うとAction Frameの送信は停止され、内容もクリアされます。@n
 *    @ref pData へのデータの入れ方は @ref PutActionFrameOneShot と同様です。@n
 *    送信中に再度本関数を呼ぶと、内容が更新されたものが送信されるようになります。実際にAction Frameに反映されるのは、セット後の次の送信タイミングとなります。
 *    @details      実行可能ステート：WlanState_Detect @n
 *
 *    @param[in]    subtype   Subtype。ActionFrameType_Detect を指定して下さい。@n
 *    @param[in]    hash      すれちがいデータのハッシュ値 @n
 *    @param[in]    data      Action Frameに付加するデータ。
 *    @param[in]    size      Action Frameに付加するデータのサイズ。5Byte以上1300Byte以下を指定してください。
 *    @param[in]    interval  Action Frameの送信間隔。0ms ~ 5000ms(暫定)
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess             成功。
 *    @retval       ResultInvalidState        不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument     不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultWlanDeviceAbnormal  無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result StartPeriodicActionFrame(ActionFrameType subtype, DetectHash hash,
        const uint8_t* pData, size_t size, uint32_t interval) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        Action Frameの周期的送信を停止します。
 *    @details      実行可能ステート：WlanState_Detect @n
 *
 *    @details      @ref SetActionFrameWithBeacon 関数で送信中の周期的Action Frameの送信を停止します。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess             成功。
 *    @retval       ResultInvalidState        不正なステート時に呼び出されました。
 *    @retval       ResultWlanDeviceAbnormal  無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 *
 */
nn::Result CancelPeriodicActionFrame() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        Action Frameの受信を行います。指定したタイプのAction Frameが返ります。@n
 *    受信にはあらかじめ@ref CreateRxEntryForActionFrameにてAction Frame用の受信エントリを作成しておく必要があります。@n
 *    送信元チャンネルとRSSI値も取得出来ます。 @n
 *    ただし、BT通信が同時に行われていた場合、RSSI値は0を返す場合があるので注意して下さい。この時の0は異常値ですので無視して下さい。@n
 *    受信データは下記のCategoryからdataの最後まで入っています。
 *    +--------------------------------------------------------------------------------+ @n
 *    | Category |      OUI      | Subtype | Reserved |  Detect Header  |      data    | @n
 *    |  1 byte  |    3 bytes    |  1 byte |  1 byte  |    12 bytes     |  ~1300 bytes | @n
 *    +--------------------------------------------------------------------------------+ @n
 *    @details      実行可能ステート：WlanState_DetectIdle @n
 *                                    WlanState_Detect @n
 *
 *    @param[out]   pOutSrcMac   送信元MACアドレスが返ります。
 *    @param[out]   pOutBuf      Action Frameが返ります。ActionFrame formatのCategoryフィールド以降が入っています。（上記フォーマット図参照）
 *    @param[in]    size         受信バッファのサイズです。
 *    @param[out]   pOutSize     実際に受信したデータのサイズが返ります。
 *    @param[in]    rxId         受信エントリID
 *    @param[out]   pOutChannel  送信元チャンネル
 *    @param[out]   pOutRssi     RSSI値
 *    @param[out]   pOutTick     受信時のシステムチック値(無線チップ側で受信した時の値です)。@n
 *                               あらかじめnn::os::GetSystemTick()でシステムチック値を控えておけば、@n
 *                               この値との差分を取ることで、システムチック値を控えてから何秒後に受信したのか計算することが出来ます。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess            成功。
 *    @retval       ResultInvalidState       不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument    不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultRxEntryNotFound    指定されたrxIdが見つかりません。
 *    @retval       ResultGetFrameCancelled  受信中止が指示されました。受信の中止は@ref CancelGetActionFrame を呼ぶか、@n
 *                  @ref DeleteRxEntry または CloseMode の処理の中で自動的に呼ばれます。
 *    @retval       ResultRxEntryIsMuted     データパスが無効化されています。CloseModeを呼ぶと、wlanプロセス内部でデータパスを一時的に無効化状態としています。@n
 *                  この状態で本関数を呼ぶと受信は行えずに本リザルトが返ります。OpenModeを呼ぶとこの無効化状態は解除されます。
 *    @retval       ResultBufferTooShort     pOutBufのバッファサイズが小さく、受信データを格納出来ません。
 *    @retval       ResultWlanDeviceAbnormal 無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result GetActionFrame(MacAddress* pOutSrcMac, uint8_t pOutBuf[], size_t size,
        size_t* pOutSize, uint32_t rxId, uint16_t* pOutChannel, int16_t* pOutRssi, nn::os::Tick* pOutTick) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        @ref GetActionFrame で受信待ちの状態を解除します。
 *    @details      実行可能ステート：WlanState_DetectIdle @n
 *                                    WlanState_Detect @n
 *
 *    @param[in] rxId         受信エントリの ID
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess            成功。
 *    @retval       ResultInvalidState       不正なステート時に呼び出されました。
 *    @retval       ResultRxEntryNotFound    指定されたrxIdが見つかりません。
 *    @retval       ResultWlanDeviceAbnormal 無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result CancelGetActionFrame(uint32_t rxId) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        Action Frame用の受信エントリを作成します。
 *    Action Frameデータの取得にはあらかじめ受信エントリを作成しておく必要があります。ここでどのActionFrameTypeのデータを受信したいか登録しておけば@n
 *    受信エントリにそのActionFrameTypeのデータが格納されていきます。@n
 *    すれちがいスリープから復帰する際、無線チップ内に溜められていたActionFrameデータが吸い上げられます。このとき、@n
 *    @ref ActionFrameType_Detect のRxEnyryに保管されるので、そのようなRxEntryが存在しない場合、吸い上げられたActionFrameは破棄されます。@n
 *    すれちがいスリープに入る前にRxEntryは削除せず、残しておいてください。
 *    @details      実行可能ステート:WlanState_DetectIdle @n
 *                                   WlanState_Detect @n
 *
 *    @param[out]   pRxId        作成された受信エントリの IDが返ります。
 *    @param[in]    pAfTypes     受信エントリに登録するActionFrameTypeの配列です。
 *    @param[in]    count        ActionFrameType配列の要素数です。最大 @ref ActionFrameTypeCountMax個登録出来ます。
 *    @param[in]    capacity     受信エントリに溜めておける受信フレーム数の上限値です。最大 @ref ActionFrameRxCapacityMax 個溜められます。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess             成功。
 *    @retval       ResultInvalidState        不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument     不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultDuplicateRxEntry    指定した ActionFrameType についての受信エントリが既に存在しています。
 *    @retval       ResultRxEntryCountIsFull  作成可能な受信エントリ数が最大に達しています。
 *    @retval       ResultWlanDeviceAbnormal  無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result CreateRxEntryForActionFrame(uint32_t* pOutRxId, const uint16_t pAfTypes[], uint32_t count, uint32_t capacity) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        Action Frame用受信エントリを削除します。
 *    すれちがいスリープから復帰する際、無線チップ内に溜められていたActionFrameデータが吸い上げられます。このとき、@n
 *    @ref ActionFrameType_Detect のRxEnyryに保管されるので、そのようなRxEntryが存在しない場合、吸い上げられたActionFrameは破棄されます。@n
 *    すれちがいスリープに入る前にRxEntryは削除せず、残しておいてください。
 *    @details      実行可能ステート:WlanState_DetectIdle @n
 *                                   WlanState_Detect @n
 *
 *    @param[in]    rxId         受信エントリの ID
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess            成功。
 *    @retval       ResultInvalidState       不正なステート時に呼び出されました。
 *    @retval       ResultRxEntryNotFound    指定されたrxIdが見つかりません。
 *    @retval       ResultWlanDeviceAbnormal 無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result DeleteRxEntryForActionFrame(uint32_t rxId) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        登録済みのAction Frame用受信エントリに指定したActionFrameTypeを追加します。
 *    @details      実行可能ステート:WlanState_DetectIdle @n
 *                                   WlanState_Detect @n
 *
 *    @param[in]    rxId        受信エントリの ID
 *    @param[in]    type        追加するActionFrameType
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess          成功。
 *    @retval       ResultInvalidState     不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument  不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultRxEntryNotFound  指定されたrxIdが見つかりません。
 *    @retval       ResultDuplicateRxEntry 受信エントリに指定したActionFrameType が既に登録されています。
 *    @retval       ResultWlanDeviceAbnormal 無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result AddSubtypeToRxEntryForActionFrame(uint32_t rxId, ActionFrameType type) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        登録済みのAction Frame用受信エントリから指定したActionFrameTypeを削除し、@n
 *    もともと登録されていた受信エントリの ID を取得します。
 *    @details      実行可能ステート:WlanState_DetectIdle @n
 *                                   WlanState_Detect @n
 *
 *    @param[in/out]   pOutRxId    受信エントリの ID
 *    @param[in]       type        削除するActionFrameType
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess           成功。
 *    @retval       ResultInvalidState      不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument   不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultRxEntryNotFound   指定されたrxIdが見つからないか、指定されたActionFrameTypeが登録されていません。
 *    @retval       ResultWlanDeviceAbnormal 無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result DeleteSubtypeFromRxEntryForActionFrame(uint32_t* pOutRxId, ActionFrameType type) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        すれちがいスリープを予約します。@n
 *    本関数を呼ぶことで、次に本体がスリープに入る時にすれちがいスリープに入るようになります。@n
 *    ただし、WoWLスリープの条件が揃っている場合、そちらが優先されます。@n
 *    また、スリープから復帰すると予約は取り消されます。スリープに入る度に毎回予約して下さい。
 *    @details      実行可能ステート：WlanState_DetectIdle @n
 *                                    WlanState_Detect @n
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess             成功。
 *    @retval       ResultInvalidState        不正なステート時に呼び出されました。
 *    @retval       ResultWlanDeviceAbnormal  無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result ReserveDetectSleep() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        すれちがいスリープの予約を取り消します。@n
 *    既に取り消されている場合、何もせずResultSuccessを返します。@n
 *    @details      実行可能ステート：WlanState_DetectIdle @n
 *                                    WlanState_Detect @n
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess             成功。
 *    @retval       ResultInvalidState        不正なステート時に呼び出されました。
 *    @retval       ResultWlanDeviceAbnormal  無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result CancelDetectSleep() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        wlanプロセスに対してスリープに入るように要求します。@n
 *    スリープモードへの遷移が成功すると、ResultSuccessが返ります。 @n
 *    スリープは通常スリープとすれちがいスリープの2種類あり、それぞれ以下の条件の時、そのスリープに入ります。@n
 *    1. 通常スリープ @n
 *        a. WlanState_Readyステートで本関数を呼んだ時 @n
 *        b. WlanState_DetectIdleステートで本関数を呼ぶ＋すれちがいスリープ予約していない時 @n
 *    2. すれちがいスリープ @n
 *        WlanState_DetectIdleステートで本関数を呼ぶ＋すれちがいスリープ予約済み＋すれちがいデータセット済みの条件が満たされている時 @n
 *    上記の条件以外で本関数を呼ぶと、エラーが返ります。@n
 *    すれちがいスリープに入ると、無線デバイスは省電力状態に入り、すれちがい通信を開始します。@n
 *    この状態にあるときは、以下の要因で起床用のGPIOが叩かれます。@n
 *        * 8個のすれちがいフレームを受信したとき
 *    これにより本体SoCが起床します。ただし、wlanプロセスはまだスリープ状態にあるので、wlanプロセスを動作させたい場合は、@n
 *    起床用API(@ref RequestWakeup )を呼ぶ必要があります。 @n
 *
 *    @details      実行可能ステート：WlanState_Ready (通常スリープに入ります) @n
 *                                    WlanState_DetectIdle @n
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         スリープモードへの遷移に成功。wlanプロセスはWlanState_Sleepに入ります。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。（developビルドの場合、アサートします）
 *    @retval       ResultNotSetDetectData  すれちがい予約はされていますが、すれちがいデータがセットされていません。
 *    @retval       ResultWlanDeviceAbnormal  無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result RequestSleep() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        wlanプロセスに対してスリープから復帰するように要求します。@n
 *    スリープモードからの復帰が成功すると、ResultSuccessが返ります。 @n
 *    本APIが呼ばれたら、wlanが取る行動は以下です。@n
 *    (すれちがいスリープからの復帰の場合)
 *        1. 無線ドライバーに復帰処理を要求し、無線デバイスと本体SoCが通信可能な状態にします。@n
 *        2. 無線FWに溜められていたすれちがいフレームを吸い上げ、@ref CreateRxEntryForActionFrame で作成されたRxEntryに格納します。
 *        3. wlanのステートをWlanState_Detectに遷移させます。@n
 *        4. ResultSuccessを呼び出し元に返します。@n
 *    (通常スリープからの復帰の場合)
 *        1. 無線ドライバーに復帰処理を要求し、無線デバイスと本体SoCが通信可能な状態にします。@n
 *        2. wlanのステートをWlanState_Readyに遷移させます。@n
 *        4. ResultSuccessを呼び出し元に返します。@n
 *
 *    @details      実行可能ステート：WlanState_Sleep @n
 *                                    WlanState_Ready (ResultAlreadyAwakeが返ります) @n
 *                                    WlanState_DetectIdle (ResultAlreadyAwakeが返ります) @n
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         スリープモードからの復帰に成功。
 *    @retval       ResultAlreadyAwake    既に起床状態です。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。（developビルドの場合、アサートします）
 *    @retval       ResultWlanDeviceAbnormal  無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result RequestWakeUp() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        すれちがいスリープ中に送信するActionFrameデータを登録します。@n
 *    このActionFrameの宛先は指定できません。Broadcastアドレス宛てとなっています。@n
 *    @ref pData へのデータの入れ方は @ref PutActionFrameOneShot と同様です。@n
 *    起床時にセットされたデータはクリアされ、送信も止まります。再度すれちがいスリープに入る時はもう一度本関数を呼んでデータをセットして下さい。
 *    @details      実行可能ステート：全ステート @n
 *
 *    @param[in]    subtype   Subtype。ActionFrameType_Detect を指定して下さい。@n
 *    @param[in]    hash      すれちがいデータのハッシュ値 @n
 *    @param[in]    data      Action Frameに付加するデータ。
 *    @param[in]    size      Action Frameに付加するデータのサイズ。5Byte以上1300Byte以下を指定してください。
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess             成功。
 *    @retval       ResultInvalidState        不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument     不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultWlanDeviceAbnormal  無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result SetActionFrameForSleep(ActionFrameType subtype, DetectHash hash,
        const uint8_t* pData, size_t size) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        すれちがいスリープ中に受信したActionFrameをフィルタリングするためのハッシュリストを登録します。@n
 *    登録したハッシュ値を持つActionFrameが来ていた場合、そのActionFrameは捨てられます。
 *    @details      実行可能ステート：全ステート @n
 *
 *    @param[in]       pHashList   ハッシュリストへのポインタ
 *    @param[in]       numOfHash   ハッシュの数
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess           成功。
 *    @retval       ResultInvalidState      不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument   不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultWlanDeviceAbnormal  無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result SetHashList(const uint64_t* pHashList, uint32_t numOfHash) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        すれちがいActionFrameの送信、受信、アイドル時間の設定を行います。（デバッグ用途）@n
 *    本関数で設定した値は@ref StartPeriodicActionFrame, @ref SetActionFrameForSleepの挙動に反映されます。
 *    targetで適用先を指定すれば、それぞれの関数に対して別々のパラメータを設定できます。@n
 *    また、スリープから復帰した後もこの値は反映され続けます。@n
 *    なお、@ref StartPeriodicActionFrameでは、txIntervalの値は @ref StartPeriodicActionFrameで指定したinterval値が使用されます。
 *    @details      実行可能ステート：全ステート @n
 *
 *    @param[in]       pattern   すれちがいActionFrameの送信、受信、アイドル時間の設定用パラメータ
 *    @param[in]       target   パラメータの設定先
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess           成功。
 *    @retval       ResultInvalidState      不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument   不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultWlanDeviceAbnormal  無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result SetPeriodicActionFrameCycle(DetectPeriodicAfCycle pattern, DetectPeriodicAfCycleTarget target) NN_NOEXCEPT;


/**
 *    @brief        Action Frameを単発で送信します。（デバッグ用途）@n
 *    BSSIDやNDHPを指定できるデバッグ用APIです。@n
 *    @details      実行可能ステート：WlanState_Detect @n
 *
 *    @param[in]    dstMac    宛先MACアドレス。ZeroMacAddressは無効です。@n
 *    @param[in]    bssid     BSSID。ZeroMacAddressを入れた場合、本体MACアドレスを使用します。@n
 *    @param[in]    subtype   Subtype。ActionFrameType_Detect を指定して下さい。@n
 *    @param[in]    dhp       NDHP @n
 *    @param[in]    pData     Action Frameに付加するデータ。上記説明参照。@n
 *    @param[in]    size      Action Frameに付加するデータのサイズ。1300Byte以下を指定してください。@n
 *    @param[in]    dwellTime 送信後、送信先チャンネルに留まっている時間。留まっている間は電波受信状態となります。@n
 *    例えば、相手からの返事を待ちたい場合はこの値を調整して下さい。単位はmsecです。最大100msec待てます。@n
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess         成功。
 *    @retval       ResultInvalidState    不正なステート時に呼び出されました。
 *    @retval       ResultInvalidArgument 不正な引数値です。（developビルドの場合、アサートします）
 *    @retval       ResultCommandFailure  送信に失敗しました。送信完了バッファの解放が追い付かず詰まっている可能性があります。少し間を空けてから再度試して下さい。
 *    @retval       ResultWlanDeviceAbnormal 無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result PutActionFrameOneShotEx(const MacAddress& dstMac, const MacAddress& bssid, ActionFrameType subtype,
        DetectHeader dhp, const uint8_t* pData, size_t size, uint32_t dwellTime) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        MACアドレスをセットします。（デバッグ用途）@n
 *    ZeroMacAddressやBroadcastMacAddressを指定した場合、オリジナルのMACアドレスがセットされます。
 *    @details      実行可能ステート：WlanState_DetectIdle @n
 *
 *    @param[in]       mac   MACアドレス
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess           成功。
 *    @retval       ResultInvalidState      不正なステート時に呼び出されました。
 *    @retval       ResultWlanDeviceAbnormal  無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result SetMacAddress(const MacAddress& mac) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        すれちがいスリープ中に受信したすれちがいパケットの累計を取得します。（デバッグ用途）@n
 *    @details      実行可能ステート：WlanState_DetectIdle @n
 *                                    WlanState_Detect @n
 *
 *    @param[out]       pCount   受信累計
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess           成功。
 *    @retval       ResultInvalidState      不正なステート時に呼び出されました。
 *    @retval       ResultWlanDeviceAbnormal  無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result GetTotalRecvCountInSleep(uint64_t* pCount) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 *    @brief        すれちがいスリープ中に受信したすれちがいパケットの累計を0に初期化します。（デバッグ用途）@n
 *    @details      実行可能ステート：WlanState_DetectIdle @n
 *                                    WlanState_Detect @n
 *
 *    @return       関数の実行結果を返します。以下に挙げる Result を返します。
 *    @retval       ResultSuccess           成功。
 *    @retval       ResultInvalidState      不正なステート時に呼び出されました。
 *    @retval       ResultWlanDeviceAbnormal  無線ドライバーが異常な状態のためwlan使用不可能です。本体再起動して下さい。
 */
nn::Result ClearTotalRecvCountInSleep() NN_NOEXCEPT;
}
}
}

