﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief Cruiser起動引数が共通で利用する API (非公開)
 */

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/account/account_Api.h>
#include <nn/applet/applet.h>
#include <nn/applet/applet_Types.h>
#include <nn/la/la_Api.h>
#include <nn/la/la_CommonArgumentsWriter.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/web/common/web_CommonTypes.h>

namespace nn { namespace web { namespace common {

class CommonArgData
{
public:
    /**
     * @brief 情報を格納するために使用するデータのサイズ（バイト）です。
     */
    static const size_t DataSize_Max = nn::applet::StartupParamSizeMax;

    /**
     * @brief ArgKind
     */
    enum class ArgKind
    {
        None,                                            //!<
        RequestURL,                                      //!< リクエストURL
        RequestRelativeURL,                              //!< リクエストURL(相対指定: Shop向け)
        CallbackURL,                                     //!< コールバックURL
        CallbackableURL,                                 //!< コールバック可能URL
        ApplicationId,                                   //!< アプリケーションID
        DocumentPath,                                    //!< ドキュメントパス
        DocumentKind,                                    //!< ドキュメントの種類
        SystemDataId,                                    //!< システムデータID
        ShareStartPage,                                  //!< Share 開始時のスタートページ
        Whitelist,                                       //!< ホワイトリスト
        AllowFirstDomain,                                //!< RequestURLとFQDNが同じURLをアクセス可能にするか
        DenyOnlyBlacklistedURL,                          //!< ブラックリストに載っているURLのみをアクセス拒否するか
        IsAnonymousMode,                                 //!< アノニマスモードを有効にするか
        Uid,                                             //!< アカウントシステムに登録されたユーザーの識別子
        AlbumEntry,                                      //!< アルバム内ファイル情報(1枚目)
        ScreenShotEnabled,                               //!< スクリーンショット撮影を許可するかどうか
        IsCalledFromPrivateApi,                          //!< PrivateなAPIから呼び出されたかどうか(web(News), offline時のプレイレポート向け)
        Deleted0,                                        //!< 廃止された
        PlayReportJsExtensionEnabled,                    //!< コンテンツ中でJavaScriptからプレイレポートを利用できるかどうか
        IsContentsDropDisabled,                          //!< コンテンツ欠落を禁止するか
        SuppressRestartDialog,                           //!< 再起動ダイアログを抑制するか
        ClearCookieAndLocalStorage,                      //!< CookieとLocalStorageを削除するか
        BootDisplayKind,                                 //!< 起動画面種類
        BackgroundKind,                                  //!< 背景種類
        FooterEnabled,                                   //!< フッターを利用するかどうか
        UseStickPointer,                                 //!< スティックポインターを利用するかどうか
        LeftStickMode,                                   //!< 起動時の左スティックモード
        KeyRepeatDelayFrame,                             //!< キーリピート開始フレーム数
        KeyRepeatPulseFrame,                             //!< キーリピート間隔フレーム数
        UsePageInfo,                                     //!< ページ情報を利用するかどうか
        UseDommainForPageInfo,                           //!< ページ情報 URL をドメイン表記にするかどうか
        UseRightStickSpatialNavi,                        //!< 右スティックを空間ナビに使用するか
        BootAsMediaPlayer,                               //!< メディアプレイヤー専用起動
        EcJumpEnabled,                                   //!< ECアプレットへのジャンプを許可するかどうか
        MediaPlayerUserGestureRestrictionEnabled,        //!< メディアプレイヤーでユーザー操作でのみ再生可能とする制限をするか
        LobbyParameter,                                  //!< Lobbyで開くURLに付加するパラメータ
        ApplicationSuspendingEnabled,                    //!< 呼び出し元アプリが自発的にSuspendする事を許可しているか
        ApplicationAlbumEntry,                           //!< アルバム内ファイル情報(nn::albumからの利用向け)
        GeneralJsExtensionEnabled,                       //!< コンテンツ中で一般向けのJavaScriptが利用できるかどうか
        AdditionalCommentText,                           //!< シェアページに受け渡す文字列
        TouchEnabledOnContents,                          //!< コンテンツのタッチ操作を有効にするかどうか
        UserAgentAdditionalString,                       //!< ユーザーエージェントへ追加する文字列
        AdditionalMediaData,                             //!< シェアページに受け渡すメディアファイルの追加情報(1枚目)
        MediaPlayerAutoCloseEnabled,                     //!< 動画が最後まで再生された際に、動画プレイヤーを自動的に終了するかどうか
        PageCacheEnabled,                                //!< ページキャッシュを有効にするかどうか
        WebAudioEnabled,                                 //!< Web Audio を利用できるようにするかどうか
        EcClientCertEnabled,                             //!< Ec 用クライアント証明書を利用するかどうか
        ClearCookieAndLocalStorageExceptNintendoAccount, //!< NA登録サイト以外のCookieとLocalStorageを削除するか
        BootAsYouTubePlayer,                             //!< YouTubePlayer として起動されたか
        FooterFixedKind,                                 //!< フッター表示の制御方法種類
        PageFadeEnabled,                                 //!< ページ間フェードを有効にするかどうか
        MediaCreatorApplicationRatingAge,                //!< シェアページに受け渡すメディアファイルのレーティング情報
        BootLoadingIconEnabled,                          //!< 起動時のローディングアイコンを表示するかどうか
        PageScrollIndicatorEnabled,                      //!< ページにスクロールインジケータを表示するかどうか
        MediaPlayerSpeedControlEnabled,                  //!< メディアプレイヤーで再生速度を変更できるようにするかどうか
        AlbumEntry2nd,                                   //!< アルバム内ファイル情報(2枚目)
        AlbumEntry3rd,                                   //!< アルバム内ファイル情報(3枚目)
        AlbumEntry4th,                                   //!< アルバム内ファイル情報(4枚目)
        AdditionalMediaData2nd,                          //!< シェアページに受け渡すメディアファイルの追加情報(2枚目)
        AdditionalMediaData3rd,                          //!< シェアページに受け渡すメディアファイルの追加情報(3枚目)
        AdditionalMediaData4th,                          //!< シェアページに受け渡すメディアファイルの追加情報(4枚目)
        BootFooterButtonVisibleSettingList,              //!< 起動時にフッターボタンを表示するかどうか
        Max                                              //!< 列挙子の総数です。
    };

    /**
     * @brief セット時のエラータイプ
     */
    enum class ErrorType
    {
        None,         //!< エラー無し
        Invalid,      //!< 無効なデータをセットしようとしました。
        Conflict,     //!< 同じ種類の異なる格納サイズを持つデータをセットしようとしました。
        OverCapacity, //!< 格納できる最大値を超えています。
        Max           //!< 列挙子の総数です。
    };

    /**
     * @brief データ全体のヘッダ構造体です。
     *
     * @details
     *   8byte使用
     */
    struct Header
    {
        uint16_t count;     //!< 総格納数
        uint16_t _padding;  //!< Padding
        uint32_t kind;      //!< 種類
    };

    /**
     * @brief データのヘッダ構造体です。
     *
     * @details
     *   8byte使用
     */
    struct DataHeader
    {
        uint16_t kind;      //!< 引数の種類
        uint16_t fixedSize; //!< 格納サイズ
        uint32_t padding;   //!< パディング領域

        inline static DataHeader Create()
        {
            DataHeader header;
            header.kind = uint16_t(ArgKind::None);
            header.fixedSize = 0;
            header.padding = 0;
            return header;
        }
    };

    /**
     * @brief データの構造体です。
     */
    struct Data
    {
        DataHeader header;          //!< ヘッダ
        const uint8_t* pBuffer;     //!< バッファ
    };

    //------------------------------------------------------------------------
    /**
     * @brief データを初期化します。
     *
     * @param[out]  pDst        初期化対象のデータを指定してください。
     * @param[in]   shimKind    起動対象となるShimの種類を指定ください。
     */
    inline static void InitializeData(uint8_t* pDst, const ShimKind shimKind) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief 対象バッファにデータをセットします。
     *        同じArgKindを指定した場合対象領域が上書きされます。
     *
     * @param[out]  pDst        セット対象となるデータを指定してください。
     * @param[in]   argKind     セットするパラメータの種類を指定してください。
     * @param[in]   fixedSize   セットするパラメータを格納する際の固定サイズを指定してください。
     * @param[in]   pValue      セットするパラメータの値を指定してください。
     *
     * @details セットに失敗した場合アサートします。
     */
    inline static void SetData(uint8_t* pDst, ArgKind argKind, size_t fixedSize, const void* pValue) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief 対象バッファに文字列をセットします。
     *        同じArgKindを指定した場合対象領域が上書きされます。
     *
     * @param[out]  pDst        セット対象となるデータを指定してください。
     * @param[in]   argKind     セットするパラメータの種類を指定してください。
     * @param[in]   fixedSize   セットするパラメータを格納する際の固定サイズを指定してください。
     * @param[in]   pValue      セットするパラメータの値を指定してください。
     *
     * @details セットに失敗した場合アサートします。
     */
    inline static void SetData(uint8_t* pDst, ArgKind argKind, size_t fixedSize, const char* pString) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief 対象バッファにデータをセットします。
     *        同じArgKindを指定した場合対象領域が上書きされます。
     *
     * @param[out]  pDst        セット対象となるデータを指定してください。
     * @param[in]   argKind     セットするパラメータの種類を指定してください。
     * @param[in]   value       セットするパラメータの値を指定してください。
     *
     * @details セットに失敗した場合アサートします。
     */
    template <typename T>
    inline static void SetData(uint8_t* pDst, ArgKind argKind, const T& value) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief 対象バッファからヘッダを取得します。
     *
     * @param[in]   pSrc    取得対象となるバッファを指定してください。
     *
     * @return 取得に成功した場合ヘッダを返します。
     */
    inline static const Header& GetHeader(const uint8_t* pSrc) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief 対象バッファから指定したデータの先頭アドレスを取得します。
     *
     * @param[in]   pSrc    取得対象となるバッファを指定してください。
     * @param[in]   argKind 取得するデータの種類を指定してください。
     *
     * @return 取得に成功した場合指定したデータの先頭アドレスを返します。
     */
    inline static const uint8_t* GetDataPos(const uint8_t* pSrc, ArgKind argKind) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief getDataPosで取得したデータの値を取得します。
     *
     * @param[in]   pSrc    取得対象となるデータの先頭アドレスを指定してください。
     *
     * @return 取得に成功した場合データの値へのアドレスを返します。
     */
    inline static const uint8_t* GetDataBuffer(const uint8_t* pSrc) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief 対象バッファから指定したデータを取得します。
     *
     * @param[in]   pSrc    取得対象となるバッファを指定してください。
     * @param[in]   argKind 取得するデータの種類を指定してください。
     *
     * @return 取得に成功した場合指定したデータを返します。
     */
    inline static const Data GetData(const uint8_t* pSrc, ArgKind argKind) NN_NOEXCEPT;

private:
    //------------------------------------------------------------------------
    /**
     * @brief 対象バッファにデータをセットします。
     *        同じArgKindを指定した場合対象領域が上書きされます。
     *
     * @param[out]  pDst        セット対象となるデータを指定してください。
     * @param[in]   argKind     セットするパラメータの種類を指定してください。
     * @param[in]   size        セットするパラメータのサイズを指定してください。
     * @param[in]   fixedSize   セットするパラメータを格納する際の固定サイズを指定してください。
     * @param[in]   pValue      セットするパラメータの値を指定してください。
     *
     * @return セットに成功した場合ErrorType::Noneが返ります。
     */
    inline static ErrorType SetData(uint8_t* pDst, ArgKind argKind,
        size_t size, size_t fixedSize, const void* pValue) NN_NOEXCEPT;

    //------------------------------------------------------------------------
    /**
     * @brief getDataPosで取得したデータのヘッダを取得します。
     *
     * @param[in]   pSrc    取得対象となるデータの先頭アドレスを指定してください。
     *
     * @return 取得に成功した場合データのヘッダを返します。
     */
    inline static const DataHeader GetDataHeader(const uint8_t* pSrc) NN_NOEXCEPT;

};

//------------------------------------------------------------------------
void CommonArgData::InitializeData(uint8_t* pDst, const ShimKind shimKind) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT(shimKind > ShimKind::None && shimKind < ShimKind::Max);

    std::memset(pDst, 0, DataSize_Max);
    auto header = reinterpret_cast<Header*>(pDst);
    header->kind = uint32_t(shimKind);
#else
    NN_UNUSED(pDst);
    NN_UNUSED(shimKind);
#endif
}

//------------------------------------------------------------------------
CommonArgData::ErrorType CommonArgData::SetData(uint8_t* pDst, ArgKind argKind, size_t size,
size_t fixedSize, const void* pValue) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT(size > 0);
    NN_SDK_ASSERT(size <= fixedSize);
    NN_SDK_ASSERT_NOT_NULL(pValue);

    // 格納数取得
    auto header = reinterpret_cast<Header*>(pDst);

    size_t offset = sizeof(Header);
    // 既に同じkindのデータがあれば上書きする
    // @note データが固定長である事を前提としているので可変長になったらデータの並び替え等の処理が必要です。
    for (int i = 0; i < header->count; ++i) {
        uint8_t* p = pDst + offset;
        auto dataHeader = *(reinterpret_cast<DataHeader*>(p));
        if (dataHeader.kind == uint16_t(argKind)) {
            NN_SDK_ASSERT(dataHeader.fixedSize == fixedSize);
            if (dataHeader.fixedSize != fixedSize) {
                return ErrorType::Conflict;
            }
            uint8_t* pBuf = p + sizeof(DataHeader);
            std::memset(pBuf, 0, fixedSize);
            std::memcpy(pBuf, pValue, size);
            return ErrorType::None;
        }
        offset += sizeof(DataHeader) + dataHeader.fixedSize;
    }

    // 書き込めるか確認
    if ((offset + fixedSize) >= DataSize_Max) {
        return ErrorType::OverCapacity;
    }

    // データ追加
    {
        // 書き込む先頭アドレス取得
        uint8_t* p = pDst + offset;

        auto dataHeader = DataHeader::Create();
        dataHeader.kind = static_cast<uint16_t>(argKind);
        dataHeader.fixedSize = static_cast<uint16_t>(fixedSize);

        // データ書き込み
        std::memcpy(p, &dataHeader, sizeof(DataHeader));
        p += sizeof(DataHeader);
        std::memcpy(p, pValue, size);
    }
    // 総格納数をインクリメント
    header->count++;

    return ErrorType::None;
#else
    NN_UNUSED(pDst);
    NN_UNUSED(argKind);
    NN_UNUSED(size);
    NN_UNUSED(fixedSize);
    NN_UNUSED(pValue);
    return ErrorType::Invalid;
#endif
}

//------------------------------------------------------------------------
void CommonArgData::SetData(uint8_t* pDst, ArgKind argKind, size_t fixedSize, const void* pValue) NN_NOEXCEPT
{
    auto errorType = SetData(pDst, argKind, fixedSize, fixedSize, pValue);
    switch (errorType) {
    case ErrorType::None:break;
    case ErrorType::Invalid:
        NN_SDK_ASSERT(0, "Invalid argKind or invalid pValue");
        break;
    case ErrorType::Conflict:
        NN_SDK_ASSERT(0, "Arg conflict. Different API set is used");
        break;
    case ErrorType::OverCapacity:
        NN_SDK_ASSERT(0, "OverCapacity");
        break;
    default:
        break;
    }
}

//------------------------------------------------------------------------
void CommonArgData::SetData(uint8_t* pDst, ArgKind argKind, size_t fixedSize, const char* pString) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pString);
    size_t size = std::strlen(pString) + 1;
    if (size <= fixedSize) {
        NN_SDK_ASSERT(pString[size - 1] == '\0', "Non null terminated string");

        auto errorType = SetData(pDst, argKind, size, fixedSize, pString);

        switch (errorType) {
        case ErrorType::None:break;
        case ErrorType::Invalid:
            NN_SDK_ASSERT(0, "Invalid argKind or invalid pValue");
            break;
        case ErrorType::Conflict:
            NN_SDK_ASSERT(0, "Arg conflict. Different API set is used");
            break;
        case ErrorType::OverCapacity:
            NN_SDK_ASSERT(0, "OverCapacity");
            break;
        default:
            break;
        }
    }
    else {
        NN_SDK_ASSERT(0, "Size over");
    }
}

//------------------------------------------------------------------------------
template <typename T>
void CommonArgData::SetData(uint8_t* pDst, ArgKind argKind, const T& value) NN_NOEXCEPT
{
    SetData(pDst, argKind, sizeof(T), &value);
}

//------------------------------------------------------------------------
const CommonArgData::Header& CommonArgData::GetHeader(const uint8_t* pSrc) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NN_SDK_ASSERT_NOT_NULL(pSrc);

    return *(reinterpret_cast<const Header*>(pSrc));
#else
    NN_UNUSED(pSrc);
    return Header();
#endif
}

//------------------------------------------------------------------------
const uint8_t* CommonArgData::GetDataPos(const uint8_t* pSrc, ArgKind argKind) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NN_SDK_ASSERT_NOT_NULL(pSrc);
    NN_SDK_ASSERT(argKind > ArgKind::None && argKind < ArgKind::Max);

    auto header = reinterpret_cast<const Header*>(pSrc);

    size_t offset = sizeof(Header);

    for (int i = 0; i < header->count; ++i) {
        const uint8_t* p = pSrc + offset;
        auto dataHeader = *(reinterpret_cast<const DataHeader*>(p));
        if (dataHeader.kind == uint16_t(argKind)) {
            return p;
        }
        offset += sizeof(DataHeader) + dataHeader.fixedSize;
    }
    return nullptr;
#else
    NN_UNUSED(pSrc);
    NN_UNUSED(argKind);
    return nullptr;
#endif
}

//------------------------------------------------------------------------
const CommonArgData::DataHeader CommonArgData::GetDataHeader(const uint8_t* pSrc) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NN_SDK_ASSERT_NOT_NULL(pSrc);

    return *(reinterpret_cast<const DataHeader*>(pSrc));
#else
    NN_UNUSED(pSrc);
    return DataHeader();
#endif
}

//------------------------------------------------------------------------
const uint8_t* CommonArgData::GetDataBuffer(const uint8_t* pSrc) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NN_SDK_ASSERT_NOT_NULL(pSrc);

    return pSrc + sizeof(DataHeader);
#else
    NN_UNUSED(pSrc);
    return nullptr;
#endif
}

//------------------------------------------------------------------------
const CommonArgData::Data CommonArgData::GetData(const uint8_t* pSrc, ArgKind argKind) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NN_SDK_ASSERT_NOT_NULL(pSrc);
    NN_SDK_ASSERT(argKind > ArgKind::None && argKind < ArgKind::Max);

    const uint8_t* dataPos = GetDataPos(pSrc, argKind);
    if (dataPos != nullptr) {
        Data data;

        data.header = GetDataHeader(dataPos);
        data.pBuffer = GetDataBuffer(dataPos);
        return data;
    }

    return Data();
#else
    NN_UNUSED(pSrc);
    NN_UNUSED(argKind);
    return Data();
#endif
}

}}} // namespace nn::web::common
