﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Functions to manipulate layer configuration.
 * @details The functions described in this file are for internal developers only
 *          and not to be released to third parties.
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/vi/vi_Types.h>
#include <nn/vi/vi_LayerStack.h>

namespace nn { namespace vi {

    /**
     * @brief Adds a layer to a particular layer stack.
     *
     * @param[in] pLayer Layer handle.
     * @param[in] id     LayerStack id.
     *
     * @return                       Whether the layer was added to the requested layer stack.
     * @retval ResultSuccess         The layer was added to the layer stack.
     * @retval ResultInvalidRange    Attempting to add the layer to an invalid layer stack.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre
     *       - VI library is initialized.
     *       - @a pLayer != nullptr and @a pLayer is a valid handle.
     * @post @a pLayer has been added to @a id's layer stack.
     */
    nn::Result AddToLayerStack(Layer* pLayer, LayerStack id) NN_NOEXCEPT;

    /**
     * @brief Removes a layer from a particular layer stack.
     *
     * @param[in] pLayer Layer handle.
     * @param[in] id     LayerStack id.
     *
     * @return                       Whether the layer was removed from the requested layer stack.
     * @retval ResultSuccess         The layer was removed from the layer stack.
     * @retval ResultInvalidRange    Attempting to remove the layer from an invalid layer stack.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre
     *       - VI library is initialized.
     *       - @a pLayer != nullptr and @a pLayer is a valid handle.
     * @post @a pLayer has been removed from @a id's layer stack.
     */
    nn::Result RemoveFromLayerStack(Layer* pLayer, LayerStack id) NN_NOEXCEPT;

    /**
     * @brief Sets whether content from the layer is visible on the display.
     *
     * @param[in] pLayer    Layer handle.
     * @param[in] isVisible If true, content from this layer will be seen on the display.
     *                      Otherwise, no content will be seen.
     *
     * @return                       Whether @a pLayer's visibility matches the requested value.
     * @retval ResultSuccess         Layer visibility is now the requested value.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre  VI library is initialized. @n
     *       @a pLayer != nullptr and @a pLayer is a valid handle.
     * @post The layer's visibility matches that of @a isVisible.
     */
    Result SetLayerVisibility(Layer* pLayer, bool isVisible) NN_NOEXCEPT;

}}
