﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief CMU settings for the VI library.
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/vi/vi_CmuMode.h>

namespace nn { namespace vi {

    class Display;

    /**
     * @brief Retrieves the display's current CMU mode.
     *
     * @param[out] pOutMode The current CMU mode.
     * @param[in]  pDisplay Display handle.
     *
     *
     * @return                       Whether the CMU mode used on @a pDisplay was retrieved successfully.
     * @retval ResultSuccess         Successfully retrieved the display's CMU mode.
     * @retval ResultNotSupported    The display does not support any CMU modes.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre
     *       - VI library is initialized.
     *       - @a pOutMode != nullptr
     *       - @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The CMU mode used on @a pDisplay is written to @a pOutMode.
     */
    Result GetDisplayCmuMode(CmuMode* pOutMode, const Display* pDisplay) NN_NOEXCEPT;

    /**
     * @brief Sets the CMU mode used by the display.
     *
     * @param[in] pDisplay Display handle.
     * @param[in] mode     The CMU mode to use on the display.
     *
     * @return                       Whether the CMU mode on @a pDisplay matches that of @a mode.
     * @retval ResultSuccess         The display's CMU mode matches @a mode.
     * @retval ResultNotSupported    The requested CMU mode is not supported on the display.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre
     *       - VI library is initialized.
     *       - @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The CMU mode used by @a pDisplay matches that of @a mode.
     *
     * @details This setting will affect all layers present on the display, regardless of originating
     *          process.
     */
    Result SetDisplayCmuMode(Display* pDisplay, CmuMode mode) NN_NOEXCEPT;

    /**
     * @brief Retrieves the display's current contrast ratio.
     *
     * @param[out] pOutRatio The contrast ratio.
     * @param[in]  pDisplay  Display handle.
     *
     * @return                       Whether the contrast ratio used on @a pDisplay was retrieved successfully.
     * @retval ResultSuccess         Successfully retrieved the display's contrast setting.
     * @retval ResultNotSupported    The high contrast CMU mode is not enabled.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre
     *       - VI library is initialized.
     *       - @a pOutRatio != nullptr
     *       - @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The contrast ratio used on @a pDisplay is written to @a pOutRatio.
     *
     * @details The current CMU mode must be set to CmuMode_HighContrast.
     */
    Result GetDisplayContrastRatio(float* pOutRatio, const Display* pDisplay) NN_NOEXCEPT;

    /**
     * @brief Sets the contrast ratio used on the display.
     *
     * @param[in] pDisplay Display handle.
     * @param[in] ratio    The contrast ratio.
     *
     * @return                       Whether the contrast ratio on @a pDisplay matches that of @a ratio.
     * @retval ResultSuccess         The display's contrast ratio matches @a ratio.
     * @retval ResultInvalidRange    The valid range is 0.f <= @a ratio <= 1.f.
     * @retval ResultNotSupported    The high contrast CMU mode is not enabled.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre
     *       - VI library is initialized.
     *       - @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The contrast ratio used by @a pDisplay matches that of @a ratio.
     *
     * @details The current CMU mode must be set to CmuMode_HighContrast.
     */
    Result SetDisplayContrastRatio(Display* pDisplay, float ratio) NN_NOEXCEPT;

    /**
     * @brief Retrieves the display's current gamma setting.
     *
     * @param[out] pOutGamma Display's gamma setting.
     * @param[in]  pDisplay  Display handle.
     *
     * @return                       Whether the gamma setting used on @a pDisplay was retrieved successfully.
     * @retval ResultSuccess         Successfully retrieved the display's gamma setting.
     * @retval ResultNotSupported    The display does not support this setting.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre
     *       - VI library is initialized.
     *       - @a pOutGamma != nullptr
     *       - @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The current gamma setting used by @a pDisplay is written to @a pOutGamma.  This setting is only
     *       applied when the CMU is set to CmuMode_Default.
     */
    Result GetDisplayGamma(float* pOutGamma, const Display* pDisplay) NN_NOEXCEPT;

    /**
     * @brief Sets the gamma used by the display.
     *
     * @param[in] pDisplay Display handle.
     * @param[in] gamma    New gamma setting to use.
     *
     * @return                       Whether the gamma setting of @a pDisplay matches that of @a gamma.
     * @retval ResultSuccess         The display's gamma setting matches @a gamma.
     * @retval ResultInvalidRange    The valid range is 1.f <= @a gamma <= 3.f.
     * @retval ResultNotSupported    The display does not support this setting.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre
     *       - VI library is initialized.
     *       - @a pDisplay != nullptr and @a pDisplay is a valid handle.
     * @post The current gamma setting of @a pDisplay matches that of @a gamma.
     *
     * @details This setting is only applied when the CMU is set to CmuMode_Default.
     */
    Result SetDisplayGamma(Display* pDisplay, float gamma) NN_NOEXCEPT;

}}
