﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief Contains functionality specific to NXCP profiler.
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEvent.h>

namespace nn { namespace vi { namespace profiler {

    /**
     * @brief Initializes VI library for NXCP profiler.
     *
     * @return                            Whether VI library was initialized successfully.
     * @retval ResultSuccess              VI library was initialized successfully.
     * @retval nn::sf::ResultNotPermitted This process does not have appropriate permissions.
     * @retval ResultOperationFailed      An unknown error occurred.
     *
     * @post VI library is initialized.
     *
     * @details This is safe to call multiple times.  If the library is already initialized,
     *          no operations occur.
     *
     *          There must be a matching call to Finalize for each call to this function or
     *          the VI library will not be finalized.
     */
    nn::Result Initialize() NN_NOEXCEPT;

    /**
     * @brief Finalizes VI library for NXCP profiler.
     *
     * @pre  No preconditions.
     * @post VI library has been shutdown.
     *
     * @details Note that nn::vi::Layer and nn::vi::Display handles will be invalidated
     *          and should not be used after calling this function.
     *
     *          This is safe to call multiple times.  If the library is not initialized, no
     *          operations occur.
     */
    void Finalize() NN_NOEXCEPT;

    /**
     * @brief Initializes an event that will signal on VSYNC.
     *
     * @param[out] pOutEvent Event to initialize.
     *
     * @return                       Whether the event was initialized successfully.
     * @retval ResultSuccess         The event was initialized successfully.
     * @retval ResultDenied          Not allowed to open any more events of this type or
     *                               system is in production mode.
     * @retval ResultOperationFailed An unknown error occurred.
     *
     * @pre
     *      - VI library is initialized through nn::vi::profiler::Initialize().
     *      - @a pOutEvent != nullptr
     * @post @a pOutEvent will signal on @a pDisplay VSYNC events.
     *
     * @details @a pOutEvent will autoclear.  Only one thread will wake up when the event is signalled.
     *
     *          This function should only be used for internal debugging.
     */
    nn::Result GetVsyncEvent(nn::os::SystemEventType* pOutEvent) NN_NOEXCEPT;

}}}
