﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/vi/fbshare/vi_SharedLayerWindow.h>
#include <nn/gfx.h>

namespace nn{ namespace vi{ namespace fbshare{

    class SharedLayerWindowGfx
    {
    public:
        static const int InternalTextureCountMax = 16;

    public:
        // フレームバッファ共有が有効になっているかを取得します。
        // この API は nn::vi::Initialize() よりも前に呼びだすことができます。
        static bool IsSupported() NN_NOEXCEPT;

        SharedLayerWindowGfx() NN_NOEXCEPT;
        ~SharedLayerWindowGfx() NN_NOEXCEPT;

        // @brief 初期化済かを返します。
        bool IsInitialized() const NN_NOEXCEPT;

        // @brief 初期化します。
        // @details
        // bufferCount には想定するバッファ数を指定します。
        // 実際のバッファ数はシステムにより割り当てられたテクスチャの枚数になります。
        // 複数のコマンドバッファをディスプレイと同期して使用するフレームワークのために用意されています。
        //
        // textureFormat には nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm または nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb を指定します。
        // シェーダ内で明示的にガンマカーブを掛ける場合には Unorm を指定します。
        // いずれの場合も、ディスプレイではテクセル値は sRGB で表現された値として解釈されます。
        nn::Result Initialize(nn::gfx::Device* pDevice, int bufferCount, nn::gfx::ImageFormat textureFormat) NN_NOEXCEPT;

        // @brief 破棄します。
        void Finalize() NN_NOEXCEPT;

        // @brief テクスチャの幅 (px) を取得します。
        int GetTextureWidth(nn::gfx::Texture* pTexture) const NN_NOEXCEPT;

        // @brief テクスチャの高さ (px) を取得します。
        int GetTextureHeight(nn::gfx::Texture* pTexture) const NN_NOEXCEPT;

    // @name 描画用テクスチャ
    // @{

        // @brief テクスチャが予約状態かを取得します。
        // @pre IsInitialized()
        bool IsTexturePreAcquired() const NN_NOEXCEPT;

        // @brief テクスチャが獲得されているかを取得します。
        // @pre IsInitialized()
        bool IsTextureAcquired() const NN_NOEXCEPT;

        // @brief タイムアウト付きでテクスチャの獲得を予約します。
        // @pre IsInitialized()
        // @pre !IsTextureAcquired()
        // @post IsTexturePreAcquired()
        // @details
        // この関数が成功した場合、テクスチャが予約状態になり次の AcquireTexture が即座に成功することが保証されます。
        // 失敗した場合、呼出元のアプレットがテクスチャの使用権を得ていない可能性があります。
        // 失敗した場合には描画を行わず、メッセージ処理を進めてください。
        //
        // 既にテクスチャが予約状態だった場合、この関数は何もせずに即座に成功を返します。
        //
        // この関数が成功した後でアプレットが BG に遷移する際には AcquireTexture と CancelTexture を呼び出すか、
        // または CancelPreAcquiredTexture を呼び出して予約したテクスチャをシステムに返却する必要があります。
        nn::Result PreAcquireTexture(nn::TimeSpan timeout) NN_NOEXCEPT;

        // @brief PreAcquireTexture を取り消します。
        // @pre IsInitialized()
        // @post !IsTexturePreAcquired()
        // @details
        // PreAcquireTexture() が成功したあとで呼び出すことで PreAcquireTexture() の結果を取り消します。
        // それ以外のタイミングで呼び出した場合、何もせずに返ります。
        //
        // アプレットが BG に遷移する際には予約状態になったテクスチャをシステムに返却する必要があります。
        void CancelPreAcquiredTexture() NN_NOEXCEPT;

        // @brief 次のフレームバッファテクスチャを取得します。
        // @pre IsInitialized()
        // @pre !IsTextureAcquired()
        // @pre pOutTextureAvailableSemaphore != nullptr
        // @pre pOutTextureIndex != nullptr
        // @pre pOutTexture != nullptr
        // @post IsTextureAcquired()
        // @post !IsTexturePreAcquired()
        // @details
        // 以下のいずれかの場合、ブロックします。
        //
        // - テクスチャがディスプレイから返却されていない
        // - 呼び出し元のアプレットがテクスチャの使用権を得ていない
        //
        // 先に PreAcquireTexture を呼び出して Acquire に成功するかを確認することを推奨します。
        //
        // この API で取得される TextureIndex は Texture と対応しません。
        // TextureIndex の値は 0 から Initialize の引数で設定した bufferCount までの値を順に繰り返します。
        // 例えば bufferCount = 2 で初期化した場合、 0, 1, 0, 1, ... の順に TextureIndex が取得されます。
        //
        // Texture のポインタは FG を取る度に変化する可能性があります。
        // まったく別のポインタになる場合もあれば、同じポインタが 2 回連続で返る場合もあります。
        //
        // pOutTextureAvailableSemaphore には初期化済のセマフォへのポインタを渡します。
        // pOutTextureAvailableFence には初期化済のフェンスへのポインタまたは nullptr を渡します。
        // 成功した場合、 *pOutTextureAvailableSemaphore および *pOutTextureAvailableFence （非 nullptr の場合）は同一の同期オブジェクトが代入されます。
        nn::Result AcquireTexture(nn::gfx::Semaphore* pOutTextureAvailableSemaphore, nn::gfx::Fence* pOutTextureAvailableFence, int* pOutTextureIndex, nn::gfx::Texture** pOutTexture) NN_NOEXCEPT;


        // @brief 予約済みの次のフレームバッファテクスチャを取得します。
        // @pre IsInitialized()
        // @pre !IsTextureAcquired()
        // @pre pOutTextureAvailableSemaphore != nullptr
        // @pre pOutTextureIndex != nullptr
        // @pre pOutTexture != nullptr
        // @post IsTextureAcquired()
        // @post !IsTexturePreAcquired()
        // @details
        // 事前に PreAcquireTexture が成功していた場合、予約済のテクスチャを取得します。
        // それ以外の場合、テクスチャとして nullptr が取得されます。この場合でも IsTextureAcquired() == true になります。
        // この関数はブロックしません。
        //
        // この API で取得される TextureIndex は Texture と対応しません。
        // TextureIndex の値は 0 から Initialize の引数で設定した bufferCount までの値を順に繰り返します。
        // 例えば bufferCount = 2 で初期化した場合、 0, 1, 0, 1, ... の順に TextureIndex が取得されます。
        //
        // Texture のポインタは FG を取る度に変化する可能性があります。
        // まったく別のポインタになる場合もあれば、同じポインタが 2 回連続で返る場合もあります。
        //
        // テクスチャとして nullptr が取得された場合でも TextureIndex は有効な値が返ります。
        // また、テクスチャとして nullptr が取得された場合でも PresentTexture または CancelTexture を呼び出す必要があります。
        //
        // pOutTextureAvailableSemaphore には初期化済のセマフォへのポインタを渡します。
        // pOutTextureAvailableFence には初期化済のフェンスへのポインタまたは nullptr を渡します。
        // *pOutTextureAvailableSemaphore および *pOutTextureAvailableFence （非 nullptr の場合）は同一の同期オブジェクトが代入されます。
        void AcquirePreAcquiredTexture(nn::gfx::Semaphore* pOutTextureAvailableSemaphore, nn::gfx::Fence* pOutTextureAvailableFence, int* pOutTextureIndex, nn::gfx::Texture** pOutTexture) NN_NOEXCEPT;

        // @brief フレームバッファテクスチャをディスプレイに送ります。
        // @pre IsInitialized()
        // @pre IsTextureAcquired()
        // @post !IsTextureAcquired()
        // @details
        // この関数はブロックせずに成功します。
        // アプレットが FG を失う際には Present してから描画スレッドを止めるようにしてください。
        // Present せずにスレッドを止めた場合、 GPU 処理の完了が保証されず、テアリングが発生する場合があります。
        void PresentTexture(nn::gfx::Queue* pQueue, int textureIndex) NN_NOEXCEPT;

        // @brief フレームバッファテクスチャをシステムに返却します。
        // @pre IsInitialized()
        // @post !IsTextureAcquired() （獲得済のインデックスを指定した場合）
        // @details
        // この関数はブロックせずに成功します。
        // AcquireTexture により獲得したテクスチャをシステムに返却します。
        // この関数により返却したテクスチャはディスプレイに出力されません。
        //
        // システムはこの関数により返却したテクスチャに対する GPU 処理の完了を待ちません。
        // 呼出元アプレットはこの関数の呼出後に返却したテクスチャに描画が行われないことを保証する必要があります。
        //
        // アプレットが BG に遷移する際にはこの関数を呼び出して獲得したテクスチャをシステムに返却する必要があります。
        // 獲得していないテクスチャに対して呼び出した場合には何も行わずに返ります。
        void CancelTexture(int textureIndex) NN_NOEXCEPT;

        // @brief スワップインターバルを設定します。
        // @pre IsInitialized()
        void SetSwapInterval(int value) NN_NOEXCEPT;

        // @brief フレームバッファのクロップ範囲を指定します。
        // @pre IsInitialized()
        void SetCrop(int x, int y, int width, int height) NN_NOEXCEPT;

        // @brief テクスチャに対するカラーターゲットビューを取得します。
        nn::gfx::ColorTargetView* GetTextureColorTargetView(nn::gfx::Texture* pTexture) NN_NOEXCEPT;

    // @}

    // @name フェード処理用テクスチャ
    // @{

        // @brief アプリケーションの最終出画のテクスチャを取得します。
        // @details
        // アプレットが FG を取っている間、同じ値が取得されることが保証されます。
        // 一旦 FG を失った後、 FG に復帰した際に再度テクスチャを利用する場合には再度この API を呼び出してください。
        nn::Result AcquireLastApplicationTexture(bool* pOutIsScreenShotPermitted, nn::gfx::Texture** pOutTexture) NN_NOEXCEPT;

        // @brief アプリケーションの最終出画のテクスチャを返却します。
        // @details
        // FG を失った後で呼び出しても構いません。この API を呼び出さずにアプレットが終了した場合、自動的にテクスチャが返却されます。
        void ReleaseLastApplicationTexture() NN_NOEXCEPT;

        // @brief 直前の FG アプレットの最終出画のテクスチャを取得します。
        // @details
        // アプレットが FG を取っている間、同じ値が取得されることが保証されます。
        // 一旦 FG を失った後、 FG に復帰した際に再度テクスチャを利用する場合には再度この API を呼び出してください。
        nn::Result AcquireLastForegroundTexture(bool* pOutIsScreenShotPermitted, nn::gfx::Texture** pOutTexture) NN_NOEXCEPT;

        // @brief 直前の FG アプレットの最終出画のテクスチャを返却します。
        // @details
        // FG を失った後で呼び出しても構いません。この API を呼び出さずにアプレットが終了した場合、自動的にテクスチャが返却されます。
        void ReleaseLastForegroundTexture() NN_NOEXCEPT;

        // @brief CallerApplet の最終出画のテクスチャを取得します。
        // @details
        // アプレットが FG を取っている間、同じ値が取得されることが保証されます。
        // 一旦 FG を失った後、 FG に復帰した際に再度テクスチャを利用する場合には再度この API を呼び出してください。
        nn::Result AcquireCallerAppletTexture(bool* pOutIsScreenShotPermitted, nn::gfx::Texture** pOutTexture) NN_NOEXCEPT;

        // @brief CallerApplet の最終出画のテクスチャを返却します。
        // @details
        // FG を失った後で呼び出しても構いません。この API を呼び出さずにアプレットが終了した場合、自動的にテクスチャが返却されます。
        void ReleaseCallerAppletTexture() NN_NOEXCEPT;
    // @}

    // @name 内部実装用
    // @{
    public:
        nn::Result GetInternalTextureIndex(int* pOutValue, const nn::gfx::Texture* pTexture) const NN_NOEXCEPT;

    private:
        void ConvertToGfxTextureImpl(nn::gfx::Texture** pOutGfxTexture, NVNtexture* pNvnTexture) NN_NOEXCEPT;
    // @}

    private:
        SharedLayerWindow m_WindowNvn;
        nn::gfx::Device* m_pDevice;
        nn::gfx::ImageFormat m_ImageFormat;
        nn::util::BitFlagSet<InternalTextureCountMax> m_IsGfxTextureInitializedList;
        nn::gfx::Texture m_GfxTextureList[InternalTextureCountMax];
        nn::gfx::ColorTargetView m_GfxColorTargetViewList[InternalTextureCountMax];
    };

}}}

