﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file    usb_Device.h
 * @brief   USB Device Stack Client Interface
 */

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_HipcSimpleClientSessionManager.h>

#include <nn/usb/usb_Limits.h>
#include <nn/usb/usb_Result.h>
#include <nn/usb/usb_DeviceTypes.h>
#include <nn/usb/ds/sfdl/usb_IDsService.sfdl.h>
#include <nn/usb/ds/sfdl/usb_IDsInterface.sfdl.h>
#include <nn/usb/ds/sfdl/usb_IDsEndpoint.sfdl.h>

namespace nn {
namespace usb {

class DsInterface;
class DsEndpoint;

class DsClient
{
    friend class DsInterface;
    friend class DsEndpoint;

public:
    DsClient() NN_NOEXCEPT
        : m_IsInitialized(false)
        , m_RefCount(0)


    {
    }

    ~DsClient() NN_NOEXCEPT
    {
    }

    /**
     * @brief Initialize the USB Device Stack client
     *
     * @param[in] complexId
     *   ID of the USB complex this client will bind to
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Initialize succeeded}
     *   @handleresult{nn::usb::ResultNoSuchDevice}
     *   @handleresult{nn::usb::ResultAlreadyInitialized, Object is already in the initialized state}
     *   @handleresult{nn::usb::ResultMemAllocFailure, Insufficient internal resources available to complete this operation}
     * @endretresult
     *
     * @pre
     *   The client is not initialized
     *
     * @post
     *   The client is initialized
     *
     * @details
     *   This call establishs a connection between the client and the server
     *   as well as binds the client to specified hardware device.
     *
     *   You must bind it to a USB 3.0 capable device if you need USB 3.0
     *   functionality.
     */
    Result Initialize(ComplexId complexId) NN_NOEXCEPT;

    /**
     * @brief Clear device data
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Initialize succeeded}
     * @endretresult
     *
     * @pre
     *   The cleint is initialized
     *
     * @post
     *   All device descriptor data is cleared
     *
     * @details
     *   This call clears all descriptor data in the USB device.
     */
    Result ClearDeviceData() NN_NOEXCEPT;

    /**
     * @brief Add UsbStringDescriptor to device
     *
     * @param[out] pIndex
     *   Index of string descriptor in device
     *
     * @param[in] pUsbStringDescriptor
     *   Pointer to UsbStringDescriptor to add to device
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Initialize succeeded}
     *   @handleresult{nn::usb::ResultImplementationLimit}
     * @endretresult
     *
     * @pre
     *   The cleint is initialized
     *
     * @post
     *   The UsbStringDescriptor is added to the device
     *
     * @details
     *   This call adds a UsbStringDescriptor to the device. The string index
     *   is returned in *pIndex on ResultSuccess(). Ths index is then
     *   referenced by other USB descriptors to point to string.
     */
    Result AddUsbStringDescriptor(uint8_t *pIndex, UsbStringDescriptor *pUsbStringDescriptor) NN_NOEXCEPT;

    /**
     * @brief Delete specified string index from USB device
     *
     * @param[in] index
     *   Index of string to deete
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Initialize succeeded}
     *   @handleresult{nn::usb::ResultInvalidParameter}
     * @endretresult
     *
     * @pre
     *   The cleint is initialized
     *
     * @post
     *   The UsbStringDescriptor specified by index is deleted from the device
     *
     * @details
     *   This call deletes a UsbStringDescriptor from the device.
     */
    Result DeleteUsbStringDescriptor(uint8_t index) NN_NOEXCEPT;

    /**
     * @brief Set UsbDeviceDescriptor to the USB device
     *
     * @param[in] pUsbDeviceDescriptor
     *   Pointer to UsbDeviceDescriptor
     *
     * @param[in] usbDeviceSpeed
     *   Device speed where the USB device returns the UsbDeviceDescriptor on
     *   GET_DESCRIPTOR ctrl transaction.
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Initialize succeeded}
     * @endretresult
     *
     * @pre
     *   The cleint is initialized
     *
     * @post
     *   The UsbDeviceDescriptor is set for the specified UsbDeviceSpeed
     *
     * @details
     *   This call sets the UsbDeviceDescriptor for specified UsbDeviceSpeed.
     *   UsbDeviceDescriptor can vary for full, high, super speed for
     *   different reasons.
     */
    Result SetUsbDeviceDescriptor(UsbDeviceDescriptor *pUsbDeviceDescriptor, UsbDeviceSpeed usbDeviceSpeed) NN_NOEXCEPT;


    /**
     * @brief Set binary object store to the USB device
     *
     * @param[in] pData
     *   Pointer to the binary object store descriptor set
     *
     * @param[in] size
     *   Size in bytes for the entire binary object store
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Initialize succeeded}
     * @endretresult
     *
     * @pre
     *   The cleint is initialized
     *
     * @post
     *   The binary object store is set for the USB device
     *
     * @details
     *   This call sets the binary object store for the USB device.
     */
    Result SetBinaryObjectStore(uint8_t *pData, int size) NN_NOEXCEPT;

    /**
     * @brief Finalize the USB Device Stack client
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Finalize succeeded}
     *   @handleresult{nn::usb::ResultNotInitialized, This object is not initialized}
     * @endretresult
     *
     * @pre
     *   The cleint is initialized
     *   Any DeviceEndpoint objects initialized against this
     *   object have since been finalized.
     *
     * @post
     *   The client is not initialized
     *
     * @details
     *   The client is disconnected from the server. All associated interfaces
     *   and endpoints are also disabled and finalized.
     */
    Result Finalize() NN_NOEXCEPT;


    /**
     * @brief Check if this device session has been initialized
     *
     * @retval Boolean
     *   - True: The device session has been initialized
     *   - False: The device session has not been initialized
     *
     * @details
     *   Determines if this device session has been initialized. Has Initialize()
     *   been called?
     */
    bool IsInitialized() NN_NOEXCEPT;

    /**
     * @brief Get the system event for USB State Change Event
     *
     * @return A pointer to the system event
     *
     * @pre
     *   The client is initialized
     *
     * @details
     *   The returned pointer points to a system event object managed by this
     *   object. Whenever the USB state (as defined in USB 2.0 Spec, Table 9-1)
     *   is changed, this system event will be triggered.
     *
     *   The returned system event should be cleared manually.
     *
     *   Note that the pointer returned will become invalid if this DsClient
     *   object is finalized or destroied.
     */
    nn::os::SystemEventType* GetStateChangeEvent() NN_NOEXCEPT;

    /**
     * @brief Get current USB state
     *
     * @param[out] pOutState
     *   Current USB state
     *
     * @retresult{Always success}
     * @endretresult
     *
     * @pre
     *   The client is initialized
     *
     * @details
     *   Query and return current USB state.
     *
     *   Valid USB states are:
     *     - nn::usb::UsbState_Detached
     *     - nn::usb::UsbState_Attached
     *     - nn::usb::UsbState_Powered
     *     - nn::usb::UsbState_Default
     *     - nn::usb::UsbState_Address
     *     - nn::usb::UsbState_Configured
     *     - nn::usb::UsbState_Suspended
     *
     *   The USB State Change Event is always cleared by this call.
     */
    Result GetState(UsbState *pOutState) NN_NOEXCEPT;

    /**
     * @brief Enable USB device
     *
     * @retresult{Always success}
     * @endretresult
     *
     * @pre
     *   The client is initialized and device data, interfaces, endpoints are setup
     *
     * @details
     *   Enables USB device.
     */
    Result EnableDevice() NN_NOEXCEPT;

    /**
     * @brief Disable USB device
     *
     * @retresult{Always success}
     * @endretresult
     *
     * @pre
     *   The device is enabled successfully
     *
     * @details
     *   Disables USB device.
     */
    Result DisableDevice() NN_NOEXCEPT;


private:
    typedef nn::sf::ExpHeapAllocator MyAllocator;

    MyAllocator                               m_Allocator;
    std::aligned_storage<32 * 1024>::type     m_HeapBuffer;
    nn::lmem::HeapHandle                      m_HeapHandle;

    nn::sf::HipcSimpleClientSessionManager    m_Domain;
    nn::sf::SharedPointer<ds::IDsService>     m_Handle;
    bool                                      m_IsInitialized;
    ::std::atomic<int>                        m_RefCount;

    nn::os::SystemEventType                   m_StateChangeEvent;

    DsInterface                              *m_pInterfaces[DsLimitMaxInterfacesPerConfigurationCount];

private:
    Result AddInterface(DsInterface                             *pInterface,
                        nn::sf::SharedPointer<ds::IDsInterface> *pHandle,
                        uint8_t                                 bInterfaceNumber) NN_NOEXCEPT;
    Result DeleteInterface(uint8_t ifNum) NN_NOEXCEPT;
};

class DsInterface
{
    friend class DsEndpoint;

public:
    /**
     * @brief Constructor of the interface session
     */
    DsInterface() NN_NOEXCEPT
        : m_pClient(nullptr)
        , m_IsInitialized(false)
        , m_RefCount(0)
        , m_IsEnabled(false)
    {
    }

    /**
     * @brief Destructor of the interface session
     */
    ~DsInterface() NN_NOEXCEPT { }

    /**
     * @brief Initialize an USB device interface
     *
     * @param[in] pClient
     *   Pointer to an USB Device Stack client
     *
     * @param[in] bInterfaceNumber
     *   bInterfaceNumber for the interface
     *
     * @retresult
     *   @handleresult{nn::usb::ResultInvalidDescriptor}
     *   @handleresult{nn::usb::ResultResourceBusy}
     * @endretresult
     *
     * @pre
     *   The client is initialized. The interface is not initialized.
     *
     * @post
     *   The interface is initialized and in disabled state. The interface
     *   is associated with the client.
     *
     * @details
     *   Initialize an interface with the given descriptor. The interface
     *   is put into disabled state and you must Enable() it before any data
     *   transaction can be made.
     *
     *   The required endpoints are not created automatically. You must
     *   create instances of DsEndpoint class and associate them with the
     *   interface before Enable() it.
     *
     *   The interface is associated with specified client.
     */
    Result Initialize(DsClient *pClient, uint8_t bInterfaceNumber) NN_NOEXCEPT;


    /**
     * @brief Append to the configuration data for the USB interface
     *
     * @param[in] usbDeviceSpeed
     *   UsbDeviceSpeed intended for this configuration data
     *
     * @param[in] pData
     *   Pointer to descriptor data
     *
     * @param[in] size
     *   Size of data in bytes
     *
     * @retresult
     *   @handleresult{nn::usb::ResultInvalidParameter}
     *   @handleresult{nn::usb::ResultBufferSizeError}
     * @endretresult
     *
     * @pre
     *   The interface is initialized.
     *
     * @post
     *   The specified data is appended to the configuration data for this interface.
     *
     * @details
     *   This function appends data to the configuration descriptors stream for
     *   the USB interface.
     */
    Result AppendConfigurationData(UsbDeviceSpeed usbDeviceSpeed, void *pData, uint32_t size) NN_NOEXCEPT;


    /**
     * @brief Finalize the USB device interface
     *
     * @retresult
     *   @handleresult{nn::usb::ResultResourceBusy}
     * @endretresult
     *
     * @pre
     *   The interface is in initialized and disabled state
     *
     * @post
     *   The interface is not initialized. All associated endpoints are also
     *   not initialized.
     *
     * @details
     *   Destroy the interface and associated endpoints. Release all the
     *   hardware resources.
     */
    Result Finalize() NN_NOEXCEPT;

    /**
     * @brief Check if this device interface has been initialized
     *
     * @retval Boolean
     *   - True: The device interface has been initialized
     *   - False: The device interface has not been initialized
     *
     * @details
     *   Determines if this device interface has been initialized. Has Initialize()
     *   been called?
     */
    bool IsInitialized() NN_NOEXCEPT;

    /**
     * @brief Get the system event for Setup Request Event
     *
     * @return A pointer to the system event
     *
     * @pre
     *   The interface is initialized
     *
     * @details
     *   The returned pointer points to a system event object managed by this
     *   object. Whenever a SETUP packet which is not a standard device request
     *   is received in default control pipe, and the packet is targeted to
     *   this interface, the system event will be triggered.
     *
     *   The returned system event should be cleared manually.
     *
     *   Note that the pointer returned will become invalid if this DsInterface
     *   object is finalized or destroied.
     */
    nn::os::SystemEventType* GetSetupEvent() NN_NOEXCEPT;

    /**
     * @brief Get the latest SETUP packet
     *
     * @param[out] pOutSetup
     *   The SETUP packet returned to user
     *
     * @retresult{Always success}
     * @endretresult
     *
     * @pre
     *   The interface is initialized
     *
     * @details
     *   Query and return the last SETUP packet received in default control
     *   pipe. The packet is not necessarily a standard device request or
     *   targeted to this interface.
     *
     *   The Setup Request Event is always cleared by this call.
     */
    Result GetSetupPacket(UsbCtrlRequest *pOutSetup) NN_NOEXCEPT;

    /**
     * @brief Enable the interface
     *
     * @retresult
     *   @handleresult{nn::usb::ResultOperationDenied}
     * @endretresult
     *
     * @pre
     *   The interface is initialized. Requested endpoints have been initialized
     *   and associated with this interface.
     *
     * @post
     *   The interface is enabled and ready for data transaction. Indication and
     *   setup packet could come in anytime.
     *
     * @details
     *   Enable the interface. Associated device will be reattached. Host
     *   controller will re-enumerate the device. Note that the re-enumeration
     *   affects all the other interfaces which are bound to the same device.
     */
    Result Enable() NN_NOEXCEPT;

    /**
     * @brief Disable the interface
     *
     * @retresult
     *   @handleresult{nn::usb::ResultOperationDenied}
     * @endretresult
     *
     * @pre
     *  The interface is initialized.
     *
     * @post
     *   Disable the interface. Associated device will be reattached. Host
     *   controller will re-enumerate the device. Note that the re-enumeration
     *   affects all the interfaces which are bound to the same device.
     */
    Result Disable() NN_NOEXCEPT;

    /**
     * @brief Do a control read through default control pipe
     *
     * @param[out] pOutBytesTransferred
     *   Actual size of the data stored in user buffer
     *
     * @param[out] pOutBuffer
     *   The user buffer used to store received data
     *
     * @param[in] bytes
     *   The size of the user buffer in bytes
     *
     * @retresult
     *   @handleresult{nn::usb::ResultOperationDenied}
     *   @handleresult{nn::usb::ResultAlignmentError}
     * @endretresult
     *
     * @pre
     *   The interface is initialized and enabled
     *
     * @details
     *   This is used to reply a SETUP packet which requires data-in stage. The
     *   status stage is also handled by this call.
     */
    Result CtrlRead(uint32_t *pOutBytesTransferred, void *pOutBuffer, uint32_t bytes) NN_NOEXCEPT;

    /**
     * @brief Do a control write through default control pipe
     *
     * @param[out] pOutBytesTransferred
     *   Actually size of the data transferred from user buffer
     *
     * @param[in] buffer
     *   The user buffer used to provide the data
     *
     * @param[in] bytes
     *   The size of the user buffer in bytes
     *
     * @retresult
     *   @handleresult{nn::usb::ResultOperationDenied}
     *   @handleresult{nn::usb::ResultAlignmentError}
     * @endretresult
     *
     * @pre
     *   The interface is initialized and enabled
     *
     * @details
     *   This is used to reply a SETUP packet which requires data-out stage. The
     *   status stage is also handled by this call.
     */
    Result CtrlWrite(uint32_t *pOutBytesTransferred, void *buffer, uint32_t bytes) NN_NOEXCEPT;

    /**
     * @brief Tell USB Device Stack that the current SETUP packet has been handled
     *
     * @retresult
     *   @handleresult{nn::usb::ResultOperationDenied}
     * @endretresult
     *
     * @pre
     *   The interface is initialized and enabled
     *
     * @details
     *   This should be called after a SETUP packet which doesn't require a data
     *   stage. The status stage is handled by this call.
     */
    Result CtrlDone() NN_NOEXCEPT;

    /**
     * @brief Stall the default control pipe
     *
     * @retresult
     *   @handleresult{nn::usb::ResultOperationDenied}
     * @endretresult
     *
     * @pre
     *   The interface is initialized and enabled
     *
     * @details
     *   This should be called when error happens while handling current SETUP
     *   packet. The default control pipe is put into stall state until next
     *   SETUP packet arrives. Status stage is not performed.
     */
    Result CtrlStall() NN_NOEXCEPT;

private:
    DsClient                                  *m_pClient;
    nn::sf::SharedPointer<ds::IDsInterface>    m_Handle;
    bool                                       m_IsInitialized;
    ::std::atomic<int>                         m_RefCount;

    nn::os::SystemEventType                    m_SetupEvent;
    nn::os::SystemEventType                    m_CtrlInCompletionEvent;
    nn::os::SystemEventType                    m_CtrlOutCompletionEvent;

    UrbReport                                  m_Report;
    uint8_t                                    m_IfNum;
    DsEndpoint                                *m_pEndpoints[UsbLimitMaxEndpointsCount];
    bool                                       m_IsEnabled;

private:
    Result AddEndpoint(DsEndpoint                             *pEndpoint,
                       uint8_t                                 bEndpointAddress,
                       nn::sf::SharedPointer<ds::IDsEndpoint> *pHandle) NN_NOEXCEPT;

    Result DeleteEndpoint(uint8_t epAddress) NN_NOEXCEPT;

    Result CtrlIn(uint32_t *pOutBytesTransferred,
                  void     *buffer,
                  uint32_t  bytes) NN_NOEXCEPT;

    Result CtrlOut(uint32_t *pOutBytesTransferred,
                   void     *pOutBuffer,
                   uint32_t  bytes) NN_NOEXCEPT;
};

class DsEndpoint
{
public:
    /**
     * @brief Constructor of the DsEndpoint
     */
    DsEndpoint() NN_NOEXCEPT
        : m_IsInitialized(false)
        , m_RefCount(0)
    {
    }
    /**
     * @brief Initialize an endpoint
     *
     * @param[in] pInterface
     *   Pointer to a USB interface
     *
     * @param[in] bEndpointAddress
     *   bEndpointAddress associated with the endpoint
     *
     * @retresult
     *   @handleresult{nn::usb::ResultOperationDenied}
     *   @handleresult{nn::usb::ResultResourceBusy}
     * @endretresult
     *
     * @pre
     *   The interface is initialized and disabled. The endpoint is not
     *   initialized.
     *
     * @post
     *   The endpoint is initialized and associated with the interface.
     *
     * @details
     *   Initialize the endpoint with the descriptor and associate it with
     *   the given interface. Necessary hardward resources are allocated to
     *   this endpoint.
     */
    Result Initialize(DsInterface *pInterface, uint8_t bEndpointAddress) NN_NOEXCEPT;

    /**
     * @brief Finalize the endpoint
     *
     * @retresult
     *   @handleresult{nn::usb::ResultOperationDenied}
     * @endretresult
     *
     * @pre
     *   The endpoint is initialized.
     *
     * @post
     *   The endpoint is not initialized. It's not associated with any
     *    interface.
     *
     * @details
     *   The endpoint is destroyed and related hardware resources are freed.
     */
    Result Finalize() NN_NOEXCEPT;

    /**
     * @brief Check if this device endpoint has been initialized
     *
     * @retval Boolean
     *   - True: The device endpoint has been initialized
     *   - False: The device endpoint has not been initialized
     *
     * @details
     *   Determines if this device endpoint has been initialized. Has Initialize()
     *   been called?
     */
    bool IsInitialized() NN_NOEXCEPT;

    /**
     * @brief Transfer data with this endpoint
     *
     * @param[out] pOutBytesTransferred
     *   The actual size of data transferred from/into user buffer
     *
     * @param[in,out] buffer
     *   The user buffer to provide data or store received data
     *
     * @param[in] bytes
     *   The size of user buffer in bytes
     *
     * @retresult
     *   @handleresult{nn::usb::ResultOperationDenied}
     *   @handleresult{nn::usb::ResultResourceBusy}
     *   @handleresult{nn::usb::ResultAlignmentError}
     * @endretresult
     *
     * @pre
     *   The endpoint is initialized and associated interface is enabled.
     *   The buffer must be aligned to nn::usb::HwLimitDmaBufferAlignmentSize.
     *
     * @details
     *   Depends on the direction of this endpoint, this call will transfer data
     *   from or receive data to the user provided buffer.
     *
     *   The buffer must be aligned to nn::usb::HwLimitDmaBufferAlignmentSize,
     *   however, the cache coherence is handled for you.
     *
     *   You are not supposed to interleave this call with PostBufferAsync().
     */
    Result PostBuffer(uint32_t *pOutBytesTransferred, void *buffer, uint32_t bytes) NN_NOEXCEPT;

    /**
     * @brief Submit an URB (USB Request Block) for data transfer
     *
     * @param[out] pOutUrbId
     *   The unique ID of submitted URB (USB Request Block)
     *
     * @param[in,out] buffer
     *   The user buffer to provide data or store received data
     *
     * @param[in] bytes
     *   The size of user buffer in bytes
     *
     * @retresult
     *   @handleresult{nn::usb::ResultOperationDenied}
     *   @handleresult{nn::usb::ResultResourceBusy}
     *   @handleresult{nn::usb::ResultAlignmentError}
     * @endretresult
     *
     * @pre
     *   The endpoint is initialized and associated interface is enabled.
     *   The buffer must be aligned to nn::usb::HwLimitDmaBufferAlignmentSize.
     *
     * @details
     *   Depends on the direction of this endpoint, this call submits a request
     *   to transfer data from or receive data to the user provided buffer. The
     *   call returns immediately and doesn't wait for data transfer completion.
     *
     *   The maximum number of pending requests is nn::usb::DsLimitRingSize.
     *   This call returns nn::usb::ResultResourceBusy if this limitation is
     *   reached. In this case, you must wait until some pending requests are
     *   completed before submitting more reuquests.
     *
     *   The transaction status of all the requests can be polled by
     *   GetUrbReport(). Look into the report with the URB ID returned by this
     *   call as the key if you need to know the status of a specific request.
     *
     *   The buffer must be aligned to nn::usb::HwLimitDmaBufferAlignmentSize.
     *   The ownership of the buffer is effectively transferred to DMA
     *   Controller, thus you shouldn't access the buffer in any means before
     *   the request is completed. You must also handle the cache coherence
     *   by yourself.
     */
    Result PostBufferAsync(uint32_t       *pOutUrbId,
                           void           *buffer,
                           uint32_t        bytes) NN_NOEXCEPT;

    /**
     * @brief Get the system event for URB Completion Event
     *
     * @return A pointer to the system event
     *
     * @pre
     *   The endpoint is initialized
     *
     * @details
     *   The returned pointer points to a system event object managed by this
     *   object. Whenever a submitted URB is completed, this system event will
     *   be triggered.
     *
     *   The returned system event should be cleared manually.
     *
     *   Note that the pointer returned will become invalid if this DsEndpoint
     *   object is finalized or destroied.
     */
    nn::os::SystemEventType* GetCompletionEvent() NN_NOEXCEPT;

    /**
     * @brief Poll the status of all submitted URBs
     *
     * @param[out] pOutReport
     *   The status report of all URBs
     *
     * @retresult{Always success}
     * @endretresult
     *
     * @details
     *   Get the status report for all the submitted URBs whos status is not
     *   nn::usb::UrbStatus_Invalid. This call updates the status of any
     *   completed URBs to nn::usb::UrbStatus_Invalid, so you won't get the
     *   report of such URB twice. The URBs in following status are considered
     *   "completed":
     *     - nn::usb::UrbStatus_Finished
     *     - nn::usb::UrbStatus_Cancelled
     *     - nn::usb::UrbStatus_Failed
     *
     *   The URB Completion Event is always cleared by this call.
     */
    Result GetUrbReport(UrbReport *pOutReport) NN_NOEXCEPT;

    /**
     * @brief Cancel all the ongoing or pending URBs
     *
     * @retresult
     *   @handleresult{nn::usb::ResultOperationDenied}
     * @endretresult
     *
     * @pre
     *   The endpoint is initialized and associated interface is enabled.
     *
     * @details
     *   Cancel all the ongoing or pending URBs on this endpoint.
     */
    Result Cancel() NN_NOEXCEPT;

    /**
     * @brief Set the transfer terminate behavior of this endpoint
     *
     * @param[in] isZeroLengthTerminate
     *   A bool flag, true or false
     *
     * @retresult
     *   @handleresult{nn::usb::ResultOperationDenied}
     * @endretresult
     *
     * @pre
     *   The endpoint is initialized and associated interface is enabled.
     *
     * @details
     *   Set whether zero length packets will be used to terminate transfers
     *   that are a multiple of the max packet size
     */
    Result SetZeroLengthTransfer(bool isZeroLengthTerminate) NN_NOEXCEPT;

private:
    bool                                      m_IsInitialized;
    ::std::atomic<int>                        m_RefCount;

    DsInterface                             *m_pInterface;

    nn::sf::SharedPointer<ds::IDsEndpoint>   m_Handle;

    uint8_t                                  m_Address;

    nn::os::SystemEventType                  m_CompletionEvent;
};

} // end of namespace usb
} // end of namespace nn
