﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   USB::PD ライブラリの型や定数の定義
 */

#pragma once

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/util/util_BitPack.h>
#include <nn/usb/pd/usb_PdPdoRdo.h>

namespace nn {
namespace usb {
namespace pd {

/**
 * @brief Status の PlugOrientation
 */
enum StatusPlugOrientation
{
    StatusPlugOrientation_Cc1                                   = 0,
    StatusPlugOrientation_Cc2                                   = 1
};

/**
 * @brief Status の PowerRole
 */
enum StatusPowerRole
{
    StatusPowerRole_Sink                                        = 0,
    StatusPowerRole_Source                                      = 1
};

/**
 * @brief Status の DataRole
 */
enum StatusDataRole
{
    StatusDataRole_Unknown                                      = 0,
    StatusDataRole_Ufp                                          = 1,    // Upstream Facing Port
    StatusDataRole_Dfp                                          = 2,    // Downstream Facing Port
    StatusDataRole_AccessoryMode                                = 3     // Not USB Device/Host
};

/**
 * @brief Status の AccessoryMode
 */
enum StatusAccessoryMode
{
    StatusAccessoryMode_None                                    = 0,
    StatusAccessoryMode_Audio                                   = 1,
    StatusAccessoryMode_Debug                                   = 2,
    StatusAccessoryMode_VConnPowered                            = 3
};

/**
 * @brief Status のデバイスタイプ
 */
enum StatusDeviceType
{
    StatusDeviceType_Unknown                                    = 0,
    StatusDeviceType_Cradle                                     = 1,
    StatusDeviceType_RelayBox                                   = 2,
    StatusDeviceType_AcAdaptor                                  = 3,
    StatusDeviceType_TableDock                                  = 4
};

/**
 * @brief Status のリクエスト
 */
enum StatusRequest
{
    StatusRequest_None                                          = 0,
    StatusRequest_PowerOutput                                   = 1,
    StatusRequest_PowerInput                                    = 2
};

/**
 * @brief Status のエラー
 */
enum StatusError
{
    StatusError_None                                            = 0,
    StatusError_OverVoltage                                     = 1,
    StatusError_CradleUsbHubUndetected                          = 2
};

/**
 * @brief 通知フラグ
 */
struct Notice : public nn::util::BitPack16
{
    typedef Field<   0,  1, bool  > ActiveNotice;
    typedef Field<   1,  1, bool  > ErrorNotice;
    typedef Field<   2,  1, bool  > DataRoleNotice;
    typedef Field<   3,  1, bool  > PowerRoleNotice;
    typedef Field<   4,  1, bool  > ConsumerContractNotice;
    typedef Field<   5,  1, bool  > ProviderContractNotice;
    typedef Field<   6,  1, bool  > DeviceNotice;
    typedef Field<   7,  1, bool  > RequestNotice;

    bool IsActiveNotice() const NN_NOEXCEPT { return this->Get<Notice::ActiveNotice>(); }
    bool IsErrorNotice() const NN_NOEXCEPT { return this->Get<Notice::ErrorNotice>(); }
    bool IsDataRoleNotice() const NN_NOEXCEPT { return this->Get<Notice::DataRoleNotice>(); }
    bool IsPowerRoleNotice() const NN_NOEXCEPT { return this->Get<Notice::PowerRoleNotice>(); }
    bool IsDeviceNotice() const NN_NOEXCEPT { return this->Get<Notice::DeviceNotice>(); }
    bool IsConsumerContractNotice() const NN_NOEXCEPT { return this->Get<Notice::ConsumerContractNotice>(); }
    bool IsProviderContractNotice() const NN_NOEXCEPT { return this->Get<Notice::ProviderContractNotice>(); }
    bool IsRequestNotice() const NN_NOEXCEPT { return this->Get<Notice::RequestNotice>(); }
};

/**
 * @brief ステータス
 */
struct Status
{
    nn::util::BitPack32 m_Data;
    NN_PADDING4;
    Pdo                 m_CurrentPdo;
    Rdo                 m_CurrentRdo;
    Bit16               m_VendorId;
    Bit16               m_ProductId;

    // status
    typedef nn::util::BitPack32::Field<  0,  1, bool  > Active;
    typedef nn::util::BitPack32::Field<  1,  1, Bit8  > PlugOrientation;
    typedef nn::util::BitPack32::Field<  2,  2, Bit8  > DataRole;
    typedef nn::util::BitPack32::Field<  4,  1, Bit8  > PowerRole;
    typedef nn::util::BitPack32::Field<  5,  1, bool  > ElectronicallyMarkedCable;  // 3～5A 対応 IC 搭載ケーブル
    typedef nn::util::BitPack32::Field<  6,  2, Bit8  > AccessoryMode;
    typedef nn::util::BitPack32::Field<  8,  1, bool  > DisplayPortAlternateMode;

    // VDM status
    typedef nn::util::BitPack32::Field< 10,  2, Bit8  > VdmFatalError;
    typedef nn::util::BitPack32::Field< 12,  3, Bit8  > VdmRelayBoxUsbPower;
    typedef nn::util::BitPack32::Field< 15,  1, bool  > VdmCradlePowerShortage;
    typedef nn::util::BitPack32::Field< 16,  1, bool  > VdmCradleWithUnofficialAcAdaptor;
    typedef nn::util::BitPack32::Field< 17,  4, Bit8  > VdmDeviceType;
    // DeviceType 拡張の余地を残す（他のフィールドが追加される可能性もあり）

    // request
    typedef nn::util::BitPack32::Field< 24,  4, Bit8  > Request;
    // error
    typedef nn::util::BitPack32::Field< 28,  4, Bit8  > Error;

    bool IsActive() const NN_NOEXCEPT { return m_Data.Get<Status::Active>(); }
    StatusPlugOrientation GetPlugOrientation() const NN_NOEXCEPT { return static_cast<StatusPlugOrientation>(m_Data.Get<Status::PlugOrientation>()); }
    StatusDataRole GetDataRole() const NN_NOEXCEPT { return static_cast<StatusDataRole>(m_Data.Get<Status::DataRole>()); }
    StatusPowerRole GetPowerRole() const NN_NOEXCEPT { return static_cast<StatusPowerRole>(m_Data.Get<Status::PowerRole>()); }
    bool IsElectronicallyMarkedCable() const NN_NOEXCEPT { return m_Data.Get<Status::ElectronicallyMarkedCable>(); }
    StatusAccessoryMode GetAccessoryMode() const NN_NOEXCEPT { return static_cast<StatusAccessoryMode>(m_Data.Get<Status::AccessoryMode>()); }
    bool IsDisplayPortAlternateMode() const NN_NOEXCEPT { return m_Data.Get<Status::DisplayPortAlternateMode>(); }

    StatusDeviceType GetDeviceType() const NN_NOEXCEPT { return static_cast<StatusDeviceType>(m_Data.Get<Status::VdmDeviceType>()); }
    bool IsCradlePowerShortage() const NN_NOEXCEPT
    {
        StatusDeviceType deviceType = GetDeviceType();
        return ( deviceType == StatusDeviceType_Cradle || deviceType == StatusDeviceType_RelayBox ) &&
               m_Data.Get<Status::VdmCradlePowerShortage>();
    }
    bool IsCradleWithUnofficialAcAdaptor() const NN_NOEXCEPT
    {
        StatusDeviceType deviceType = GetDeviceType();
        return ( deviceType == StatusDeviceType_Cradle || deviceType == StatusDeviceType_RelayBox ) &&
               m_Data.Get<Status::VdmCradleWithUnofficialAcAdaptor>();
    }
    bool IsTableDockPowerShortage() const NN_NOEXCEPT
    {
        return GetDeviceType() == StatusDeviceType_TableDock && m_Data.Get<Status::VdmCradlePowerShortage>();
    }
    bool IsTableDockWithUnofficialAcAdaptor() const NN_NOEXCEPT
    {
        return GetDeviceType() == StatusDeviceType_TableDock && m_Data.Get<Status::VdmCradleWithUnofficialAcAdaptor>();
    }

    StatusRequest GetRequest() const NN_NOEXCEPT { return static_cast<StatusRequest>(m_Data.Get<Status::Request>()); }
    StatusError GetError() const NN_NOEXCEPT { return static_cast<StatusError>(m_Data.Get<Status::Error>()); }

    void Clear() NN_NOEXCEPT
    {
        std::memset( this, 0, sizeof(Status) );
    }
};

} // pd
} // usb
} // nn
