﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   USB::PD ライブラリの API 宣言。
 */

#pragma once

#include <nn/usb/pd/usb_PdLib.h>
#include <nn/usb/pd/usb_PdResult.h>

namespace nn {
namespace usb {
namespace pd {

/**
 * @brief セッション用パラメータ
 */
struct ManufactureSession : public SessionCore
{
};

/**
 * @brief USB::PD 製造用ライブラリを初期化します。
 *
 * @post        ライブラリは初期化済の状態になります。
 * @details     USB::PD:M ライブラリを初期化します。
 *              本関数の呼び出し回数はライブラリ内部でカウントされます。
 */
void InitializeManufacture() NN_NOEXCEPT;

/**
 * @brief USB::PD 製造用ライブラリを終了します。
 *
 * @post        ライブラリは初期化前の状態になります。
 * @details     USB::PD:M ライブラリを終了します。
 *              ライブラリ初期化を行った回数分、本関数が呼ばれた場合にのみ、
 *              ライブラリが初期化前の状態になります。
 */
void FinalizeManufacture() NN_NOEXCEPT;

/**
 * @brief 製造用セッションをオープンします。
 *
 * @param[out]  pOutSession セッション用パラメータ
 * @pre         ライブラリは初期化済の状態である必要があります。
 * @post        指定したセッションがオープン状態になります。
 */
void OpenManufactureSession( ManufactureSession* pOutSession ) NN_NOEXCEPT;

/**
 * @brief 製造用セッションをクローズします。
 *
 * @param[in]   pSession セッション用パラメータ
 * @pre         ライブラリは初期化済の状態である必要があります。
 * @post        指定したセッションがクローズ状態になります。
 * @details     オープンされていないセッションに対して実行した場合は何もしません。
 *              送信予約されているがまだ送信されていないデータについては失われる可能性があります。
 */
void CloseManufactureSession( ManufactureSession* pSession ) NN_NOEXCEPT;

/**
 * @brief 本体 PDC FW タイプを取得します。
 *
 * @param[out]  pData           データのポインタ
 * @param[in]   pSession        製造用セッションのポインタ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess           データの取得に成功しました。
 * @pre         指定したセッションはオープン状態である必要があります。
 */
Result GetHostPdcFirmwareType( uint16_t* pData, ManufactureSession* pSession ) NN_NOEXCEPT;

/**
 * @brief 本体 PDC FW リビジョンを取得します。
 *
 * @param[out]  pData           データのポインタ
 * @param[in]   pSession        製造用セッションのポインタ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess           データの取得に成功しました。
 * @pre         指定したセッションはオープン状態である必要があります。
 */
Result GetHostPdcFirmwareRevision( uint16_t* pData, ManufactureSession* pSession ) NN_NOEXCEPT;

/**
 * @brief 本体 PDC 製造 ID を取得します。
 *
 * @param[out]  pData           データのポインタ
 * @param[in]   pSession        製造用セッションのポインタ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess           データの取得に成功しました。
 * @pre         指定したセッションはオープン状態である必要があります。
 */
Result GetHostPdcManufactureId( uint16_t* pData, ManufactureSession* pSession ) NN_NOEXCEPT;

/**
 * @brief 本体 PDC デバイス ID を取得します。
 *
 * @param[out]  pData           データのポインタ
 * @param[in]   pSession        製造用セッションのポインタ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess           データの取得に成功しました。
 * @pre         指定したセッションはオープン状態である必要があります。
 */
Result GetHostPdcDeviceId( uint16_t* pData, ManufactureSession* pSession ) NN_NOEXCEPT;

} // pd
} // usb
} // nn

