﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   USB::PD ライブラリの API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/usb/pd/usb_PdResult.h>
#include <nn/usb/pd/usb_PdType.h>

namespace nn {
namespace usb {
namespace pd {

/**
 * @brief セッション用パラメータ
 */
struct SessionCore
{
    void* _handle;
    nn::os::SystemEventType* _pEvent;
};
struct Session : public SessionCore
{
};

/**
 * @brief USB::PD ライブラリを初期化します。
 *
 * @post        ライブラリは初期化済の状態になります。
 * @details     USB::PD ライブラリを初期化します。
 *              本関数の呼び出し回数はライブラリ内部でカウントされます。
 */
void Initialize() NN_NOEXCEPT;

/**
 * @brief USB::PD ライブラリを終了します。
 *
 * @post        ライブラリは初期化前の状態になります。
 * @details     USB::PD ライブラリを終了します。
 *              ライブラリ初期化を行った回数分、本関数が呼ばれた場合にのみ、
 *              ライブラリが初期化前の状態になります。
 */
void Finalize() NN_NOEXCEPT;

/**
 * @brief セッションをオープンします。
 *
 * @param[out]  pOutSession セッション用パラメータ
 * @pre         ライブラリは初期化済の状態である必要があります。
 * @post        指定したセッションがオープン状態になります。
 */
void OpenSession( Session* pOutSession ) NN_NOEXCEPT;

/**
 * @brief セッションをクローズします。
 *
 * @param[in]   pSession セッション用パラメータ
 * @pre         ライブラリは初期化済の状態である必要があります。
 * @post        指定したセッションがクローズ状態になります。
 * @details     オープンされていないセッションに対して実行した場合は何もしません。
 *              送信予約されているがまだ送信されていないデータについては失われる可能性があります。
 */
void CloseSession( Session* pSession ) NN_NOEXCEPT;

//! @name 通知とイベントの紐付けと紐付解除
//! @{
/**
 * @brief       通知とイベントを紐付けます。
 *
 * @param[in]   pEvent           通知が発生した際にシグナル化するイベントを指定します。
 * @param[in]   pSession         セッション用パラメータ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess        成功しました。
 * @retval      ResultAlreadyBound   session で指定した GPIO パッドがすでに特定イベントと紐付けされています。
 */
Result BindNoticeEvent(nn::os::SystemEventType* pEvent, Session* pSession) NN_NOEXCEPT;

/**
 * @brief       通知とイベントの紐付を解除します。
 *
 * @param[in]   pSession         セッション用パラメータ
 *
 * @pre         BindNoticeEvent で SystemEvent が session と紐付けされている必要があります。@n
 *              session に紐付いている SystemEvent があるかどうかは ASSERT でチェックされます。@n
 *              紐付いている SystemEvent を待機しているスレッドがいない状態で実行してください。
 * @post        pSession と紐付けられた SystemEvent が紐付解除状態となります。@n
 *              また、紐付いていた SystemEvent は破棄され、未初期化状態となります。
 * @details     pSession と紐付けられた SystemEvent が紐付解除状態となります。
 */
void UnbindNoticeEvent(Session* pSession) NN_NOEXCEPT;

/**
 * @brief ステータスを取得します。
 *
 * @param[in]   pStatus      ステータスのポインタ
 * @param[in]   pSession     セッション用パラメータ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess             ステータス取得に成功しました。
 * @pre         指定したセッションはオープン状態である必要があります。
 */
Result GetStatus( Status* pStatus, Session* pSession ) NN_NOEXCEPT;

/**
 * @brief 通知フラグを取得します。
 *
 * @param[in]   pStatus      通知フラグのポインタ
 * @param[in]   pSession     セッション用パラメータ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess             通知フラグ取得に成功しました。
 * @pre         指定したセッションはオープン状態である必要があります。
 * @post        読み出した通知フラグはクリアされます。
 */
Result GetNotice( Notice* pNotice, Session* pSession ) NN_NOEXCEPT;

/**
 * @brief 電力設定要求通知を有効にします。
 *
 * @param[in]   pSession     セッション用パラメータ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess             ステータス取得に成功しました。
 * @pre         指定したセッションはオープン状態である必要があります。
 * @details     特定プロセスで有効にした状態では全てのポートで ResultAlreadyEnabled
 *              が返るようになります。
*/
Result EnablePowerRequestNotice( Session* pSession ) NN_NOEXCEPT;

/**
 * @brief 電力設定要求通知を無効にします。
 *
 * @param[in]   pSession     セッション用パラメータ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess             ステータス取得に成功しました。
 * @pre         指定したセッションはオープン状態である必要があります。
*/
Result DisablePowerRequestNotice( Session* pSession ) NN_NOEXCEPT;

/**
 * @brief 電力設定要求に対して返信します。
 *
 * @param[in]   pSession     セッション用パラメータ
 * @param[in]   isSuccess    処理の正否
 * @return      処理の結果を返します。
 * @retval      ResultSuccess             ステータス取得に成功しました。
 * @pre         指定したセッションはオープン状態である必要があります。
 * @details     EnablePowerRequestNotice で有効にしたポートしか成功しません。
*/
Result ReplyPowerRequest( Session* pSession, bool isSuccess ) NN_NOEXCEPT;

} // pd
} // usb
} // nn

