﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   USB::PD ライブラリの API 宣言。
 */

#pragma once

#include <nn/usb/pd/usb_PdLib.h>
#include <nn/usb/pd/usb_PdResult.h>
#include <nn/usb/pd/usb_PdCradleType.h>
#include <nn/usb/pd/usb_PdCradleVdm.h>

namespace nn {
namespace usb {
namespace pd {

/**
 * @brief セッション用パラメータ
 */
struct CradleSession : public SessionCore
{
};

/**
 * @brief USB::PD クレードル用ライブラリを初期化します。
 *
 * @post        ライブラリは初期化済の状態になります。
 * @details     USB::PD:C ライブラリを初期化します。
 *              本関数の呼び出し回数はライブラリ内部でカウントされます。
 */
void InitializeCradle() NN_NOEXCEPT;

/**
 * @brief USB::PD クレードル用ライブラリを終了します。
 *
 * @post        ライブラリは初期化前の状態になります。
 * @details     USB::PD:C ライブラリを終了します。
 *              ライブラリ初期化を行った回数分、本関数が呼ばれた場合にのみ、
 *              ライブラリが初期化前の状態になります。
 */
void FinalizeCradle() NN_NOEXCEPT;

/**
 * @brief クレードル用セッションをオープンします。
 *
 * @param[out]  pOutSession セッション用パラメータ
 * @pre         ライブラリは初期化済の状態である必要があります。
 * @post        指定したセッションがオープン状態になります。
 */
void OpenCradleSession( CradleSession* pOutSession ) NN_NOEXCEPT;

/**
 * @brief クレードル用セッションをクローズします。
 *
 * @param[in]   pSession セッション用パラメータ
 * @pre         ライブラリは初期化済の状態である必要があります。
 * @post        指定したセッションがクローズ状態になります。
 * @details     オープンされていないセッションに対して実行した場合は何もしません。
 *              送信予約されているがまだ送信されていないデータについては失われる可能性があります。
 */
void CloseCradleSession( CradleSession* pSession ) NN_NOEXCEPT;

/**
 * @brief クレードル VDO を設定します。
 *
 * @param[in]   pSession                クレードル用セッションのポインタ
 * @param[in]   pVdo                    VDO のポインタ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess           VDO の設定に成功しました。
 *              ResultInactive          クレードルとの通信が Active 状態でない。
 *              ResultInvalidCommand    不正なコマンドの発行。
 *              ResultInvalidDevice     クレードル／リレーボックス以外のデバイス。
 *              ResultTimeout           クレードルからの Reply 待ちがタイムアウト。
 *              ResultErrorReply        クレードルから失敗の Reply が返ってきた（UsbHubVBusDetect のみ）。
 * @pre         指定したセッションはオープン状態である必要があります。
 */
Result SetCradleVdo( CradleSession* pSession, Vdo* pVdo, CradleVdmCommand command ) NN_NOEXCEPT;

/**
 * @brief クレードル VDO を取得します。
 *
 * @param[out]  pVdo            VDO のポインタ
 * @param[in]   pSession        クレードル用セッションのポインタ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess           VDO の設定に成功しました。
 *              ResultInactive          クレードルとの通信が Active 状態でない。
 *              ResultInvalidCommand    不正なコマンドの発行。
 *              ResultInvalidDevice     クレードル／リレーボックス以外のデバイス。
 *              ResultTimeout           クレードルからの Reply 待ちがタイムアウト。
 * @pre         指定したセッションはオープン状態である必要があります。
 */
Result GetCradleVdo( Vdo* pVdo, CradleSession* pSession, CradleVdmCommand command ) NN_NOEXCEPT;

/**
 * @brief クレードル USB HUB をリセットします。
 *
 * @param[in]   pSession        クレードル用セッションのポインタ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess           リセットに成功しました。
 *              ResultInactive          クレードルとの通信が Active 状態でない。
 *              ResultInvalidDevice     クレードル／リレーボックス以外のデバイス。
 *              ResultTimeout           クレードルからの Reply 待ちがタイムアウト。
 * @pre         指定したセッションはオープン状態である必要があります。
 */
Result ResetCradleUsbHub( CradleSession* pSession ) NN_NOEXCEPT;

/**
 * @brief クレードルとの接続異常発生からの復旧を許可します。
 *
 * @param[out]  pIsSuspended    上記の復旧処理が中断中かどうかのポインタ。
 *                              true の場合は本関数呼び出し後に復旧処理が再開されます。
 *                              false の場合は中断中の復旧処理が無いため再開されません。
 * @param[in]   pSession        クレードル用セッションのポインタ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess           許可に成功しました。
 *              ResultInvalidSession    既に他のセッションに制御されている。
 * @pre         指定したセッションはオープン状態である必要があります。
 * @details     症状：
 *                  本体とクレードルの VDM 送信が衝突することで PDC にほぼ同時に Hard Reset が
 *                  発生することがあり、中途半端に PDC が初期化されます。
 *                  後続の VDM 通信と USB 通信は継続可能なものの、Date Role が逆、DP 出力不可、HPD 入力不可、
 *                  15V / 0.5A 給電に切り替わる等の異常が発生します。
 *              復旧処理：
 *                  通常時はクレードルとの接続を切断するコマンドが発行されることによって
 *                  自動で再接続処理が行われて復旧しますが、間欠起動中にこの再接続処理を行った場合、
 *                  FullAwake してクレードル内の DP2HDMI が電源 ON されることになり、
 *                  これに TV が反応して HDMI 入力を切り替えたりすることがあります。
 *              対応手順：
 *                  間欠起動中は DisableCradleRecovery() / EnableCradleRecovery()
 *                  で囲って復旧処理を遅延させることによって上記の問題を回避できます。
 *                  pIsSuspended が true を返す時の中断中の復旧処理が再開している場合、
 *                  クレードルと再接続された ActiveNotice が再通知されるまで
 *                  クレードル内の DP2HDMI へアクセスしないで下さい。
 */
Result EnableCradleRecovery( bool* pIsSuspended, CradleSession* pSession ) NN_NOEXCEPT;

/**
 * @brief クレードルとの接続異常発生からの復旧処理を禁止します。
 *
 * @param[out]  pIsSuspended    上記の復旧処理が中断中かどうかのポインタ。
 *                              true、false ともに復旧処理は再開されません。
 * @param[in]   pSession        クレードル用セッションのポインタ
 * @return      処理の結果を返します。
 * @retval      ResultSuccess           禁止に成功しました。
 *              ResultInvalidSession    既に他のセッションに制御されている。
 * @pre         指定したセッションはオープン状態である必要があります。
 * @details     EnableCradleRecovery() を参照。
 */
Result DisableCradleRecovery( bool* pIsSuspended, CradleSession* pSession ) NN_NOEXCEPT;

} // pd
} // usb
} // nn

