﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/result/result_ResultBase.h>
#include <nn/ncm/ncm_SystemContentMetaId.h>
#include <nn/spl/spl_Types.h>
#include <nn/updater/updater_Result.h>

namespace nn { namespace updater {

enum class TargetBootMode
{
    Normal,
    Safe,
};

enum class BootImageUpdateType
{
    Original,
    OdinMariko,
};

struct VerifyingRequiredFlags
{
    bool normalRequired;
    bool safeRequired;
};


/**
*   @brief      BootImages の検証が必要ということをシステムに保存します。
*   @details    MarkVerified を呼ぶまで、CheckVerifyingRequired で取得できるフラグが true となります。
*
*   @param[in]  mode       対象のモード
*   @param[in]  workBuffer 更新処理に利用するワークバッファ
*   @param[in]  workSize   ワークバッファのサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                      成功しました。
*   @retval     ResultOutOfWorkBuffer              指定したワークバッファのサイズが足りません。
*   @retval     ResultInvalidAlignmentOfWorkBuffer ワークバッファのアライメントが正しくありません。
*   @retval     ResultInvalidBootImagePackage      BootImagePackage のフォーマットが正しくありません。
*/
Result MarkVerifyingRequired(TargetBootMode mode, void* workBuffer, size_t workSize) NN_NOEXCEPT;

/**
*   @brief      BootImages の更新が正しく終了したというフラグをシステムに保存します。
*   @details    UpdateBootImagesFromPackage 直後か、
*               VerifyBootImages を行い、Success が返った場合に呼ぶことを推奨します。
*               以後、CheckVerifyingRequired で取得できるフラグが false となります。
*
*   @param[in]  mode       対象のモード
*   @param[in]  workBuffer 更新処理に利用するワークバッファ
*   @param[in]  workSize   ワークバッファのサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                      成功しました。
*   @retval     ResultOutOfWorkBuffer              指定したワークバッファのサイズが足りません。
*   @retval     ResultInvalidAlignmentOfWorkBuffer ワークバッファのアライメントが正しくありません。
*   @retval     ResultInvalidBootImagePackage      BootImagePackage のフォーマットが正しくありません。
*/
Result MarkVerified(TargetBootMode mode, void* workBuffer, size_t workSize) NN_NOEXCEPT;

/**
*   @brief      BootImages の検証が必要かどうかを取得します。
*   @details    更新が必要な場合、pOutVerifyingRequired が true となります。
*               その場合、VerifyBootImages を呼び、壊れていないかを確認してください。
*               セーフモード起動中はこのフラグの値にかかわらず
*               VerifyBootImages や UpdateBootImagesFromPackage を行うべきです。
*
*   @param[out] pOutFlags      VerifyBootImage が必要かどうか
*   @param[in]  workBuffer     更新処理に利用するワークバッファ
*   @param[in]  workSize       ワークバッファのサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                      成功しました。
*   @retval     ResultOutOfWorkBuffer              指定したワークバッファのサイズが足りません。
*   @retval     ResultInvalidAlignmentOfWorkBuffer ワークバッファのアライメントが正しくありません。
*   @retval     ResultInvalidBootImagePackage      BootImagePackage のフォーマットが正しくありません。
*/
Result CheckVerifyingRequired(VerifyingRequiredFlags* pOutFlags, void* workBuffer, size_t workSize) NN_NOEXCEPT;

/**
*   @brief      指定した BootImagePackage を用いて、BootImages を更新します。
*   @details    更新ロジックは下記ページに従います。
*               http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=166060264
*               この API を複数個所で呼ぶようにすることで、安全な更新ができるようになります。
*
*   @param[in]  dataId                更新に用いる BootImagePackage
*   @param[in]  mode                  対象のモード
*   @param[in]  workBuffer            更新処理に利用するワークバッファ
*   @param[in]  workSize              ワークバッファのサイズ
*   @param[in]  bootImageUpdateType   更新対象のシステムの種類
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                      成功しました。
*   @retval     ResultBootImageNotFound            指定した BootImagePackage が見つかりません。
*   @retval     ResultOutOfWorkBuffer              指定したワークバッファのサイズが足りません。
*   @retval     ResultInvalidAlignmentOfWorkBuffer ワークバッファのアライメントが正しくありません。
*   @retval     ResultInvalidBootImagePackage      BootImagePackage のフォーマットが正しくありません。
*/
Result UpdateBootImagesFromPackage(ncm::SystemDataId dataId, TargetBootMode mode, void* workBuffer, size_t workSize, BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT;

/**
*   @brief      指定した BootImagePackage と、現在書き込まれている BootImages を比較し、正しいか検証を行います。
*   @details    齟齬がなければ Success を返します。
*               齟齬があった場合、ResultWrittenDataMissMatch を返します。
*               その場合、UpdateBootImagesFromPackage で BootImage を更新することを期待します。
*
*   @param[in]  dataId                比較に用いる BootImagePackage
*   @param[in]  mode                  対象のモード
*   @param[in]  workBuffer            比較処理に利用するワークバッファ
*   @param[in]  workSize              ワークバッファのサイズ
*   @param[in]  bootImageUpdateType   更新対象のシステムの種類
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                      正しく書き込まれていました。
*   @retval     ResultRequiresUpdateBootImages     比較結果が、不一致でした。
*   @retval     ResultBootImageNotFound            指定した BootImagePackage が見つかりません。
*   @retval     ResultOutOfWorkBuffer              指定したワークバッファのサイズが足りません。
*   @retval     ResultInvalidAlignmentOfWorkBuffer ワークバッファのアライメントが正しくありません。
*   @retval     ResultInvalidBootImagePackage      BootImagePackage のフォーマットが正しくありません。
*/
Result VerifyBootImages(ncm::SystemDataId dataId, TargetBootMode mode, void* workBuffer, size_t workSize, BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT;

/**
*   @brief      更新に利用する BootImagePackage の ID を取得します。
*   @details    ncm のコンテントメタデータベースから BootImagePackage を取得し、ID を返します。
*               あらかじめ、ncm::Initialize() を呼んでおく必要があります。
*
*   @param[out] pOutId      利用すべき BootImagePackage の ID
*   @param[in]  mode        対象のモード
*   @param[in]  workBuffer  ワークバッファ
*   @param[in]  workSize    ワークバッファのサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                      処理が成功しました
*   @retval     ResultBootImageNotFound            BootImagePackage が存在しませんでした。
*/
Result GetBootImagePackageId(ncm::SystemDataId *pOutId, TargetBootMode mode, void* workBuffer, size_t workSize) NN_NOEXCEPT;

/**
*   @brief      BootImages の検証と、必要なら更新を行うユーティリティ関数です。
*   @details    boot プロセスで利用することを想定しています。
*               検証フラグがたっていた場合、詳細な検証と、必要なら更新を行います。
*               検証フラグがたっている場合は、内部で ncm::Initialize と ncm::Finalize を呼びます。
*               更新を行った場合、pOutNormalUpdated / pOutSafeUpdated が true となります。
*               現在実行中のモードの更新があった場合、再起動を行ってください。
*               BootImagePackage/BootImagePackageSafe が見つからない場合も、本 API は Success を返します。
*
*   @param[out] pOutNormalUpdated     更新を行った場合は true。行わなかったら false
*   @param[out] pOutSafeUpdated       更新を行った場合は true。行わなかったら false
*   @param[in]  workBuffer            比較処理に利用するワークバッファ
*   @param[in]  workSize              ワークバッファのサイズ
*   @param[in]  bootImageUpdateType   更新対象のシステムの種類
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                      処理が成功しました
*   @retval     ResultOutOfWorkBuffer              指定したワークバッファのサイズが足りません。
*   @retval     ResultInvalidAlignmentOfWorkBuffer ワークバッファのアライメントが正しくありません。
*   @retval     ResultInvalidBootImagePackage      BootImagePackage のフォーマットが正しくありません。
*/
Result VerifyBootImagesAndUpdateIfNeeded(bool* pOutNormalUpdated, bool* pOutSafeUpdated, void* workBuffer, size_t workSize, BootImageUpdateType bootImageUpdateType) NN_NOEXCEPT;

/**
*   @brief      settings から取得した boot_image_update_type から、updater::BootImageUpdateType を取得します。
*
*   @param[in]  socType    settings から取得した boot_image_update_type
*
*   @return     updater::BootImageUpdateType が返ります。
*/
BootImageUpdateType GetBootImageUpdateType(int bootImageUpdateType) NN_NOEXCEPT;

/**
*   @brief      spl から取得した spl::HardwareType から、updater::BootImageUpdateType を取得します。
*
*   @param[in]  hardwareType    spl から取得した spl::HardwareType
*
*   @return     updater::BootImageUpdateType が返ります。
*/
BootImageUpdateType GetBootImageUpdateType(spl::HardwareType hardwareType) NN_NOEXCEPT;


}}

