﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/time/detail/time_CommonDetail.h>

#include <nn/time/time_LocationName.h>
#include <nn/time/time_TimeZoneRuleVersion.h>
#include <siglo/Include/time_TimeZoneResourcesType.h>

namespace nn
{
namespace timesrv
{
namespace detail
{
namespace fs
{

/**
 * @brief   タイムゾーンバイナリを扱うクラス
 */
class TimeZoneBinaryManager
{
public:

    //!< マウント
    static Result MountFs() NN_NOEXCEPT;

    /**
     * @brief   タイムゾーンバイナリ読み込み
     * @param[out]  pOutTimeZoneBinary  読み込みバッファのポインタを入れるポインタ
     * @param[out]  pOutSize            読み込んだサイズ
     * @param[in]   locationName        地域名
     */
    static Result ReadTimeZoneBinary(
        char** pOutTimeZoneBinary, size_t* pOutSize, const nn::time::LocationName locationName) NN_NOEXCEPT;

    //!< 地域名のトータル数取得
    static int32_t GetLocationNameCount() NN_NOEXCEPT;

    //!< 指定の LocationName が存在するかどうか
    static bool IsValidLocationName(const nn::time::LocationName& locationName) NN_NOEXCEPT;

    /**
     * @brief   地域名リストの取得
     * @param[out]  pOutCount               地域名の出力数
     * @param[out]  pOutLocationNameList    地域名の出力リスト
     * @param[in]   locationNameListCount   pOutLocationNameList 要素数
     * @param[in]   offset                  読込オフセット
     */
    static Result LoadLocationNameList(
        int32_t* pOutCount,
        nn::time::LocationName* pOutLocationNameList,
        size_t locationNameListCount,
        int32_t offset) NN_NOEXCEPT;

    //!< タイムゾーンバイナリのバージョン取得
    static Result GetTimeZoneRuleVersion(nn::time::TimeZoneRuleVersion* pOut) NN_NOEXCEPT;

private:

    static char m_WorkBuffer[1024 * 20]; // タイムゾーンバイナリ、タイムゾーンリストの読み込み用バッファ

    //!< ソースリンクのゾーンバイナリを利用するかどうか
    static constexpr bool HasSourceLinkTimeZoneBinary() NN_NOEXCEPT;

    //!< タイムゾーンバイナリ読み込み
    static Result ReadTimeZoneListData(
        size_t* pOutSize,
        char* pOutBuffer,
        size_t outBufferSize) NN_NOEXCEPT;

    //!< CRLF区切りで地域名が記述されたバッファをパースして地域名のトータル数を取得(データタイトル時の利用を想定)
    static void CountLocationNameFromListBuffer(
        int32_t* pOutCount,
        const char* pListBuffer,
        size_t listBufferSize) NN_NOEXCEPT;

    //!< ソースリンクの nne::tz::BinaryResource の先頭ポインタから地域名リストを抽出
    static Result RetrieveLocationNameList(
        int32_t* pOutCount,
        nn::time::LocationName* pOutLocationNameList,
        size_t count,
        int32_t offset,
        const nne::tz::BinaryResource* pBinaryResource,
        size_t binaryResourceCount) NN_NOEXCEPT;

protected:
    // テスト利用のため protected に

    //!< CRLF区切りで地域名が記述されたバッファをパースして地域名リストを抽出(データタイトル時の利用を想定)
    static Result RetrieveLocationNameList(
        int32_t* pOutCount,
        nn::time::LocationName* pOutLocationNameList,
        size_t count,
        int32_t offset,
        const char* pListBuffer,
        size_t listBufferSize) NN_NOEXCEPT;
};

}
}
}
}
