﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "swkbd_Types.h"
#include "swkbd_AppearArg.h"

/**
 * @file
 * @brief   ソフトウェアキーボードで使用される型や定数の宣言(非公開)
 */

namespace nn { namespace swkbd {


// メジャーバージョン ( 1 ... 2.0.0 )
const uint32_t c_MajorVersion_0 = 0;
// 2.0.0 NUP 向け
const uint32_t c_MajorVersion_1 = 1;
// 3.0.0 NUP 向け
// (メジャーバージョン 2 は欠番)
const uint32_t c_MajorVersion_3 = 3;
// 4.0.0 NUP 向け
const uint32_t c_MajorVersion_4 = 4;
// 5.0.0 NUP 向け
const uint32_t c_MajorVersion_5 = 5;

// 1 ... 38303 まで
const uint32_t c_Version_0_1 = 1;
// 2 ... 38304 ～ 39367 : Storage 周りの修正を加えたバージョン
const uint32_t c_Version_0_2 = 2;
// 3 ... 39368 ～ : 不要な構造体内変数を除いたバージョン
const uint32_t c_Version_0_3 = 3;
// 4 ... 39951 ～ : テキストチェック用変数を内部化したバージョン
const uint32_t c_Version_0_4 = 4;
// 5 ... 41853 ～ : 内部変数をなくしたバージョン(プリイン Fix バージョン)
const uint32_t c_Version_0_5 = 5;
// 6 ... 2.0.0 向け
const uint32_t c_Version_1_6 = 6 + c_MajorVersion_1 * 0x10000;
// 7 ... 3.0.0 向け
// (メジャーバージョン 2 は欠番)
const uint32_t c_Version_3_7 = 7 + c_MajorVersion_3 * 0x10000;
// 8 ... 4.0.0 向け
const uint32_t c_Version_4_8 = 8 + c_MajorVersion_4 * 0x10000;
// 9 ... 5.0.x 向け
const uint32_t c_Version_5_9 = 9 + c_MajorVersion_5 * 0x10000;

// 現在の swkbd のバージョン
const uint32_t c_CurrentVersion = c_Version_5_9;
const uint16_t c_CurrentMajorVersion = c_CurrentVersion >> 16;
const uint16_t c_CurrentMinorVersion = c_CurrentVersion & 0xFFFF;


//! UTF-16 時の1文字あたりのバイト数
static const int32_t c_Utf16ByteSize = 2;
//! UTF-8 時の1文字あたりの最大バイト数( 絵文字を考慮した最大バイト数 )
static const int32_t c_Utf8ByteSize = 4;

//! 最大文字バッファサイズ。最大入力数 + 1(null終端用)
static const int32_t c_FixedStringLength = TextMaxLength + 1;
//! ( TextMaxLength + 1 ) * 2
static const int32_t c_FixedStringUtf16BufSize = c_FixedStringLength * c_Utf16ByteSize;
//! ( TextMaxLength + 1 ) * 4
static const int32_t c_FixedStringUtf8BufSize = c_FixedStringLength * c_Utf8ByteSize;


//--------------------------------------------------------------------------
/**
 * BG 起動時にソフトウェアキーボードに送るコマンドの種類です。
 */
enum RequestCommand : uint32_t
{
    RequestCommand_Appear,                  //!< ソフトウェアキーボードを表示します。
    RequestCommand_Disappear,               //!< ソフトウェアキーボードを非表示にします。
    RequestCommand_SetStringInfo,           //!< 文字列情報を設定します。
    RequestCommand_SetCursorPos,            //!< カーソル位置を指定します。
    RequestCommand_Finalize,                //!< ソフトウェアキーボードを終了します。
    RequestCommand_SetVolume,               //!< 全体音量を設定します。
    RequestCommand_SetUserWordInfo,         //!< ユーザー辞書を設定します。
    RequestCommand_SetCustomizeDic,         //!< カスタマイズ辞書を設定します。
    RequestCommand_SetUtf8Mode,             //!< ソフトウェアキーボードが返す文字列を UTF-8 にするかどうかを設定します。
    RequestCommand_UnsetCustomizeDic,       //!< カスタマイズ辞書の設定を解除します。
    RequestCommand_SetKeyboardStrorageInfo, //!< １つのストレージに大抵の情報が入ったコマンドを送信します。
    RequestCommand_Max                      //!< 列挙子の総数です。
};

//--------------------------------------------------------------------------
/**
 * ソフトウェアキーボードの終了理由の種類です。
 */
enum CloseResult : uint32_t
{
    CloseResult_Enter,  //!< 確定終了しました
    CloseResult_Cancel, //!< キャンセルが押されました
    CloseResult_Max     //!< 列挙子の総数です。
};


//--------------------------------------------------------------------------
/**
 * 起動パラメータで利用する WorkBuf の中身です。
 * 現状以下のような中身になっています。
 * 0  ... 3  (byte) | 初期文字列バッファオフセット値
 * 4  ... 7  (byte) | 初期文字数
 * 8  ... 11 (byte) | ユーザー辞書バッファオフセット値
 * 12 ... 15 (byte) | ユーザー辞書登録単語数
 * 16 ...           | 初期文字列 + ユーザ辞書
 */
struct WorkBufHeaderV2
{
    int32_t initialStringOffset; //!< 初期文字列バッファオフセット値
    int32_t initialStringLength; //!< 初期文字列の文字数
    int32_t userDictionaryOffset; //!< ユーザー辞書バッファオフセット値
    int32_t userDictionaryNum; //!< ユーザー辞書登録単語数
};

//--------------------------------------------------------------------------
/**
 * swkbd workBuf のヘッダ構造 version 3
 * c_Version = 3 以降に対応
 */
struct WorkBufHeaderV3
{
    int32_t initialStringOffset; //!< 初期文字列バッファオフセット値
    int32_t initialStringLength; //!< 初期文字列の文字数
    int32_t userDictionaryOffset; //!< ユーザー辞書バッファオフセット値
    int32_t userDictionaryNum; //!< ユーザー辞書登録単語数
    bool    isUseTextCheck; //!< ソフトウェアキーボードの入力確定後に、一度アプリ側で入力結果文字列を確認するかどうか
};


//--------------------------------------------------------------------------
/**
 * @brief   ソフトウェアキーボードの設定引数を担う構造体です。
 *          2.0.0 NUP までの構造体。
 */
struct KeyboardConfigV1
{
    //! @brief ソフトウェアキーボードの盤面の種類です。@n
    //!        nn::swkbd::KeyboardMode を元に、表示したい盤面の種類を指定してください。
    KeyboardMode      keyboardMode;
    //! @brief OKボタンの文字列です。@n
    //!        最大で nn::swkbd::OkTextMaxLength 文字までです。@n
    //!        文字の表示のつぶれ方は実際の表示を見て確認してください。@n
    //!        現段階では未対応です。
    char16_t          okText[ OkTextMaxLength + 1 ];
    //! @brief 数字キーボードモードに追加できる左側の文字です。@n
    //!        nn::swkbd::KeyboardMode が KeyboardMode_Numeric の時のみ有効です。@n
    //!        キーを使用しない場合は 0 を指定してください。
    char16_t          leftOptionalSymbolKey;
    //! @brief 数字キーボードモードに追加できる右側の文字です。@n
    //!        nn::swkbd::KeyboardMode が KeyboardMode_Numeric の時のみ有効です。@n
    //!        キーを使用しない場合は 0 を指定してください。
    char16_t          rightOptionalSymbolKey;
    //! @brief 予測変換機能を使用するか否かを指定できます。@n
    //!        使用する場合は true を設定ください。
    bool              isPredictionEnabled;
    //! @brief 入力禁止文字フラグです。@n
    //!        入力禁止したい文字（キー）群を表すビットフラグ（nn::swkbd::InvalidChar の各値を指定）を指定してください。
    uint32_t          invalidCharFlag;
    //! @brief ソフトウェアキーボード起動時の、プレビュー欄に表示されるテキストカーソルの初期位置です。@n
    //!        テキストカーソルを先頭に合わせるか、末尾に合わせるかを指定できます。
    InitialCursorPos  initialCursorPos;
    //! @brief ヘッダーテキストに表示する文字列です。@n
    //!        何を入力するべきかを簡潔に表示する文字列を指定してください。@n
    //!        プレビュー欄が１行の時のみ、プレビュー欄上部に表示されます。@n
    //!        最大 nn::swkbd::HeaderTextMaxLength 文字が指定可能です。
    char16_t          headerText[ HeaderTextMaxLength + 1 ];
    //! @brief サブテキストに表示する文字列です。@n
    //!        ヘッダーテキストに書かれた内容を補足する文字列を指定してください。@n
    //!        プレビュー欄が１行の時のみ、ヘッダーテキストより下部に表示されます。@n
    //!        最大 nn::swkbd::SubTextMaxLength 文字が指定可能です。
    char16_t          subText[ SubTextMaxLength + 1 ];
    //! @brief ガイド文字列です。
    //!        プレビュー欄に文字が１つも入力されていない時に表示されます。@n
    //!        未入力時に入力の目的や、入力例などを表示する文字列を指定してください。@n
    //!        また、複数行の時にも表示したい説明がある場合はこちらを指定してください。@n
    //!        最大 nn::swkbd::GuideTextMaxLength 文字が指定可能です。
    char16_t          guideText[ GuideTextMaxLength + 1 ];
    //! @brief 入力テキストとして有効な最大文字数です。@n
    //!        プレビュー欄の右下に「入力中の文字数/最大文字数」が表示されます。@n
    //!        最大 nn::swkbd::TextMaxLength まで指定可能です。
    int32_t           textMaxLength;
    //! @brief 入力テキストとして有効な最小文字数です。@n
    //!        プレビュー欄の右下に「入力中の文字数/最大文字数」が表示されます。@n
    //!        最大 nn::swkbd::TextMaxLength まで指定可能です。
    int32_t           textMinLength;
    //! @brief パスワードキーボードモードのときの入力文字隠蔽モードです。@n
    //!        指定していない場合、入力した文字がそのまま表示されます。
    PasswordMode      passwordMode;
    //! @brief プレビュー欄の表示の種類です。@n
    //!        プレビュー欄を1行の見た目で表示するか、複数行の見た目で表示するかを設定できます。
    InputFormMode     inputFormMode;
    //! @brief 改行を有効にするかどうかを指定できます。@n
    //!        true にすると、改行ボタンが押せるようになります。@n
    //!       inputFromMode が InputFormMode_MultiLine でなければ、改行は有効になりません。
    bool              isUseNewLine;
    //! @brief ソフトウェアキーボード終了時に返る文字列を UTF-8 にするかどうかを指定できます。@n
    //!        true にすると、ソフトウェアキーボード終了時に返る文字列が UTF-8 になります。@n
    //!        false にすると、ソフトウェアキーボード終了時に返る文字列が UTF-16 になります。
    bool              isUseUtf8;
    //! @brief ソフトウェアキーボード開始時に、現在表示中の画面をぼかすかどうかを指定できます。@n
    //!        ソフトウェアキーボードの開始時、起動したアプリ側の画面がキャプチャされ、
    //!        キャプチャ結果の画像がソフトウェアキーボードの背景に描画されます。@n
    //!        true にすると、キャプチャ結果の画像がぼかした状態で描画されます。
    bool              isUseBlurBackground;

    // ワークバッファの先頭から起動時に入力されている文字列までのオフセットです。@n
    // この値を書き換えないでください。
    int32_t           _initialStringOffset;
    // 起動時に入力されている文字列数です。@n
    // この値を書き換えないでください。
    int32_t           _initialStringLength;
    // ワークバッファの先頭からユーザ登録単語までのオフセットです。@n
    // この値を書き換えないでください。
    int32_t           _userDictionaryOffset;
    // ワークバッファの先頭からユーザ登録単語数です。@n
    // この値を書き換えないでください。
    int32_t           _userDictionaryNum;
    // ソフトウェアキーボードの入力確定後に、一度アプリ側で入力結果文字列を確認するかどうかの判定フラグです。@n
    // この値を書き換えないでください。
    bool              _isUseTextCheck;
    // 入力結果の文字列を確認するコールバック関数です。@n
    // この値を書き換えないでください。
    TextCheckCallback _textCheckCallback;

    /**
     * @brief version 1 から 3.0.0 NUP に向けた更新関数
     *        private にするたび内部的に更新関数を追加する
     * @param[in] config 更新先のファイル
     */
    void UpgradeTo( KeyboardConfig& config )
    {
        config.keyboardMode = keyboardMode;
        std::memcpy( config.okText, okText,
            ( sizeof( char16_t ) * ( OkTextMaxLength + 1 ) ) );
        config.leftOptionalSymbolKey = leftOptionalSymbolKey;
        config.rightOptionalSymbolKey = rightOptionalSymbolKey;
        config.isPredictionEnabled = isPredictionEnabled;
        config.invalidCharFlag = invalidCharFlag;
        config.initialCursorPos = initialCursorPos;
        std::memcpy( config.headerText, headerText,
            ( sizeof( char16_t ) * ( HeaderTextMaxLength + 1 ) ) );
        std::memcpy( config.subText, subText,
            ( sizeof( char16_t ) * ( SubTextMaxLength + 1 ) ) );
        std::memcpy( config.guideText, guideText,
            ( sizeof( char16_t ) * ( GuideTextMaxLength + 1 ) ) );
        config.textMaxLength = textMaxLength;
        config.textMinLength = textMinLength;
        config.passwordMode = passwordMode;
        config.inputFormMode = inputFormMode;
        config.isUseNewLine = isUseNewLine;
        config.isUseUtf8 = isUseUtf8;
        config.isUseBlurBackground = isUseBlurBackground;
        config._initialStringOffset = _initialStringOffset;
        config._initialStringLength = _initialStringLength;
        config._userDictionaryOffset = _userDictionaryOffset;
        config._userDictionaryNum = _userDictionaryNum;
        config._isUseTextCheck = _isUseTextCheck;
        config._textCheckCallback = _textCheckCallback;
    }
};

//--------------------------------------------------------------------------
/**
 * @brief   ソフトウェアキーボードの起動引数を定義する構造体です。
 *          2.0.0 NUP までの構造体。
 */
struct ShowKeyboardArgV1
{
    //! @brief ソフトウェアキーボードの設定引数です。
    KeyboardConfigV1  keyboardConfig;

    //! @brief ソフトウェアキーボードに渡すワークバッファです。@n
    //!        nn::swkbd::SetInitialText() で設定した初期入力文字列や、
    //!        アプリ側で指定したユーザー辞書が格納されます。@n
    //!        nn::swkbd::GetRequiredWorkBufferSize() で取得できるバイトサイズ分確保したバッファを設定してください。@n
    //!        nn::os::MemoryPageSize でアライメントされたものでなければなりません。
    void* workBuf;
    //! @brief ソフトウェアキーボードに渡すワークバッファのバイトサイズです。@n
    //!        nn::swkbd::GetRequiredWorkBufferSize() で取得できるバイトサイズを指定してください。
    size_t workBufSize;

    //! @brief ソフトウェアキーボードの入力確定後に、アプリ側で入力結果文字列を確認する際に用いるワークバッファです。@n
    //!        nn::swkbd::GetRequiredTextCheckWorkBufferSize() で取得できるバイトサイズ分確保したバッファを設定してください。@n
    //!        アプリ側で確認する必要のない場合、設定する必要はありません。
    void* textCheckWorkBuf;
    //! @brief ソフトウェアキーボードの入力確定後に、アプリ側で入力結果文字列を確認する際に用いるワークバッファのバイトサイズです。@n
    //!        nn::swkbd::GetRequiredTextCheckWorkBufferSize() で取得できるバイトサイズを指定してください。
    size_t textCheckWorkBufSize;

    /**
     * @brief version 1 から 3.0.0 NUP に向けた更新関数
     *        private にするたび内部的に更新関数を追加する
     * @param[in] arg 更新先のファイル
     */
    void UpgradeTo( ShowKeyboardArg& arg )
    {
        keyboardConfig.UpgradeTo( arg.keyboardConfig );
        arg.workBuf = workBuf;
        arg.workBufSize = workBufSize;
        arg.textCheckWorkBuf = textCheckWorkBuf;
        arg.textCheckWorkBufSize = textCheckWorkBufSize;
    }
};


}} // namespace nn::swkbd
