﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       障害情報収集に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/settings/system/settings_Common.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace settings { namespace system {

/**
 * @brief       障害情報収集向けダーティーフラグの定義です。
 */
struct TelemetryDirtyFlag
{
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<0>
            LanguageCode;               //!< 言語設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<1>
            NetworkSettings;            //!< ネットワーク設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<2>
            RegionCode;                 //!< リージョンコード
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<3>
            LockScreenFlag;             //!< 誤操作防止画面の有無
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<4>
            BacklightSettings;          //!< 画面の明るさの設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<5>
            BluetoothDevicesSettings;   //!< Bluetooth デバイス設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<6>
            ExternalSteadyClockSourceId;    //!< ExternalSteadyClock のソース ID
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<7>
            UserSystemClockContext;     //!< UserSystemClock のコンテキスト
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<8>
            AccountSettings;            //!< アカウント設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<9>
            AudioVolume;                //!< 本体音量設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<10>
            EulaVersions;               //!< 本体 EULA の同意バージョン
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<11>
            ColorSetId;                 //!< カラーセット ID
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<12>
            ConsoleInformationUploadFlag;   //!< 本体情報アップロードフラグ
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<13>
            AutomaticApplicationDownloadFlag;   //!< 自動配信フラグ
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<14>
            NotificationSettings;       //!< 通知設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<15>
            AccountNotificationSettings;    //!< アカウント毎の通知設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<16>
            VibrationMasterVolume;      //!< 振動のマスターボリューム
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<17>
            TvSettings;                 //!< TV 設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<18>
            Edid;                       //!< EDID
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<19>
            AudioOutputMode;            //!< 本体オーディオ出力設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<20>
            ForceMute;                  //!< 強制ミュート設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<21>
            QuestFlag;                  //!< Quest メニュー起動フラグ
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<22>
            DataDeletionSettings;       //!< おすすめ削除の設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<23>
            DeviceTimeZoneLocationName; //!< デバイスの地域名
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<24>
            NetworkSystemClockContext;  //!< NetworkSystemClock のコンテキスト
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<25>
            UserSystemClockAutomaticCorrection; //!< UserSystemClock の自動補正
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<26>
            PrimaryAlbumStorage;        //!< 画面写真を保存するストレージ
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<27>
            Usb30EnableFlag;            //!< USB 3.0 が有効か否か
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<28>
            NfcEnableFlag;              //!< NFC が有効か否か
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<29>
            SleepSettings;              //!< スリープ設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<30>
            WirelessLanEnableFlag;      //!< 無線 LAN が有効か否か
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<31>
            InitialLaunchSettings;      //!< 初回起動設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<32>
            DeviceNickName;             //!< 本体の識別名
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<33>
            LdnChannel;                 //!< ローカル通信のチャンネル
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<34>
            BluetoothEnableFlag;        //!< Bluetooth が有効か否か
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<35>
            ShutdownRtcValue;           //!< シャットダウン時の RTC 値
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<36>
            ExternalSteadyClockInternalOffset;  //!< ExternalSteadyClock の内部オフセット
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<37>
            ErrorReportSharePermission; //!< エラーレポートを他社に提供してよいかの許可状態
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<38>
        ConsoleSixAxisSensorAccelerationBias;      //!< 本体 6 軸センサーの加速度バイアス設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<39>
        ConsoleSixAxisSensorAngularVelocityBias;   //!< 本体 6 軸センサーの角速度バイアス設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<38>
        ConsoleSixAxisSensorAccelerationGain;      //!< 本体 6 軸センサーの加速度ゲイン設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<39>
        ConsoleSixAxisSensorAngularVelocityGain;   //!< 本体 6 軸センサーの角速度ゲイン設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<40>
        ConsoleSixAxisSensorAngularVelocityTimeBias;   //!< 本体 6 軸センサーの角速度の時間バイアス設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<41>
        ConsoleSixAxisSensorAngularAcceleration;   //!< 本体 6 軸センサーの角加速度の係数設定
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<42>
            DeviceTimeZoneLocationUpdatedTime; //!< デバイス地域の更新時間
    typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag>::Flag<43>
            UserSystemClockAutomaticCorrectionUpdatedTime; //!< UserSystemClock の自動補正フラグの更新時間
};

/**
 * @brief       障害情報収集向けダーティーフラグ集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<128, TelemetryDirtyFlag> TelemetryDirtyFlagSet;

/**
 * @brief       障害情報収集向けダーティーフラグの変更通知をバインドします。
 *
 * @param[in]   pEvent                      イベントオブジェクトを指すポインタ
 * @param[in]   clearMode                   イベントオブジェクトのクリアモード
 *
 * @pre
 *              - pEvent != nullptr
 *              - *pEvent は未初期化状態
 * @post
 *              - *pEvent は初期化状態
 *              - *pEvent はシグナル状態
 */
void BindTelemetryDirtyFlagEvent(
    ::nn::os::SystemEventType* pEvent,
    ::nn::os::EventClearMode clearMode) NN_NOEXCEPT;

/**
 * @brief       障害情報収集向けダーティーフラグを取得します。
 *
 * @details     ダーティーフラグは設定項目に対応する変更 API の内部処理が
 *              完了した際に true となります。 本 API でその状態を取得すると、
 *              入れ替わる形で全てのダーティーフラグは false となります。
 *
 * @param[out]  pOutValue                   ダーティーフラグを読み出すバッファ
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetTelemetryDirtyFlags(TelemetryDirtyFlagSet* pOutValue) NN_NOEXCEPT;

}}} // namespace nn::settings::system
