﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       HID 設定に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_StaticAssert.h>
#include <nn/bluetooth/bluetooth_AddressTypes.h>
#include <nn/settings/system/settings_Common.h>
#include <nn/settings/system/settings_BluetoothDevices.h>
#include <nn/util/util_MathTypes.h>

namespace nn { namespace settings { namespace system {

NN_STATIC_ASSERT(sizeof(float) == 4);

const int NxControllerSettingsCountMax = BluetoothDevicesSettingsCountMax;  //!< コントローラ情報の最大数

/**
 * @brief       コントローラ情報を表す構造体です。
 */
struct NxControllerSettings
{
    ::nn::bluetooth::Address bd_addr;     // 0-5
    uint8_t device_type;            // 6
    char identification_code[16];   // 7-22
    ::nn::util::Color4u8Type mainColor; // 23-26
    ::nn::util::Color4u8Type subColor; // 27-30
    ::nn::util::Color4u8Type thirdColor; // 31-34
    ::nn::util::Color4u8Type forthColor; // 25-38
    uint8_t design_Variation; // 39
    uint8_t interface_type;   // 40
};

//! @name HID 設定関連 API
//! @{

/**
 * @brief       Pro Controller 有線 USB 通信機能が有効か否かを表す値を返します。
 *
 * @return      Pro Controller 有線 USB 通信機能が有効か否かを表す値です。
 */
bool IsUsbFullKeyEnabled() NN_NOEXCEPT;

/**
 * @brief       Pro Controller 有線 USB 通信機能が有効か否かを表す値を設定します。
 *
 * @param[in]   value                       Pro Controller 有線 USB 通信機能が有効か否かを表す値
 */
void SetUsbFullKeyEnabled(bool value) NN_NOEXCEPT;

/**
 * @brief       振動のマスターボリュームを返します。
 *
 * @return      振動のマスターボリュームです。
 */
float GetVibrationMasterVolume() NN_NOEXCEPT;

/**
 * @brief       振動のマスターボリュームを設定します。
 *
 * @param[in]   value                       振動のマスターボリューム
 */
void SetVibrationMasterVolume(float value) NN_NOEXCEPT;

/**
 * @brief       コントローラ情報を取得します。
 *
 * @details     コントローラ情報として指定されている値を取得可能な数だけ読み出します。
 *              取得可能なコントローラ情報の数より大きなバッファ（配列）が指定された場合、
 *              余った領域に対しては何も行いません。
 *              読み出し可能なコントローラ情報の最大数は NxControllerSettingsCountMax 個です。
 *
 * @param[out]  outValues                   コントローラ情報を読み出すバッファ（配列）
 * @param[in]   count                       outValues の数
 *
 * @return      読み出したコントローラ情報の数を返します。
 *
 * @pre
 *              - outValues != nullptr
 */
int GetNxControllerSettings(
    NxControllerSettings outValues[], int count) NN_NOEXCEPT;

/**
 * @brief       コントローラ情報を変更します。
 *
 * @details     コントローラ情報を指定された値に変更します。
 *              コントローラ情報の変更によって、以前に設定されていた値は全て消去されます。
 *              指定可能なコントローラ情報の最大数は NxControllerSettingsCountMax 個です。
 *
 * @param[in]   values                      コントローラ情報（配列）
 * @param[in]   count                       values の数
 *
 * @pre
 *              - values != nullptr
 *              - 0 <= count && count <= NxControllerSettingsCountMax
 */
void SetNxControllerSettings(
    const NxControllerSettings values[], int count) NN_NOEXCEPT;

//! @}

}}} // namespace nn::settings::system
