﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       オーディオ設定に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/settings/system/settings_Common.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace settings { namespace system {

/**
 * @brief       本体音量設定の制御フラグ定義です。
 */
struct AudioVolumeFlag
{
    /**
     * @brief       ミュート状態か否かを表すフラグです。
     */
    typedef ::nn::util::BitFlagSet<32, AudioVolumeFlag>::Flag<0> Mute;

    /**
     * @brief        出力レベルアップ状態か否かを表すフラグです。
     */
    typedef ::nn::util::BitFlagSet<32, AudioVolumeFlag>::Flag<1> HighPower;
};

/**
 * @brief       本体音量設定の制御フラグ集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, AudioVolumeFlag> AudioVolumeFlagSet;

/**
 * @brief       本体音量設定を表す構造体です。
 */
struct AudioVolume
{
    AudioVolumeFlagSet flags;   //!< 制御フラグ
    int8_t volume;              //!< 音量値
    ::nn::Bit8 _reserved[3];
};

/**
 * @brief       本体音量設定の操作対象を表す列挙型です。
 */
enum AudioVolumeTarget
{
    AudioVolumeTarget_Speaker = 0,      //!< スピーカ
    AudioVolumeTarget_Headphone = 1,    //!< ヘッドフォン
    AudioVolumeTarget_Usb = 2,          //!< USB オーディオ
};

/**
 * @brief       本体オーディオ出力設定を表す列挙型です。
 */
enum AudioOutputMode
{
    AudioOutputMode_1ch,            //!< 1 ch
    AudioOutputMode_2ch,            //!< 2 ch
    AudioOutputMode_5_1ch,          //!< 5.1 ch
    AudioOutputMode_7_1ch,          //!< 7.1 ch
    AudioOutputMode_5_1ch_Forced,   //!< 5.1 ch (強制)
};

/**
 * @brief       本体オーディオ出力設定の操作対象を表す列挙型です。
 */
enum AudioOutputModeTarget
{
    AudioOutputModeTarget_Hdmi,         //!< HDMI
    AudioOutputModeTarget_Speaker,      //!< スピーカ
    AudioOutputModeTarget_Headphone,    //!< ヘッドフォン
    AudioOutputModeTarget_Usb,          //!< USB オーディオ
};

//! @name オーディオ設定関連 API
//! @{

/**
 * @brief       本体音量設定を取得します。
 *
 * @param[out]  pOutAudioVolume             本体音量設定を読み出すバッファ
 * @param[in]   target                      本体音量設定の操作対象
 *
 * @pre
 *              - pOutAudioVolume != nullptr
 */
void GetAudioVolume(
    AudioVolume* pOutAudioVolume, AudioVolumeTarget target) NN_NOEXCEPT;

/**
 * @brief       本体音量設定を変更します。
 *
 * @details     本体音量設定を指定された値に変更します。
 *
 * @param[in]   audioVolume                 本体音量設定
 * @param[in]   target                      本体音量設定の操作対象
 */
void SetAudioVolume(
    const AudioVolume& audioVolume, AudioVolumeTarget target) NN_NOEXCEPT;

/**
 * @brief       本体オーディオ出力設定を取得します。
 *
 * @param[in]   target                      本体オーディオ出力設定の操作対象
 *
 * @return      target に対する本体オーディオ出力設定を返します。
 */
AudioOutputMode GetAudioOutputMode(AudioOutputModeTarget target) NN_NOEXCEPT;

/**
 * @brief       本体オーディオ出力設定を変更します。
 *
 * @param[in]   audioOutputMode             本体オーディオ出力設定
 * @param[in]   target                      本体オーディオ出力設定の操作対象
 */
void SetAudioOutputMode(
    AudioOutputMode audioOutputMode, AudioOutputModeTarget target) NN_NOEXCEPT;

/**
 * @brief       ヘッドフォン抜け時の強制ミュート設定を取得します。
 *
 * @return      強制ミュート設定を返します。
 */
bool IsForceMuteOnHeadphoneRemoved() NN_NOEXCEPT;

/**
 * @brief       ヘッドフォン抜け時の強制ミュート設定を変更します。
 *
 * @param[in]   forceMute                   強制ミュート設定
 */
void SetForceMuteOnHeadphoneRemoved(bool forceMute) NN_NOEXCEPT;

/**
 * @brief       本体ヘッドホン音量の警告表示イベントの回数を取得します。
 *
 * @return      警告表示イベントの回数を返します。
 */
int GetHeadphoneVolumeWarningDisplayedEventCount() NN_NOEXCEPT;

/**
 * @brief       本体ヘッドホン音量の警告表示イベントの回数を設定します。
 *
 * @param[in]   count                       警告表示イベントの回数
 */
void SetHeadphoneVolumeWarningDisplayedEventCount(int count) NN_NOEXCEPT;

/**
 * @brief       本体ヘッドホン音量を更新する必要があるか否かを表す値を返します。
 *
 * @param[out]  pOutIsHighPowerAvailable    出力レベルアップが利用可能か否か
 * @param[out]  pOutVolume                  音量
 * @param[in]   isRestricted                ペアレンタルコントロールが有効か否か
 *
 * @return      本体ヘッドホン音量を更新する必要があるか否かを表す値です。
 */
bool NeedsToUpdateHeadphoneVolume(
    bool* pOutIsHighPowerAvailable, int8_t* pOutVolume, bool isRestricted
    ) NN_NOEXCEPT;

//! @}

}}} // namespace nn::settings::system
