﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 *  @file
 *  @brief   amiibo に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/settings/factory/settings_Result.h>

namespace nn { namespace settings { namespace factory {

/**
 * @brief       amiibo 用本体秘密鍵を表す構造体です。
 */
struct AmiiboKey
{
    uint32_t size;          //!< amiibo 用本体秘密鍵のバイト数
    ::nn::Bit8 data[84];    //!< amiibo 用本体秘密鍵
};

/**
 * @brief       amiibo 用 ECQV 証明書を表す構造体です。
 */
struct AmiiboEcqvCertificate
{
    uint32_t size;          //!< amiibo 用 ECQV 証明書のバイト数
    ::nn::Bit8 data[20];    //!< amiibo 用 ECQV 証明書
};

/**
 * @brief       amiibo 用 ECDSA 証明書を表す構造体です。
 */
struct AmiiboEcdsaCertificate
{
    uint32_t size;          //!< amiibo 用 ECDSA 証明書のバイト数
    ::nn::Bit8 data[112];   //!< amiibo 用 ECDSA 証明書
};

/**
 * @brief       amiibo 用本体秘密鍵 (ECQV-BLS 版) を表す構造体です。
 */
struct AmiiboEcqvBlsKey
{
    uint32_t size;          //!< amiibo 用本体秘密鍵 (ECQV-BLS 版) のバイト数
    ::nn::Bit8 data[68];    //!< amiibo 用本体秘密鍵 (ECQV-BLS 版)
};

/**
 * @brief       amiibo 用 ECQV-BLS 証明書を表す構造体です。
 */
struct AmiiboEcqvBlsCertificate
{
    uint32_t size;          //!< amiibo 用 ECQV-BLS 証明書のバイト数
    ::nn::Bit8 data[32];    //!< amiibo 用 ECQV-BLS 証明書
};

/**
 * @brief       amiibo 用ルート証明書 (ECQV-BLS 版) を表す構造体です。
 */
struct AmiiboEcqvBlsRootCertificate
{
    uint32_t size;          //!< amiibo 用ルート証明書 (ECQV-BLS 版) のバイト数
    ::nn::Bit8 data[144];   //!< amiibo 用ルート証明書 (ECQV-BLS 版)
};

/**
 * @brief       amiibo 用本体秘密鍵を取得します。
 *
 * @param[out]  pOutValue                   amiibo 用本体秘密鍵を読みだすバッファ
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
::nn::Result GetAmiiboKey(AmiiboKey* pOutValue) NN_NOEXCEPT;

/**
 * @brief       amiibo 用 ECQV 証明書を取得します。
 *
 * @param[out]  pOutValue                   amiibo 用 ECQV 証明書を読みだすバッファ
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
::nn::Result GetAmiiboEcqvCertificate(
    AmiiboEcqvCertificate* pOutValue) NN_NOEXCEPT;

/**
 * @brief       amiibo 用 ECDSA 証明書を取得します。
 *
 * @param[out]  pOutValue                   amiibo 用 ECDSA 証明書を読みだすバッファ
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
::nn::Result GetAmiiboEcdsaCertificate(
    AmiiboEcdsaCertificate* pOutValue) NN_NOEXCEPT;

/**
 * @brief       amiibo 用本体秘密鍵 (ECQV-BLS 版) を取得します。
 *
 * @param[out]  pOutValue                   amiibo 用本体秘密鍵 (ECQV-BLS 版) を読みだすバッファ
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
::nn::Result GetAmiiboEcqvBlsKey(AmiiboEcqvBlsKey* pOutValue) NN_NOEXCEPT;

/**
 * @brief       amiibo 用 ECQV-BLS 証明書を取得します。
 *
 * @param[out]  pOutValue                   amiibo 用 ECQV-BLS 証明書を読みだすバッファ
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
::nn::Result GetAmiiboEcqvBlsCertificate(
    AmiiboEcqvBlsCertificate* pOutValue) NN_NOEXCEPT;

/**
 * @brief       amiibo 用ルート証明書 (ECQV-BLS 版) を取得します。
 *
 * @param[out]  pOutValue                   amiibo 用ルート証明書 (ECQV-BLS 版) を読みだすバッファ
 *
 * @retresult
 * @handleresult{nn::settings::factory::ResultCalibrationDataFileSystemCorrupted}
 * @handleresult{nn::settings::factory::ResultCalibrationDataCrcError}
 * @endretresult
 *
 * @pre
 *              - pOutValue != nullptr
 */
::nn::Result GetAmiiboEcqvBlsRootCertificate(
    AmiiboEcqvBlsRootCertificate* pOutValue) NN_NOEXCEPT;

}}} // nn::settings::factory
