﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   SASBUS ドライバライブラリのバスアクセス機能の API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/sasbus/sasbus_DeviceName.h>
#include <nn/sasbus/sasbus_PeriodicReceiveModeType.h>
#include <nn/sasbus/detail/sasbus_RingLifo.h>


namespace nn {
namespace sasbus {
namespace driver {

namespace detail {
const int SessionSize = 128;
const int SessionAlignment = 8;
struct SessionImplPadded;
}

/**
* @brief   セッションパラメータです。メンバに直接アクセスしないでください。
*/
struct Session
{
    util::TypedStorage<detail::SessionImplPadded, detail::SessionSize, detail::SessionAlignment>  _impl;
    char*           _receiveBuffer;
    size_t          _receiveBufferSize;
};


/**
 * @brief   指定した sasbus デバイスとのセッションをオープンします。
 *
 * @param[out]  pOutSession オープンされたセッション
 * @param[in]   device      デバイス識別子
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  指定したデバイスは存在している必要があります。@n
 *  @a pOutSession はアクセス可能なアドレスを指している必要があります。@n
 *
 * @post
 *  また、@a pOutSession に有効なセッションが格納され、指定したデバイスにアクセス可能になります。
 *
 * @details
 *  指定した sasbus デバイスとのセッションをオープンします。
 *
 */
void OpenSession(Session* pOutSession, SasbusDevice device) NN_NOEXCEPT;

/**
 * @brief   指定したレジスタアドレスへデータ列を書き込みます。
 *
 * @param[in]   pSession        セッション
 * @param[in]   pInData         送信するデータ列
 * @param[in]   dataBytes       送信するデータ列の長さ [bytes]
 * @param[in]   reg             書き込むレジスタアドレス
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess                   データの書き込みに成功しました。
 * @retval      ResultSprioutInterruptOccured   想定外の割り込みが発生しました。送信が失敗している可能性があります。セッションをクローズし、再オープンしたのち、再度送信をやり直してください。
 * @retval      ResultTimeout                   送信・受信でタイムアウトが発生しました。
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。
 *
 * @details
 *  指定したレジスタアドレスへデータ列を書き込みます。@n
 *  内部的に 1 秒間デバイスからの反応がなかった時に timeout を返します。
 */
nn::Result Send(Session* pSession, const void* pInData, size_t dataBytes, const char reg) NN_NOEXCEPT;

/**
 * @brief   指定したレジスタアドレスのデータ列を受信します。
 *
 * @param[out]  pOutData        受信したデータ列を格納するバッファ
 * @param[in]   pSession        セッション
 * @param[in]   dataBytes       受信するデータ列の長さ [bytes]
 * @param[in]   reg             読み出すレジスタアドレス
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess                   データの書き込みに成功しました。
 * @retval      ResultSprioutInterruptOccured   想定外の割り込みが発生しました。受信が失敗している可能性があります。セッションをクローズし、再オープンしたのち、再度受信をやり直してください。
 * @retval      ResultTimeout                   送信・受信でタイムアウトが発生しました。
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。@n
 *
 * @details
 *  受信済データ列を、指定したバッファに、最大で @a dataBytes で指定した長さだけコピーします。@n
 *  内部的に 1 秒間デバイスからの反応がなかった時に timeout を返します。
 */
nn::Result Receive(void* pOutData, Session* pSession, size_t dataBytes, const char reg) NN_NOEXCEPT;

/**
 * @brief   指定したセッションをクローズします。
 *
 * @param[in]   pSession    セッション
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。
 *
 * @post
 *  指定したセッションがクローズ状態になります。
 *
 * @details
 *  指定したセッションをクローズします。@n
 */
void CloseSession(Session* pSession) NN_NOEXCEPT;

/**
 * @brief   特定のレジスタからの定期的なデータの受信を開始します。
 *
 * @param[in]   pSession             セッション
 * @param[in]   dataBytes            受信するデータ列の長さ [bytes]
 * @param[in]   reg                  読み出すレジスタアドレス
 * @param[in]   interval             読み出す間隔
 * @param[in]   receiveBuffer        受信バッファ
 * @param[in]   receiveBufferLength  受信バッファのサイズ
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess                           PeriodicReceiveMode の開始に成功しました。
 * @retval      ResultAlreadyStartPeriodicReceiveMode       すでに PeriodicReceiveMode が開始されています。新しく実行したい場合は StopPeriodicReceiveMode() を呼んでください。
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。@n
 *  PeriodicReceiveMode が開始されていない状態である必要があります。
 *
 * @details
 *  reg で指定したレジスタに対して dataSize 分の定期的なデータの受信を行います。@n
 *  receiveBuffer で指定したバッファに interval で指定した間隔でデータが書き込まれ続けるため、連続して Receive を実行する場合と比較して処理が軽くなります。@n
 *  receiveBuffer の中には　RingLifo　のデータ形式で書き込みを行います。取り出す場合は GetPeriodicReceiveModeData() を使用してください。@n
 *  PeriodicReceiveMode はシステム全体で 1 つだけ設定できます。すでに実行された PeriodicReceiveMode がある場合、 ResultAlreadyStartPeriodicReceiveMode が返ります。
 *  interval を短くすると CPU 負荷の増大につながります。定期的な受信をやめる場合は必ず StopPeriodicReceiveMode() を呼んでください。
 */
Result StartPeriodicReceiveMode(Session* pSession, size_t dataSize, char reg, nn::TimeSpan interval, char* receiveBuffer, size_t receiveBufferLength) NN_NOEXCEPT;

/**
 * @brief   特定のレジスタからの定期的なデータの受信を停止します。
 *
 * @param[in]   pSession        セッション
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。@n
 *  PeriodicReceiveMode が実行されている必要があります。
 *
 * @details
 *    StartPeriodicReceiveMode() で開始した PeriodicReceiveMode を停止します。
 */
void StopPeriodicReceiveMode(Session* pSession) NN_NOEXCEPT;

/**
 * @brief   PeriodicReceive によって読み出された受信データを受信バッファから取り出します。
 *
 * @param[out]  pOutData        受信したデータ列を格納するバッファ
 * @param[in]   pSession        セッション
 * @param[in]   count           取り出すデータ数
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pSession は有効なセッションである必要があります。@n
 *  PeriodicReceiveMode が実行されている必要があります。
 *
 * @details
 *  PeriodicReceive によって読み出された受信データを受信バッファから PeriodicReceiveModeDataT 形式で　count 個取り出します。@n
 *  PeriodicReceiveMode 中に発生したエラーは PeriodicReceiveModeDataT 構造体の result メンバ変数を参照することで取得できます。
 */
template <typename PeriodicReceiveModeDataT, const int LifoEntryCount>
void GetPeriodicReceiveModeData(PeriodicReceiveModeDataT* pOutData, Session* pSession, int count) NN_NOEXCEPT
{
    auto pRingLifo = reinterpret_cast<nn::sasbus::detail::RingLifo<PeriodicReceiveModeDataT, LifoEntryCount>*>(pSession->_receiveBuffer);

    pRingLifo->Read(pOutData, count);
}

} // driver
} // sasbus
} // nn


