﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   VR モード関連機能に関する API の宣言
 *
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/oe/oe_NotificationMessage.h>

namespace nn { namespace oe {

/**
 * @brief   アプリに対する VR モード終了を要求するメッセージ通知です。
 *
 * @details
 *  VR モードの 2 画面出力から HOME メニューなどの 1 画面出力のシーンに
 *  画面遷移する際は、システムがユーザに対して注意と確認を求めます。
 *  このメッセージは、ユーザにその了承が得られた際に、アプリに対しても
 *  VR モードを終了し、2 画面出力から 1 画面出力に切り替えるように
 *  要求するためのメッセージです。
 *
 *  アプリケーションはこのメッセージ通知を受けたら、
 *
 *  - まず先に nn::pl::EndVrMode() を発行する
 *  - その次に 2 画面出力から 1 画面出力に切り替える
 *
 *  という処理を行なうようにして下さい。既に VR モードを終了した状態で
 *  nn::pl::EndVrMode() を発行しても特に問題ありません。
 *
 *  VR モードの詳細については @confluencelink{233737220, 「VR モードについて」}を参照してください。
 *
 */
const Message MessageRequestToEndVrMode = 71;

}}  // namespace nn::oe


namespace nn { namespace pl {

//--------------------------------------------------------------------------

//! @name VR モード関連機能
//! @{

/**
    @brief VR モードを開始します。

    @details
     VR モードを開始する前に本 API を発行してください。
     VR モードを終了するには nn::pl::EndVrMode() を発行して下さい。

     VR モードの詳細については @confluencelink{233737220, 「VR モードについて」}を参照してください。

    @platformbegin{Windows}
     - Windows 環境では何もせずにリターンします。

    @platformend

*/
void BeginVrMode() NN_NOEXCEPT;

/**
    @brief VR モードを終了します。

    @details
     VR モードを終了する時に本 API を発行してください。
     詳細は nn::pl::BeginVrMode() を参照して下さい。

    @platformbegin{Windows}
     - Windows 環境では何もせずにリターンします。

    @platformend

*/
void EndVrMode() NN_NOEXCEPT;

/**
    @brief  現在が VR モードか否かを返します。

    @return 現在が VR モードなら true を、そうでないなら false を返します。

    @details
     現在が VR モードか否かを bool 値で返します。

*/
bool IsVrMode() NN_NOEXCEPT;

//! @}

//-----------------------------------------------------------------------------

}} // namespace nn::pl

