﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   照度の取得機能に関する API の宣言
 *
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>

namespace nn { namespace pl {

//--------------------------------------------------------------------------

//! @name   照度の取得
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   現在のプレイ環境での照度を取得します。
 *
 * @return  現在の照度を返します。値の単位は lux です。
 *
 * @details
 *  現在のプレイ環境での照度を取得します。 @n
 *  @n
 *  照度が取得できない環境では、 0.0f が返ります。
 *  照度が取得できる環境であるかどうかの判定には nn::pl::IsIlluminanceAvailable() を使用してください。
 *  @n
 *  光量過多の状態を一緒に返すオーバーロードされた関数も提供されています。  @ref nn::pl::GetCurrentIlluminance(bool*) を参照してください。
 */
float GetCurrentIlluminance() NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 * @brief   現在のプレイ環境での照度を、光量過多アラートと共に取得します。
 *
 * @return  現在の照度を返します。値の単位は lux です。
 *
 * @param[out] pOutOverflowAlert 光量過多のため内部の計算でオーバーフローが発生した場合に true をセットします。それ以外で false をセットします。
 *
 * @pre
 *  - pOutOverflowAlert != nullptr
 *
 * @details
 *  現在のプレイ環境での照度を取得します。 @n
 *  光量過多のため内部の計算でオーバーフローが発生し、取得された照度値の精度が低い可能性がある場合には、 @a pOutOverflowAlert に true をセットします。
 *  @n
 *  照度が取得できない環境では、 0.0f が返ります。
 *  照度が取得できる環境であるかどうかの判定には nn::pl::IsIlluminanceAvailable() を使用してください。
 */
float GetCurrentIlluminance(bool* pOutOverflowAlert) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 * @brief   照度が取得できるかどうかを返します。
 *
 * @return  照度が取得できるかどうかを返します。
 *
 * @details
 *  照度が取得できるかどうかを返します。
 */
bool IsIlluminanceAvailable() NN_NOEXCEPT;

//! @}

//-----------------------------------------------------------------------------

}} // namespace nn::pl

