﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/*
 * @file
 * @brief  PlayDataManager が内部で使用する型を定義します。
 */

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/pdm/pdm_SystemTypes.h>
#include <nn/time/time_PosixTime.h>

namespace nn { namespace pdm {

namespace detail {
    //! イベント情報の中でカテゴリ毎に記録するデータのサイズ
    const int PlayEventDataContentSize = 28;
    //! イベント情報 1エントリーのサイズ（カテゴリ毎に記録する情報のサイズ + 全てのイベントで共通する情報のサイズ）
    const int PlayEventDataSize = PlayEventDataContentSize + 28;
}

//! @brief 記録するイベントのカテゴリーです。
enum class PlayEventCategory : uint8_t
{
    Applet                  = 0, //!< アプレット（アプリケーション含む）の起動・遷移に関するイベント
    UserAccount             = 1, //!< ユーザーアカウントに関するイベント
    PowerStateChange        = 2, //!< 電源・スリープ状態の変更に関するイベント
    OperationModeChange     = 3, //!< 動作モードの変更に関するイベント
    SteadyClockReset        = 4, //!< SteadyClock のリセットのイベント
};

//! @brief ライブラリアプレットのイベントが持つデータです。
struct LibraryAppletExtraInfo
{
    Bit8 version;           //!< LibraryAppletExtraInfo のバージョン。
    Bit8 libraryAppletMode; //!< applet::LibraryAppletMode
    Bit8 reserved[2];
};

//! @brief PlayEventCategory::Applet のイベントが持つデータです。
struct AppletEventData
{
    /**
        @brief イベントに関連するアプレットのプログラムIDの上位32bit
        @details
            - 6.0.0以降のライブラリアプレットの場合、イベントを発生させたアプレットのメインアプレット（呼び出し元）のプログラムIDの上位32bit
            - それ以外の場合、イベントを発生させたアプレットのプログラムIDの上位32bit
    */
    uint32_t            programIdHi;
    /**
        @brief イベントに関連するアプレットのプログラムIDの下位32bit
        @details
            - 6.0.0以降のライブラリアプレットの場合、イベントを発生させたアプレットのメインアプレット（呼び出し元）のプログラムIDの下位32bit
            - それ以外の場合、イベントを発生させたアプレットのプログラムIDの下位32bit
    */
    uint32_t            programIdLow;
    union
    {
        uint32_t               version; //!< イベントを発生させたアプリケーションのバージョン。アプリケーションのみ有効。
        LibraryAppletExtraInfo laInfo;  //!< イベントを発生させたライブラリアプレットの情報。ライブラリアプレットのみ有効。
    };
    uint8_t             appletId;       //!< イベントを発生させたアプレットのアプレットID（nn::applet::AppletId を uint8_t で保存）
    ncm::StorageId      storageId;      //!< イベントを発生させたアプレットが格納されたストレージの種類
    ns::PlayLogPolicy   playLogPolicy;  //!< イベントを発生させたアプレットのプレイログ記録方針
    AppletEventType     eventType;      //!< イベントの種類
    Bit8                reserved[detail::PlayEventDataContentSize - 16];
};
NN_STATIC_ASSERT( sizeof(AppletEventData) == detail::PlayEventDataContentSize );

//! @brief PlayEventCategory::UserAccount の UserAccountEventType::NetworkServiceAccountAvailable のイベントが追加で持つデータです。
struct UserAccountEventNetworkServiceAccountAvailableData
{
    uint32_t    networkServiceAccountIdHi;  //!< 関連付いたネットワークサービスアカウントIDの上位32bit。
    uint32_t    networkServiceAccountIdLow; //!< 関連付いたネットワークサービスアカウントIDの下位32bit。
};

//! @brief PlayEventCategory::UserAccount のイベントが持つデータです。
struct UserAccountEventData
{
    uint32_t    userId0Hi;  //!< イベントを発生させたアカウントのユーザーアカウントのID。nn::account::Uid の内部データ Bit64[2]の [0] の上位32bit。
    uint32_t    userId0Low; //!< イベントを発生させたアカウントのユーザーアカウントのID。nn::account::Uid の内部データ Bit64[2]の [0] の下位32bit。
    uint32_t    userId1Hi;  //!< イベントを発生させたアカウントのユーザーアカウントのID。nn::account::Uid の内部データ Bit64[2]の [1] の上位32bit。
    uint32_t    userId1Low; //!< イベントを発生させたアカウントのユーザーアカウントのID。nn::account::Uid の内部データ Bit64[2]の [1] の下位32bit。
    union
    {
        UserAccountEventNetworkServiceAccountAvailableData   networkServiceAccountAvailableData;   //!< eventType == NetworkServiceAccountAvailable 時に有効なデータです。
    };
    UserAccountEventType    eventType;  //!< イベントの種類
};
NN_STATIC_ASSERT( sizeof(UserAccountEventData) == detail::PlayEventDataContentSize );

//! @brief PlayEventCategory::PowerState のイベントが持つデータです。
struct PowerStateChangeEventData
{
    PowerStateChangeEventType eventType;  // !< イベントの種類
    Bit8                      reserved[detail::PlayEventDataContentSize - sizeof(PowerStateChangeEventType)];
};
NN_STATIC_ASSERT( sizeof(PowerStateChangeEventData) == detail::PlayEventDataContentSize );

//! @brief PlayEventCategory::OperationModeChange のイベントの持つデータです。
struct OperationModeChangeEventData
{
    pdm::OperationMode  operationMode;    //!< イベント発生後の動作モード
    Bit8                reserved[detail::PlayEventDataContentSize - sizeof(pdm::OperationMode)];
};
NN_STATIC_ASSERT( sizeof(OperationModeChangeEventData) == detail::PlayEventDataContentSize );


//! @brief PlayDataManager が内部で使用するイベント情報を記録するための構造体です。
struct PlayEvent
{
    union
    {
        AppletEventData                 appletEventData;
        UserAccountEventData            userAccountEventData;
        PowerStateChangeEventData       powerStateChangeEventData;
        OperationModeChangeEventData    operationModeEventData;
        Bit8                            eventData[detail::PlayEventDataContentSize];
    };
    PlayEventCategory   eventCategory;  //!< イベントの種類
    time::PosixTime     userTime;       //!< イベント発生時のユーザー時計時刻（協定世界時 (UTC) の1970年1月1日午前0時0分0秒からの経過秒数）。
    time::PosixTime     networkTime;    //!< イベント発生時のネットワーク時計時刻（協定世界時 (UTC) の1970年1月1日午前0時0分0秒からの経過秒数）。ネットワーク時計が無効な場合は 0。
    int64_t             steadyTime;     //!< イベント発生時の単調増加時計時刻（記録されたベース時刻からの経過秒）。
};
NN_STATIC_ASSERT( sizeof(PlayEvent) == detail::PlayEventDataSize );

inline bool operator ==(const AppletEventData& lhs, const AppletEventData& rhs) NN_NOEXCEPT
{
    return (lhs.programIdHi     == rhs.programIdHi)
        && (lhs.programIdLow    == rhs.programIdLow)
        && (lhs.version         == rhs.version)
        && (lhs.appletId        == rhs.appletId)
        && (lhs.storageId       == rhs.storageId)
        && (lhs.playLogPolicy   == rhs.playLogPolicy)
        && (lhs.eventType       == rhs.eventType);
}

inline bool operator ==(const UserAccountEventNetworkServiceAccountAvailableData& lhs, const UserAccountEventNetworkServiceAccountAvailableData& rhs) NN_NOEXCEPT
{
    return (lhs.networkServiceAccountIdHi == rhs.networkServiceAccountIdHi)
        && (lhs.networkServiceAccountIdLow == rhs.networkServiceAccountIdLow);
}

inline bool operator ==(const UserAccountEventData& lhs, const UserAccountEventData& rhs) NN_NOEXCEPT
{
    return (lhs.userId0Hi   == rhs.userId0Hi)
        && (lhs.userId0Low  == rhs.userId0Low)
        && (lhs.userId1Hi   == rhs.userId1Hi)
        && (lhs.userId1Low  == rhs.userId1Low)
        && (lhs.eventType   == rhs.eventType)
        && ((lhs.eventType != UserAccountEventType::NetworkServiceAccountAvailable) || (lhs.networkServiceAccountAvailableData == rhs.networkServiceAccountAvailableData));
}

inline bool operator ==(const PowerStateChangeEventData& lhs, const PowerStateChangeEventData& rhs) NN_NOEXCEPT
{
    return lhs.eventType == rhs.eventType;
}

inline bool operator ==(const OperationModeChangeEventData& lhs, const OperationModeChangeEventData& rhs) NN_NOEXCEPT
{
    return lhs.operationMode == rhs.operationMode;
}

inline bool operator ==(const PlayEvent& lhs, const PlayEvent& rhs) NN_NOEXCEPT
{
    if( (lhs.eventCategory == rhs.eventCategory) && (lhs.userTime == rhs.userTime) && (lhs.networkTime == rhs.networkTime) && (lhs.steadyTime == rhs.steadyTime) )
    {
        switch( lhs.eventCategory )
        {
        case PlayEventCategory::Applet:
            {
                return lhs.appletEventData == rhs.appletEventData;
            }
            break;

        case PlayEventCategory::UserAccount:
            {
                return lhs.userAccountEventData == rhs.userAccountEventData;
            }
            break;

        case PlayEventCategory::PowerStateChange:
            {
                return lhs.powerStateChangeEventData == rhs.powerStateChangeEventData;
            }
            break;

        case PlayEventCategory::OperationModeChange:
            {
                return lhs.operationModeEventData == rhs.operationModeEventData;
            }
            break;

        case PlayEventCategory::SteadyClockReset:
            {
                return true;
            }

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    return false;
}

//! @name アカウントイベントレコード定義
//@{

/**
 * @brief   PlayDataManager が内部で使用するアカウントイベントエントリを記録するための構造体です。
 * @details 本構造体内で使われる構造体群は全て以下の条件を満たす必要があります。
 *          - 最小容量になるように考慮した構造体パッキングアライメント。
 *          - POD ( Plain Old Data ) 構造。
 */
struct AccountPlayEvent
{
    /**
     * @brief 内部用途明示化用型定義です。
     */
    typedef Bit8    AppletId8;      //!< nn::applet::AppletId を Bit8 で縮小表現するための型定義です。
    typedef Bit32   DurationType;   //!< 期間表現を行う表現幅の共通化のための型定義です。

    NN_FORCEINLINE static AppletId8 ConvertFrom(const applet::AppletId source) NN_NOEXCEPT;
    NN_FORCEINLINE static applet::AppletId ConvertFrom(const AppletId8 source) NN_NOEXCEPT;

    /**
     * @brief 構造体の packing alignment を32bit境界にするために32bitx2 メンバ構成で定義した64bit表現IDの代替型です。
     */
    struct PackedId64
    {
        Bit32 hi;     //!< プログラムIDの上位32bit
        Bit32 low;    //!< プログラムIDの下位32bit

        static const Bit64 InvalidValue = 0ull;

        NN_IMPLICIT PackedId64(const Bit64 value = InvalidValue) NN_NOEXCEPT
            : hi(static_cast<Bit32>((value >> 32) & 0xffffffff))
            , low(static_cast<Bit32>((value >> 0) & 0xffffffff)) {}
        explicit PackedId64(const Bit32 hi_, const Bit32 low_) NN_NOEXCEPT : hi(hi_), low(low_) {}
        NN_FORCEINLINE const Bit64 ToValue64() const NN_NOEXCEPT;
        NN_FORCEINLINE ncm::ProgramId ToProgramId() const NN_NOEXCEPT;
        NN_FORCEINLINE account::NetworkServiceAccountId ToNetworkServiceAccountId() const NN_NOEXCEPT;
    };

    /**
     * @brief 発生時刻など、時間情報を保持する構造体です。
     */
    struct Time
    {
        time::PosixTime user;       //!< ユーザー時計時刻 (協定世界時 (UTC) の1970年1月1日午前0時0分0秒からの経過秒数)
        time::PosixTime network;    //!< ネットワーク時計時刻 (協定世界時 (UTC) の1970年1月1日午前0時0分0秒からの経過秒数), ネットワーク時計が無効な場合は 0.

        static const int64_t InvalidValue = 0ll;

        NN_IMPLICIT Time(const int64_t value = InvalidValue) NN_NOEXCEPT
        {
            user.value = value;
            network.value = value;
        }
        explicit Time(const PlayEvent& source) NN_NOEXCEPT : user(source.userTime), network(source.networkTime) {}
        NN_FORCEINLINE bool IsValid() const NN_NOEXCEPT;
        NN_FORCEINLINE void Invalidate() NN_NOEXCEPT;
        NN_FORCEINLINE void Apply(const PlayEvent& source) NN_NOEXCEPT;
    };

    /**
     * @brief 記録契機種別
     */
    enum class RecordCause : Bit8
    {
        None,           //!< 記録されるべきではない状態
        PowerOff,       //!< 電源OFF契機により記録された状態
        FocusOut,       //!< Applet Focus Out 契機により記録された状態
        AccountClose,   //!< Account Close 契機により記録された状態
    };

    /**
     * @brief アプレットイベント情報構造体。
     */
    struct Applet
    {
        /**
            @brief イベントに関連するアプレットのプログラムID
            @details
                - 6.0.0以降のライブラリアプレットの場合、イベントを発生させたアプレットのメインアプレット（呼び出し元）のプログラムID
                - それ以外の場合、イベントを発生させたアプレットのプログラムID
        */
        PackedId64          programId;
        union
        {
            uint32_t                version;//!< イベントを発生させたアプリケーションのバージョン。アプリケーションのみ有効。
            LibraryAppletExtraInfo  laInfo; //!< イベントを発生させたライブラリアプレットの追加情報。ライブラリアプレットのみ有効。
        };
        ncm::StorageId      storageId;      //!< イベントを発生させたアプレットが格納されたストレージの種類
        AppletId8           appletId;       //!< イベントを発生させたアプレットのアプレットID
        RecordCause         cause;          //!< イベント発生時の記録契機種別
        pdm::OperationMode  operationMode;  //!< イベント発生時の動作モード( dock in/out ), ( Category::InFocus のみ有効 )

        NN_IMPLICIT Applet() NN_NOEXCEPT
        {
            Invalidate();
        }
        inline void Invalidate() NN_NOEXCEPT;
        inline void Apply(const AppletEventData& source) NN_NOEXCEPT;
        inline void Apply(const AppletEventData& source, const RecordCause& recordCause, const pdm::OperationMode& dockMode) NN_NOEXCEPT;
        inline void Apply(const Applet& source, const RecordCause& recordCause, const pdm::OperationMode& dockMode) NN_NOEXCEPT;
    };

    /**
     * @brief アカウントイベントレコードデータベースへ記録されるエントリ毎のバリエーション吸収用コンテキスト構造体です。
     */
    union Context
    {
        /**
         * @brief アカウントオープン期間情報を保持する構造体です。
         */
        struct InOpen
        {
            Applet              applet;         //!< アプレット情報
            DurationType        duration;       //!< イベント発生期間( 秒 )

            NN_IMPLICIT InOpen() NN_NOEXCEPT : applet(), duration(0) {}
        };

        /**
         * @brief アプレットのInFocus期間情報を保持する構造体です。( アカウントオープン期間中のInFocus )
         */
        typedef InOpen InFocus;

        /**
         * @brief ネットワークサービスアカウント(NSA)の利用可能時情報を保持する構造体です。
         */
        struct NetworkServiceAccountAvailable
        {
            PackedId64          nsaId;          //!< 利用可能になったネットワークサービスアカウントID ( account::NetworkServiceAccountId )

            NN_IMPLICIT NetworkServiceAccountAvailable() NN_NOEXCEPT : nsaId() {}
        };

        /**
         * @brief ネットワークサービスアカウント(NSA)の利用不可時情報を保持する構造体です。
         */
        struct NetworkServiceAccountUnavailable
        {
        };

        /**
         * @brief 最大領域を占めるコンテキスト型エイリアス。
         */
        typedef InOpen MaximumVolumeContextType;

        InFocus                             inFocus;
        InOpen                              inOpen;
        NetworkServiceAccountAvailable      nsaAvailable;
        NetworkServiceAccountUnavailable    nsaUnavailable;
        Bit8    data[sizeof(MaximumVolumeContextType)];   //!< 記録イベント別コンテキスト領域

        /**
         * @brief デフォルトコンストラクタ
         */
        NN_IMPLICIT Context() NN_NOEXCEPT
        {
            new(data) MaximumVolumeContextType();
        }
    };

    /**
     * @brief アカウントイベントレコードデータベースに記録するイベントエントリのカテゴリーです。
     */
    enum class Category : Bit8
    {
        InOpen          = 0,    //!< アカウントオープン期間情報エントリ
        InFocus         = 1,    //!< アプレットのInFocus期間情報エントリ ( アカウントオープン期間中 )
        NsaAvailable    = 2,    //!< ネットワークサービスアカウントの利用可能遷移通知エントリ
        NsaUnavailable  = 3,    //!< ネットワークサービスアカウントの利用不可遷移通知エントリ
    };

    /**
     * @brief アカウントイベントレコードデータベースに記録するイベントエントリの PowerState変化検知情報種別です。
     */
    enum class PowerChange : Bit8
    {
        None,       //!< Power の変化は検出していない。
        PowerOn,    //!< PowerOn への変化を検知した。
    };

    Bit8            reserved[2];    //!< 未使用( 0 固定 )
    PowerChange     powerChange;    //!< このエントリが最初に検知したPowerState状態
    Category        category;       //!< このエントリのカテゴリー
    Context         context;        //!< このエントリのイベントコンテキスト
    Time            time;           //!< イベント発生時の時刻

    inline NN_IMPLICIT AccountPlayEvent() NN_NOEXCEPT;
    inline explicit AccountPlayEvent(const Category& category_, const PlayEvent& time_) NN_NOEXCEPT;
};

//@}

NN_STATIC_ASSERT(sizeof(AccountPlayEvent::Context::InOpen) == 20);
NN_STATIC_ASSERT(sizeof(AccountPlayEvent::Context::InFocus) == 20);
NN_STATIC_ASSERT(sizeof(AccountPlayEvent::Context::NetworkServiceAccountAvailable) == 8);
NN_STATIC_ASSERT(sizeof(AccountPlayEvent::Context) == 20);
NN_STATIC_ASSERT(sizeof(AccountPlayEvent) == 40);

}}

#include <algorithm>

namespace nn { namespace pdm {

//! @name アカウントイベントレコード, インラインメソッド実装
//@{

NN_FORCEINLINE const Bit64 AccountPlayEvent::PackedId64::ToValue64() const NN_NOEXCEPT
{
    return (static_cast<Bit64>(hi) << 32) | low;
}

NN_FORCEINLINE ncm::ProgramId AccountPlayEvent::PackedId64::ToProgramId() const NN_NOEXCEPT
{
    return ncm::ProgramId{ToValue64()};
}

NN_FORCEINLINE account::NetworkServiceAccountId AccountPlayEvent::PackedId64::ToNetworkServiceAccountId() const NN_NOEXCEPT
{
    return account::NetworkServiceAccountId{ToValue64()};
}

inline bool operator ==(const AccountPlayEvent::PackedId64& lhs, const AccountPlayEvent::PackedId64& rhs) NN_NOEXCEPT
{
    return (lhs.hi == rhs.hi && lhs.low == rhs.low);
}

inline bool operator !=(const AccountPlayEvent::PackedId64& lhs, const AccountPlayEvent::PackedId64& rhs) NN_NOEXCEPT
{
    return !(lhs == rhs);
}

NN_FORCEINLINE AccountPlayEvent::AppletId8 AccountPlayEvent::ConvertFrom(const applet::AppletId source) NN_NOEXCEPT
{
    return static_cast<AppletId8>(source);
}

NN_FORCEINLINE applet::AppletId AccountPlayEvent::ConvertFrom(const AccountPlayEvent::AppletId8 source) NN_NOEXCEPT
{
    return static_cast<applet::AppletId>(source);
}

NN_FORCEINLINE bool AccountPlayEvent::Time::IsValid() const NN_NOEXCEPT
{
    // ユーザー時計時刻の取得失敗は想定外のため、userTime が 0 でなければ、有効な Data が格納されているとする。
    return user.value != InvalidValue;
}

NN_FORCEINLINE void AccountPlayEvent::Time::Invalidate() NN_NOEXCEPT
{
    user.value = InvalidValue;
    network.value = InvalidValue;
}

NN_FORCEINLINE void AccountPlayEvent::Time::Apply(const PlayEvent& source) NN_NOEXCEPT
{
    network = source.networkTime;
    user = source.userTime;
}

inline void AccountPlayEvent::Applet::Invalidate() NN_NOEXCEPT
{
    programId = PackedId64::InvalidValue;
    version = 0;
    storageId = ncm::StorageId::None;
    appletId = 0;
    cause = RecordCause::None;
}

inline void AccountPlayEvent::Applet::Apply(const AppletEventData& source) NN_NOEXCEPT
{
    programId.hi = source.programIdHi;
    programId.low = source.programIdLow;
    version = source.version;
    storageId = source.storageId;
    appletId = source.appletId;
}

inline void AccountPlayEvent::Applet::Apply(const AppletEventData& source, const RecordCause& recordCause, const pdm::OperationMode& dockMode) NN_NOEXCEPT
{
    Apply(source);
    cause = recordCause;
    operationMode = dockMode;
}

inline void AccountPlayEvent::Applet::Apply(const Applet& source, const RecordCause& recordCause, const pdm::OperationMode& dockMode) NN_NOEXCEPT
{
    programId = source.programId;
    version = source.version;
    storageId = source.storageId;
    appletId = source.appletId;
    cause = recordCause;
    operationMode = dockMode;
}

inline AccountPlayEvent::AccountPlayEvent() NN_NOEXCEPT
{
    powerChange = PowerChange::None;
    std::fill_n(reserved, NN_ARRAY_SIZE(reserved), static_cast<Bit8>(0));
}

inline AccountPlayEvent::AccountPlayEvent(const Category& category_, const PlayEvent& time_) NN_NOEXCEPT
{
    powerChange = PowerChange::None;
    category = category_;
    time.Apply(time_);
    std::fill_n(reserved, NN_ARRAY_SIZE(reserved), static_cast<Bit8>(0));
}

//@}

}}
