﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/pdm/pdm_PrivateTypes.h>
#include <nn/pdm/pdm_SystemTypes.h>
#include <nn/account.h>
#include <nn/os.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/prepo/prepo_SystemPlayReport.h>
#include <nn/time.h>
#include <nn/time/time_PosixTime.h>
#include <nn/util/util_Optional.h>
#include <functional>

namespace nn { namespace pdm { namespace detail {

/**
* @brief プレイログのエントリー作成の起点・終点となる時刻情報を表す構造体です。
*/
struct PlayLogEntryPointTime
{
    time::PosixTime                 userClockTime;              //!< ユーザー時計時刻
    util::optional<time::PosixTime> networkClockTime;           //!< ネットワーク時計時刻
    int64_t                         steadyClockTimePointValue;  //!< 単調増加時計時刻の値

    static PlayLogEntryPointTime MakeWithCurrentTime() NN_NOEXCEPT;
};

/**
* @brief プレイログのエントリーに保存されるアプレットの情報を表す構造体です。
*/
struct AppletData
{
    ncm::ApplicationId          applicationId;
    applet::AppletId            appletId;
    uint32_t                    version;
    ncm::StorageId              storageId;

    void Set(const ncm::ApplicationId& applicationId, applet::AppletId appletId, uint32_t version, const ncm::StorageId& storageId) NN_NOEXCEPT;
};

/**
* @brief    プレイログのエントリーの起点となるイベントの情報を表すクラスです。
*/
template<typename Data>
struct PlayLogEntryStartPoint
{
    Data                    data;
    PlayLogEntryPointTime   time;

    void Invalidate() NN_NOEXCEPT;
    bool IsValid() const NN_NOEXCEPT;
};

template<typename Data>
void PlayLogEntryStartPoint<Data>::Invalidate() NN_NOEXCEPT
{
    time.userClockTime.value = 0;
    time.networkClockTime = nullptr;
    time.steadyClockTimePointValue = 0;
}

template<typename Data>
bool PlayLogEntryStartPoint<Data>::IsValid() const NN_NOEXCEPT
{
    return time.userClockTime.value != 0; // ユーザー時計時刻の取得失敗は想定外。
}

/**
* @brief    PlayEvent から PlayReport を作成するクラスです。
*/
class PlayLogGenerator
{
    NN_DISALLOW_COPY(PlayLogGenerator);
    NN_DISALLOW_MOVE(PlayLogGenerator);
public:

    /**
    * @brief        インスタンスを取得します。
    */
    static PlayLogGenerator& GetInstance() NN_NOEXCEPT
    {
        static PlayLogGenerator instance;
        return instance;
    }

    /**
    * @brief        前回起動時の電源OFFイベントを。
    * @details      電源ONイベントが通知される前にこの関数を呼び前回起動時の電源OFFイベントを処理しておくことで、
    *               電源ONイベントが通知された際に OFF ~ ON までのプレイログを作成します。
    */
    void SetLastPowerOffTime(const PlayEvent& powerOffEvent) NN_NOEXCEPT;

    /**
    * @brief        アプレットに関するイベントを処理します。
    * @param[in]    eventType アプレットのイベントの種類。
    * @param[in]    applicationId イベントを発生させたアプレットのアプリケーションID。
    * @param[in]    appletId イベントを発生させたアプレットのアプレットID。
    * @param[in]    version イベントを発生させたアプレットのバージョン。
    * @param[in]    storageId イベントを発生させたアプレットの存在するストレージ。
    * @param[in]    time イベントの発生時刻情報。
    * @details      入力されたイベントがプレイログの起点となるイベント（Launch, Resume）の場合、そのイベントを内部に保存します。
    *               入力されたイベントがプレイログの終点となるイベント（Exit, Suspend）の場合、内部に保存されたイベントからその起点を探してプレイログを作成します。
    */
    void ProcessAppletEvent(AppletEventType eventType, const ncm::ApplicationId& applicationId, applet::AppletId appletId, uint32_t version, const ncm::StorageId& storageId, const PlayLogEntryPointTime& time) NN_NOEXCEPT;

    /**
    * @brief        アカウントオープンのイベントを処理します。
    * @param[in]    uid アカウントのユーザーID。
    * @param[in]    time イベントの発生時刻情報。
    * @details      アカウントオープンのイベントを内部に保存します。
    */
    void ProcessAccountOpenEvent(const account::Uid& uid, const PlayLogEntryPointTime& time) NN_NOEXCEPT;

    /**
    * @brief        アカウントクローズのイベントを処理します。
    * @param[in]    uid アカウントのユーザーID。
    * @param[in]    time イベントの発生時刻情報。
    * @details      アカウントのクローズが終点となるプレイログを作成します。
    */
    void ProcessAccountCloseEvent(const account::Uid& uid, const PlayLogEntryPointTime& time) NN_NOEXCEPT;

    /**
    * @brief        電源状態の変更のイベントを処理します。
    * @param[in]    eventType 電源状態。
    * @param[in]    time イベントの発生時刻情報。
    * @details      前回の電源状態の変更イベントを起点、今回の電源状態の変更のイベントを終点としてプレイログを作成します。
    */
    void ProcessPowerStateChangeEvent(PowerStateChangeEventType eventType, const PlayLogEntryPointTime& time) NN_NOEXCEPT;

    /**
    * @brief        内部に保持しているイベントを全て削除します。
    */
    void Clear() NN_NOEXCEPT;

    /**
    * @brief        fwdbg 設定を読み込んでプレイログの生成の有効・無効を設定します。
    * @details      fwdbg 設定の読み込みに失敗した場合はプレイログの生成を無効にします。
    */
    void SetEnabledFromFwdbgSettings() NN_NOEXCEPT;

    /**
    * @brief        プレイログの生成が有効かどうかを返します。
    * @return       プレイログの生成が有効かどうか。
    */
    bool IsEnabled() const NN_NOEXCEPT;

private:

    PlayLogGenerator() NN_NOEXCEPT;
    void ProcessAppletLaunchEvent(const ncm::ApplicationId applicationId, applet::AppletId appletId, uint32_t version, const ncm::StorageId storageId, const PlayLogEntryPointTime& time) NN_NOEXCEPT;
    void ProcessAppletInFocusEvent(const ncm::ApplicationId applicationId, applet::AppletId appletId, uint32_t version, const ncm::StorageId storageId, const PlayLogEntryPointTime& time) NN_NOEXCEPT;
    void ProcessAppletLostFocusEvent(const ncm::ApplicationId applicationId, applet::AppletId appletId, uint32_t version, const ncm::StorageId storageId, const PlayLogEntryPointTime& time) NN_NOEXCEPT;
    void ProcessAppletExitEvent(const ncm::ApplicationId applicationId, applet::AppletId appletId, uint32_t version, const ncm::StorageId storageId, const PlayLogEntryPointTime& time) NN_NOEXCEPT;

    /**
    * @brief        プレイログを保存します。
    * @param[in]    eventId プレイログの種類。
    * @param[in]    startPointTime プレイログの起点の時間に関する情報。
    * @param[in]    endPointTime プレイログの終点の時間に関する情報。
    * @param[in]    addKeyValueFunc プレイログ用のプレイレポートにキーバリューを追加する関数。
    * @details      全てのプレイログに共通のキーバリュー（下記参照）はこの関数によって追加されます。
    *               addKeyValueFunc では eventId に固有のキーバリューを追加してください。
    *               - アプリケーションID
    *               - 起点のユーザー時計時刻
    *               - 起点のネットワーク時計時刻
    *               - 終点 ～ 起点までの期間（秒）
    */
    void SavePlayLog(const char* eventId, const PlayLogEntryPointTime& startPointTime, const PlayLogEntryPointTime& endPointTime, std::function<void(prepo::SystemPlayReport* report)> addKeyValueFunc) const NN_NOEXCEPT;

    /**
    * @brief        アカウント情報に紐付くプレイログを保存します。
    * @param[in]    eventId プレイログの種類。
    * @param[in]    startPointTime プレイログの起点の時間に関する情報。
    * @param[in]    endPointTime プレイログの終点の時間に関する情報。
    * @param[in]    addKeyValueFunc プレイログ用のプレイレポートにキーバリューを追加する関数。
    * @param[in]    uid アカウント。
    * @details      アカウント情報に紐付く以外は、アカウント情報を持たないプレイログと同じです。
    */
    void SavePlayLog(const char* eventId, const PlayLogEntryPointTime& startPointTime, const PlayLogEntryPointTime& endPointTime, std::function<void(prepo::SystemPlayReport* report)> addKeyValueFunc, const account::Uid& uid) const NN_NOEXCEPT;

    /**
    * @brief        アプレットの生存期間（起動から終了まで）に関するプレイログを作成します。
    * @param[in]    appletLaunchPoint アプレットの起動に関する情報。
    * @param[in]    endPointTime アプレットの終了の時刻に関する情報。
    */
    void SaveAppletLifePlayLog(const PlayLogEntryStartPoint<AppletData>& appletLaunchPoint, const PlayLogEntryPointTime& endPointTime) const NN_NOEXCEPT;

    /**
    * @brief        アプレットのFG期間（再開から中断まで）に関するプレイログを作成します。
    * @param[in]    appletResumePoint アプレットの起動に関する情報。
    * @param[in]    endPointTime アプレットの中断の時刻に関する情報。
    */
    void SaveAppletForegroundPlayLog(const PlayLogEntryStartPoint<AppletData>& appletResumePoint, const PlayLogEntryPointTime& endPointTime) const NN_NOEXCEPT;

    /**
    * @brief        アカウントのオープン期間に関するプレイログを作成します。
    * @param[in]    accountOpenPoint アカウントのオープンに関する情報。
    * @param[in]    appletData アカウントをオープンしたアプリケーションに関する情報。
    * @param[in]    endPointTime アカウントのクローズの時刻に関する情報。
    */
    void SaveAccountOpenPlayLog(const PlayLogEntryStartPoint<account::Uid>& accountOpenPoint, const AppletData& appletData, const PlayLogEntryPointTime& endPointTime) const NN_NOEXCEPT;

    /**
    * @brief        アカウントがオープンした状態でのアプレットのFG期間に関するプレイログを作成します。
    * @param[in]    accountOpenPoint アカウントのオープンに関する情報。
    * @param[in]    appletResumePoint アプレットの再開に関する情報。
    * @param[in]    endPointTime レポートの終点の時刻に関する情報。アプレットのBG遷移もしくはアカウントのクローズ。
    * @details      アカウントのオープンとアプレットのFG遷移、後で発生したイベントを起点として
    *              「起点 ～ endPointTime までアカウントAがオープンした状態でアプレットXがFGにいた」という旨のプレイログを作成します。
    */
    void SaveAppletForegroundWithAccountPlayLog(const PlayLogEntryStartPoint<account::Uid>& accountOpenPoint, const PlayLogEntryStartPoint<AppletData>& appletResumePoint, const PlayLogEntryPointTime& endPointTime) const NN_NOEXCEPT;

    /**
    * @brief        電源状態に関するプレイログを作成します。
    * @param[in]    lastPowerStateChangePoint 前回の電源状態の変更に関する情報。
    * @param[in]    nextPowerState 次の電源状態。
    * @details      endPointTime 電源状態の変更の時刻に関する情報。
    */
    void SavePowerStatePlayLog(const PlayLogEntryStartPoint<PowerStateChangeEventType>& lastPowerStateChangePoint, PowerStateChangeEventType nextPowerState, const PlayLogEntryPointTime& endPointTime) const NN_NOEXCEPT;

    bool                                                m_IsEnabled;                                //!< プレイログの生成を有効にするかどうかのフラグ
    static const int                                    AliveAppletCountMax = 10;                   //!< 生存期間を同時に追跡するアプレットの数。想定上の最大（SA + A + LA * 2 + OA = 5）の 2倍確保。
    PlayLogEntryStartPoint<AppletData>                  m_AppletLaunchPoint[AliveAppletCountMax];   //!< アプレットの起動に関する情報。
    static const int                                    FocusedAppletCountMax = 4;                  //!< 同時にフォーカス状態になるアプレットの数。想定上の最大（何か + 非同期 swkbd = 2）の 2倍確保。
    PlayLogEntryStartPoint<AppletData>                  m_AppletInFocusPoint[FocusedAppletCountMax];//!< アプレットのフォーカス取得に関する情報。
    PlayLogEntryStartPoint<AppletData>                  m_LastApplicationLaunchEvent;               //!< 最後のアプリ起動イベント。アカウント系のレポート作成時にアプリの情報を挿入するために保存。
    PlayLogEntryStartPoint<account::Uid>                m_AccountOpenPoint[account::UserCountMax];  //!< アカウントのオープンに関する情報。
    PlayLogEntryStartPoint<PowerStateChangeEventType>   m_PowerStateChangePoint;                    //!< 電源状態の変更に関する情報。
    static const size_t                                 PrepoBufferSize = prepo::SystemPlayReport::BufferSizeMin + prepo::KeyValueSizeMax * 8;
    mutable char*                                       m_PrepoBuffer[PrepoBufferSize];             //!< プレイレポート作成用のバッファ。
    mutable os::SdkMutex                                m_ProcessEventMutex;                        //!< イベント処理の排他制御用の Mutex。
};

}}}
