﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/pdm/detail/pdm_EventEntryStream.h>

namespace nn { namespace pdm { namespace detail {

/**
 * @brief   アカウントデータベース移行用コンテキスト。
 */
typedef SimpleEntryRecorder<AccountPlayEvent> MigrationRecorder;

/**
 * @brief   アカウントデータベース移行状態定数。
 */
enum class MigrationState : Bit8
{
    None = 0x00,
    Done = 0xff,
};

/**
 * @brief   参照カウンタによる利用可否管理クラス
 *
 * @details 参照カウンタによるネスト管理を行っています。@n
 *          期待するネスト基準状態(参照カウンタがゼロの状態)は「利用可能」です。つまり、「利用停止( @ref Suspend() )」で +1, 「利用再開( @ref Resume() )」で -1 されます。@n
 *          尚、参照カウンタのクラス初期化値は 1 にしています。これは初期状態が「利用停止」から始まる事を想定しています。@n
 *          ゼロ状態(利用可能状態)からの「利用再開」要求はネストされません。@n
 *          非スレッドセーフです。
 */
class AvailableCondition
{
public:
    typedef int32_t ReferenceCounter;
    static const ReferenceCounter InitialReferenceCount = 1;

    /**
     * @brief       コンストラクタ。
     */
    explicit AvailableCondition(ReferenceCounter initialReferenceCount = InitialReferenceCount) NN_NOEXCEPT : m_EnsureCount(initialReferenceCount) {}

    /**
     * @brief       インスタンスを初期化して利用可能にします。
     *
     * @param[in]   initialReferenceCount   初期参照カウンタ値。
     */
    void Initialize(ReferenceCounter initialReferenceCount = InitialReferenceCount) NN_NOEXCEPT
    {
        m_EnsureCount = initialReferenceCount;
    }

    /**
     * @brief       利用停止要求を行います。
     *
     * @retresult
     *      @handleresult   {nn::ResultSuccess}
     *      @handleresult   {nn::pdm::ResultNotExecutedByWhileNesting}
     * @endresult
     */
    Result Suspend(bool ignoreNestCondition) NN_NOEXCEPT;

    /**
     * @brief       利用再開要求を行います。
     *
     * @retresult
     *      @handleresult   {nn::ResultSuccess}
     *      @handleresult   {nn::pdm::ResultNotExecutedByWhileNesting}
     * @endresult
     */
    Result Resume() NN_NOEXCEPT;

    /**
     * @brief       利用可能かどうか確認します。
     */
    NN_FORCEINLINE bool IsAvailable() const NN_NOEXCEPT
    {
        return 0 == m_EnsureCount;
    }

private:
    ReferenceCounter    m_EnsureCount;  //!< 再開/停止要求カウント状態
};


/**
 * @brief   マウントハンドル
 *
 * @details 非スレッドセーフです。AccountPlayEventBuffer のストリーム制御に紐づいて排他される想定。
 *          マウント要求のネスト管理されません。
 */
class AccountSaveMountHandle
{
public:
    static const int MountVolumeNameCapacity = fs::MountNameLengthMax + 1;    //!< マウントボリューム名上限( 15文字 + null終端 )

    /**
     * @brief       コンストラクタ。
     */
    AccountSaveMountHandle() NN_NOEXCEPT;

    /**
     * @brief       指定マウントインデクスでハンドルを初期化して利用可能にします。
     *
     * @param[in]   mountIndex  他のハンドルと競合しないユニークなインデクス。
     *
     *  @details    インデクスに応じたマウントボリューム名を生成してマウント管理に用います。
     */
    void Initialize(uint32_t mountIndex) NN_NOEXCEPT;

    /**
     * @brief       マウント管理対象のアカウントUIDを割り当てます。
     *
     * @param[in]   userId  管理対象アカウントUID。
     *
     * @pre
     *              true == static_cast<bool>(userId)
     */
    Result Activate(const account::Uid& userId) NN_NOEXCEPT;

    /**
     * @brief       管理対象アカウント情報を破棄します。
     *
     * @param[in]   withDeleteSaveData true の場合、管理対象アカウントのセーブデータが存在する場合、対象セーブデータの削除を実施します。
     *
     * @details     引数 @ref withDeleteSaveData の指定によって、管理対象アカウントのセーブデータが存在する場合、対象セーブデータの削除を実施する可能性があります。
     */
    void Deactivate(bool withDeleteSaveData) NN_NOEXCEPT;

    /**
     * @brief       マウント要求。
     *
     * @param[in]   isAlreadySaveDataMountOnly  true の場合、セーブデータが既存する時のみマウントします。@n
     *                                          false の場合、セーブデータが存在しない場合、セーブデータを作成してマウントします。
     *
     * @retresult
     *      @handleresult   {nn::ResultSuccess}
     *      @handleresult   {nn::pdm::ResultInvalidUserId}
     *      @handleresult   {nn::pdm::ResultUserNotExist}
     *      @handleresult   {nn::pdm::ResultDatabaseStorageLocked}
     *      @handleresult   {nn::pdm::ResultDatabaseStorageFailure}
     * @endresult
     *
     * @details     管理対象アカウントのセーブデータへのマウントを試みます。@n
     *              対象セーブデータが存在しない場合は @ref isAlreadySaveDataMountOnly 引数の内容に伴った挙動を行います。
     */
    Result Mount(bool isAlreadySaveDataMountOnly) NN_NOEXCEPT;

    /**
     * @brief       アンマウント要求。
     *
     * @retresult
     *      @handleresult   {nn::ResultSuccess}
     * @endresult
     */
    Result Unmount() NN_NOEXCEPT;

    /**
     * @brief       コミット要求。
     *
     * @retresult
     *      @handleresult   {nn::ResultSuccess}
     * @endresult
     */
    Result Commit() NN_NOEXCEPT;

    inline const account::Uid& GetUid() const NN_NOEXCEPT
    {
        return m_Uid;
    }

    inline const char* GetVolumeName() const NN_NOEXCEPT
    {
        return m_VolumeName;
    }

    inline bool IsMounted() const NN_NOEXCEPT
    {
        return m_IsMounted;
    }

private:
    /**
     * @brief       指定UIDのユーザーが存在するかチェック。
     *
     * @retresult
     *      @handleresult   {nn::pdm::ResultInvalidUserId}
     *      @handleresult   {nn::pdm::ResultUserNotExist}
     * @endresult
     */
    Result VerifyUserExistence(const account::Uid& uid) const NN_NOEXCEPT;

    account::Uid        m_Uid;                                  //!< マウント対象ユーザーアカウント
    char                m_VolumeName[MountVolumeNameCapacity];  //!< マウントボリューム名
    bool                m_IsMounted;                            //!< マウント状態値
};






/**
 * @brief   AccountPlayEvent の書き込み・読み込みを管理するクラスです。
 *
 * @details 運用用途想定として、最大ユーザー分( 8 )のインスタンスを初回生成して維持します。@n
 *          状態として Activate / Idle / Suspend があります。@n
 *          Activate    : @ref Activate() により稼働状態になっている。@n
 *                      : 状態開始契機@n
 *                          - pdm起動時@n
 *                          - ユーザーアカウント登録時@n
 *                      : 状態終了契機@n
 *                          - Idle 状態開始契機に基づく。@n
 *          Idle        : @ref Deactivate() により非稼働状態になっている。@n
 *                      : 状態開始契機@n
 *                          - ユーザーアカウント削除時@n
 *                      : 状態終了契機@n
 *                          - Activate 状態開始契機に基づく。@n
 *          Suspend     : @ref Suspend() により一時停止状態になっている。再開には @ref Resume() を行う。@n
 *                      : 状態開始契機@n
 *                          - サービス停止APIコール時@n
 *                      : 状態終了契機@n
 *                          - サービス再開APIコール時@n
 */
class AccountPlayEventBuffer : public EventEntryFileStream
{
    NN_DISALLOW_COPY(AccountPlayEventBuffer);
    NN_DISALLOW_MOVE(AccountPlayEventBuffer);

public:
    typedef AccountPlayEvent::PowerChange PowerChange;
    typedef EventEntryCache<AccountPlayEvent>  Buffer;
    typedef DisallowCopyTuple<LockGuard, LockGuard, LockGuard> LockGuardFull;

    /**
     * @brief   @ref AccountPlayEventBuffer::AcquireRead() 用エイリアスミューテックス.
     */
    class ScopedReadMutex
    {
        NN_DISALLOW_COPY(ScopedReadMutex);
        NN_DISALLOW_MOVE(ScopedReadMutex);

    public:
        explicit ScopedReadMutex(AccountPlayEventBuffer* pOwner) NN_NOEXCEPT
            : m_pOwner(pOwner) {}

        void lock() NN_NOEXCEPT
        {
            m_pOwner->BeginRead();
        }

        void unlock() NN_NOEXCEPT
        {
            m_pOwner->EndRead();
        }

    private:
        AccountPlayEventBuffer* const m_pOwner;
    };
    typedef util::UniqueLock<ScopedReadMutex> ScopedReadLock;

    /**
     * @brief   プレイレポート用のデータ。
     */
    struct ReportData
    {
        account::Uid uid;
        uint32_t count;
        uint32_t startIndex;
        uint32_t lastIndex;
    };

    /**
     * @brief       コンストラクタです。
     */
    AccountPlayEventBuffer() NN_NOEXCEPT;

    /**
     * @brief       バッファインスタンスの初期化を行います。
     */
    void Initialize(uint32_t mountIndex) NN_NOEXCEPT;

    /**
     * @brief       バッファストリームを開きます。
     *
     * @param[in]   uid             割り当てる管理対象アカウントUID。
     * @param[in]   migrationState  ストリームヘッダに保存するデバイスDBマイグレーションの完了情報。
     * @param[in]   withOpenStream  遅延ストリームオープン機能の選択オプションです。@n
     *                              遅延ストリームオープンを利用する場合は false を指定します。
     *
     * @details     以前に割り当てた管理対象アカウントUIDが有効な値の場合、@ref Deactivate() を呼び出します。@n
     *              新規の管理対象UIDが有効な場合は、管理対象アカウントのセーブデータへのマウントを試みます。@n
     *              対象セーブデータが存在しない場合は新規に作成してからマウントします。@n
     *              ユーザーアカウントの作成時に呼び出される事を想定しています。
     */
    Result Activate(const account::Uid& uid, const MigrationState migrationState = MigrationState::Done, bool withOpenStream = false) NN_NOEXCEPT;

    /**
     * @brief       バッファストリームを閉じます。
     *
     * @details     @ref Activate() で指定された管理対象アカウントのセーブデータを強制的にアンマウントして、対象セーブデータの削除を実施します。@n
     *              ユーザーアカウントの削除時に呼び出される事を想定しています。
     */
    void Deactivate() NN_NOEXCEPT;

    /**
     * @brief       バッファストリームの保存対象ストレージファイルを削除します。
     *
     * @details     内部用です。
     */
    Result Discard() NN_NOEXCEPT;

    /**
     * @brief       バッファストリームを一時停止します。
     */
    Result Suspend() NN_NOEXCEPT;

    /**
     * @brief       バッファストリームを再開します。
     */
    Result Resume() NN_NOEXCEPT;

    /**
     * @brief       データベース読み込み用スコープロックの取得。
     *
     * @retresult
     *      @handleresult   {nn::ResultSuccess}
     *      @handleresult   {nn::pdm::ResultServiceNotAvailable}
     * @endresult
     */
    Result AcquireRead(ScopedReadLock* pOutScopedLock) NN_NOEXCEPT;

    /**
     * @brief       データベース読み込み。
     *
     * @param[out]  pOutValue   読み込み先。コンストラクタで指定したエントリ単位サイズを１ベントとして outCountMax 分受け取れる容量が必要です。
     * @param[in]   startIndex  読み込みを開始するイベントのインデックス。
     * @param[in]   outCountMax 読み込むイベントの最大数。
     *
     * @return      読み込んだイベントの数。
     *
     * @pre
     *              - outCountMax > 0
     *              - startIndex > @ref GetStartIndex()
     *              - pOutValue != nullptr
     *              - pOutValue >= sizeof(AccountPlayEvent) * outCountMax
     *              - @ref AcquireRead() を呼び出し、有効な ManagedRead のスコープ内で実施している。
     */
    inline uint32_t Read(AccountPlayEvent pOutValue[], uint32_t startIndex, uint32_t outCountMax) const NN_NOEXCEPT
    {
        return EventEntryFileStream::ReadImpl(pOutValue, startIndex, outCountMax);
    }

    /**
     * @brief       @ref PlayStatistics 情報を問い合わせます。
     *
     * @param[out]  pOutValue           問い合わせ結果出力バッファ
     * @param[in]   targetApplicationId 問い合わせ対象のアプリケーションID
     * @param[in]   pReadTemporary      データベース読み込み用一時記憶メモリ領域の先頭アドレス
     * @param[in]   byteOfTemporary     データベース読み込み用一時記憶メモリ領域の容量( byte 単位 )
     *
     * @retresult
     *      @handleresult   {nn::ResultSuccess}
     *      @handleresult   {nn::pdm::ResultServiceNotAvailable}
     * @endresult
     *
     * @pre
     *              - pOutValue != nullptr
     *
     * @details     内部で @ref AcquireRead() と @ref Read() を用います。
     */
    Result Query(PlayStatistics* pOutValue, const ncm::ApplicationId& targetApplicationId, Bit8* const pReadTemporary, uint32_t byteOfTemporary) NN_NOEXCEPT;

    /**
     * @brief       @ref PlayStatistics 情報を問い合わせます。
     *              ユーザーアカウントに紐づくアプリケーションを全て照合します。
     *
     * @param[out]  pOutCount           問い合わせ結果の有効検出数
     * @param[out]  pOutValues          問い合わせ結果出力バッファ
     * @param[in]   receiveCapacity     問い合わせ結果出力バッファ容量( PlayStatistics単位 )
     * @param[in]   pReadTemporary      データベース読み込み用一時記憶メモリ領域の先頭アドレス
     * @param[in]   byteOfTemporary     データベース読み込み用一時記憶メモリ領域の容量( byte 単位 )
     *
     * @retresult
     *      @handleresult   {nn::ResultSuccess}
     *      @handleresult   {nn::pdm::ResultServiceNotAvailable}
     * @endresult
     *
     * @pre
     *              - pOutCount != nullptr
     *              - pOutValues != nullptr
     *              - pOutValues > receiveCapacity * sizeof(PlayStatistics)
     *              - receiveCapacity > 0
     *
     * @details     内部で @ref AcquireRead() と @ref Read() を用います。
     */
    Result Query(uint32_t* pOutCount, PlayStatistics* pOutValues, uint32_t receiveCapacity, Bit8* const pReadTemporary, uint32_t byteOfTemporary) NN_NOEXCEPT;

    /**
     * @brief       @ref AccountEvent 情報を問い合わせます。
     *
     * @param[out]  pOutCount           問い合わせ結果の有効検出数
     * @param[out]  pOutValues          問い合わせ結果出力バッファ
     * @param[in]   receiveCapacity     問い合わせ結果出力バッファ容量( AccountEvent単位 )
     * @param[in]   eventIndexOffset    取得を開始するイベント全体の先頭からのオフセット。取得対象に含まれます。
     * @param[in]   pReadTemporary      データベース読み込み用一時記憶メモリ領域の先頭アドレス
     * @param[in]   byteOfTemporary     データベース読み込み用一時記憶メモリ領域の容量( byte 単位 )
     *
     * @retresult
     *      @handleresult   {nn::ResultSuccess}
     *      @handleresult   {nn::pdm::ResultServiceNotAvailable}
     * @endresult
     *
     * @pre
     *              - pOutCount != nullptr
     *              - pOutValues != nullptr
     *              - pOutValues > receiveCapacity * sizeof(AccountEvent)
     *              - receiveCapacity > 0
     *
     * @details     システムが記録しているイベント全体を eventIndexOffset 番目から走査し、アカウントのオープン・クローズのイベントを出力します。@n
     *              システムはアカウントのオープン・クローズ以外のイベントも記録しており、eventIndexOffset はそれらを含めた全てのイベントの列の中での値です。@n
     *              出力のイベントのインデックスも同様に、全てのイベントを通してのインデックス値であるため、連続的な値にはなりません。@n
     *              内部で @ref AcquireRead() と @ref Read() を用います。
     */
    Result Query(uint32_t* pOutCount, AccountEvent* pOutValues, uint32_t receiveCapacity, uint32_t eventIndexOffset, Bit8* const pReadTemporary, uint32_t byteOfTemporary) NN_NOEXCEPT;

    /**
     * @brief       最近遊んだアプリケーションのアプリケーションIDを問い合わせます。
     * @param[out]  pOutCount           問い合わせ結果の有効検出数
     * @param[out]  pOutValues          問い合わせ結果出力バッファ
     * @param[in]   receiveCapacity     問い合わせ結果出力バッファ容量( ncm::ApplicationId単位 )
     * @param[in]   presetCount         pOutValues に予め格納されているアプリケーションIDの数
     * @param[in]   pReadTemporary      データベース読み込み用一時記憶メモリ領域の先頭アドレス
     * @param[in]   byteOfTemporary     データベース読み込み用一時記憶メモリ領域の容量( byte 単位 )
     *
     * @retresult
     *      @handleresult   {nn::ResultSuccess}
     *      @handleresult   {nn::pdm::ResultServiceNotAvailable}
     * @endresult
     *
     * @pre
     *              - pOutCount != nullptr
     *              - pOutValues != nullptr
     *              - receiveCapacity > 0
     */
    Result QueryRecentlyPlayedApplication(uint32_t* pOutCount, ncm::ApplicationId* pOutValues, uint32_t receiveCapacity, uint32_t presetCount, Bit8* const pReadTemporary, uint32_t byteOfTemporary) NN_NOEXCEPT;

    /**
     * @brief       イベントをバッファに追加します。
     *
     * @param[in]   eventEntry          バッファに追加するイベント。@n
     *                                  入力された構造体メンバ powerChange を管理中の PowerChange 検出状態で上書きします。
     * @param[in]   checkAvailability   追加対象イベントが所属するユーザーアカウント。@n
     *                                  イベント追加時にバッファ対象のアカウントが期待するアカウントであるかをチェックします。@n
     *                                  アカウント相違の場合は追加イベントは無視されます。@n
     *                                  account::InvalidUid を指定した場合はチェックされません。
     *
     * @return      追加に成功した場合は true, false の場合は、アカウントチェックの失敗を示します。
     *
     * @details     バッファに追加されたイベントは @ref Flush() が実行されるまでファイルには書き込まれず、@ref Read() の対象になりません。
     *              Get 系の関数で得られるインデックス値やカウントにも反映されません。@n
     *              イベント追加用のバッファに空きがない場合は、関数内部でファイルへのフラッシュ処理を行い、
     *              空き領域を確保して追加を行います。
     */
    bool Add(AccountPlayEvent& eventEntry, const account::Uid& checkAvailability) NN_NOEXCEPT;

    /**
     * @brief       バッファに追加されたイベントをファイルに書き込みフラッシュします。@n
     *              フラッシュされた場合、イベント追加用のバッファは空になります。@n
     *              超過率条件を満たさない場合は残ります。
     *
     * @param[in]   excessRate  100 分率で表現された超過率です。ゼロ指定は常に書き込みます。@n
     *                          指定率を超過時にのみ Flush を実施します。
     */
    void Flush(const uint32_t excessRate = 0) NN_NOEXCEPT;

    /**
     * @brief       バッファ中にある未フラッシュのイベントの数を取得します。
     * @return      イベントの数。
     */
    uint32_t GetFilledBufferSpaceCount() const NN_NOEXCEPT;

    /**
     * @brief       バッファに溜められるイベントの最大数を取得します。
     * @return      イベントの最大数。
     */
    NN_FORCEINLINE static uint32_t GetBufferSpaceCountMax() NN_NOEXCEPT
    {
        return Buffer::Capacity;
    }

    /**
     * @brief       Idle(deactivate) 状態か確認します。
     */
    bool IsIdle() const NN_NOEXCEPT;

    /**
     * @brief       Suspend 状態か確認します。
     */
    bool IsSuspended() const NN_NOEXCEPT;

    /**
     * @brief       Suspend or Idle 状態か確認します。
     */
    bool IsUnavailable() const NN_NOEXCEPT;

    /**
     * @brief       プレイレポート用のデータを返します。
     */
    Result GetReportData(ReportData* pOut) const NN_NOEXCEPT;

    /**
     * @brief       指定UIDリストに自身が存在するか確認します。
     *
     * @return      存在する場合、配列インデクス, 存在しない場合は、引数に指定した @ref count が返されます。@n
     *              バッファストリームがIdle状態場合は負値(-1)が返されます。
     */
    int Find(const account::Uid* pUsers, int count) const NN_NOEXCEPT;

    /**
     * @brief       PowerState の変更を通知します。
     */
    inline void NotifyPowerChange(const PowerChange& changeState) NN_NOEXCEPT
    {
        m_DetectedPowerChange = changeState;
    }

    inline LockGuardFull AcquireLockFull() const NN_NOEXCEPT
    {
        // 呼び出し順は Flush() 時と同じにしてください。
        return LockGuardFull(
            m_EntryCache.AcquireLockForAdded(),
            m_EntryCache.AcquireLockForFlush(),
            AcquireLockForFile()
        );
    }

    //!------------------------------------------------------------------------
    //! @name Debug インタフェース
    //@{

    Result QueryAvailableRange(uint32_t* pOutCount, uint32_t* pOutStartIndex, uint32_t* pOutLastIndex) NN_NOEXCEPT;

    Result QueryRaw(uint32_t* pOutCount, AccountPlayEvent* pOutValues, uint32_t receiveCapacity, uint32_t eventIndexOffset) NN_NOEXCEPT;

    //@}
    //!------------------------------------------------------------------------

    //!------------------------------------------------------------------------
    //! @name   Migration ( デバイスDB → アカウントDB 変換 )関連
    //!         高速化のためにバッファ有効性確認は「Uid の割り当て( idle チェック )」のみです。Suspend 状態の確認はしません。
    //!         非スレッドセーフです。
    //@{

    void PrepareMigration(MigrationRecorder* pRecorder) NN_NOEXCEPT;

    bool ImportMigration(MigrationRecorder* pRecorder, AccountPlayEvent& entry, const account::Uid& checkAvailability) NN_NOEXCEPT;

    void FinishMigration(MigrationRecorder* pRecorder) NN_NOEXCEPT;

    NN_FORCEINLINE bool IsCompleteMigration() const NN_NOEXCEPT
    {
        return (MigrationState::Done == m_CustomHeader.migration);
    }

    NN_FORCEINLINE void DiscardCompleteMigration() NN_NOEXCEPT
    {
        m_CustomHeader.migration = MigrationState::None;
    }

    //@}
    //!------------------------------------------------------------------------

protected:
    virtual Result OnHeaderRead(fs::FileHandle fileHandle, int64_t position, size_t* pOutReadSize) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result OnHeaderWrite(fs::FileHandle fileHandle, int64_t position, const fs::WriteOption inheritedOption, size_t* pOutWriteSize) NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnHeaderReset() NN_NOEXCEPT NN_OVERRIDE;

    virtual void BeginRead() NN_NOEXCEPT NN_OVERRIDE;
    virtual void EndRead() NN_NOEXCEPT NN_OVERRIDE;

private:
    struct CustomHeader
    {
        static const uint32_t CurrentVersion = 0;
        const uint32_t  magic;      //!< ヘッダ判別情報なので @ref Reset() では変化しない。
        const uint32_t  version;    //!< ヘッダ書式バージョン情報なので @ref Reset() では変化しない。
        MigrationState  migration;  //!< 起動時移行完了情報なので @ref Reset() では変化しない。
        Bit8            reserved[sizeof(Bit32) - sizeof(MigrationState)];

        CustomHeader() NN_NOEXCEPT : magic(0xDEADC0DE), version(CurrentVersion), migration(MigrationState::Done)
        {
            std::fill_n(reserved, NN_ARRAY_SIZE(reserved), static_cast<Bit8>(0));
        }

        inline void Reset() NN_NOEXCEPT {}
    };

    /**
     * @brief       Idle(deactivate) 状態か確認します。
     */
    inline bool IsIdleUnsafe() const NN_NOEXCEPT
    {
        return !m_MountHandle.GetUid();
    }

    /**
     * @brief       Suspend 状態か確認します。
     */
    inline bool IsSuspendedUnsafe() const NN_NOEXCEPT
    {
        return !m_ServiceCondition.IsAvailable();
    }

    /**
     * @brief       Suspend or Idle 状態か確認します。
     */
    inline bool IsUnavailableUnsafe() const NN_NOEXCEPT
    {
        return (IsSuspendedUnsafe() || false == m_MountHandle.GetUid());
    }

    inline const PowerChange GetDetectedPowerChange() NN_NOEXCEPT
    {
        const auto result = m_DetectedPowerChange;
        m_DetectedPowerChange = PowerChange::None;
        return result;
    }

    void    FlushImpl(const uint32_t excessRate = 0) NN_NOEXCEPT;
    Result  SuspendImpl(bool ignoreNestCondition) NN_NOEXCEPT;

    bool    DiscardUnsafe() NN_NOEXCEPT;
    Result  DeactivateUnsafe() NN_NOEXCEPT;
    Result  PrepareStreamUnsafe(bool isAlreadyExistDataOnly) NN_NOEXCEPT;

    AccountSaveMountHandle  m_MountHandle;
    Buffer::Switcher        m_EntryCache;
    CustomHeader            m_CustomHeader;
    ScopedReadMutex         m_ScopedReadLock;
    ClientIdleTracker       m_ClientIdleSync;
    const uint32_t          m_EventCountMax;
    AvailableCondition      m_ServiceCondition;
    PowerChange             m_DetectedPowerChange;
};





/**
 * @brief   ユーザーアカウントイベントレコード生成ファクトリ
 */
class AccountPlayEventFactory
{
    NN_DISALLOW_COPY(AccountPlayEventFactory);
    NN_DISALLOW_MOVE(AccountPlayEventFactory);

public:
    typedef AccountPlayEvent::RecordCause   RecordCause;

    static const int NestAppletCapacity = 4;    //!< 同時にフォーカス状態になるアプレットの数。想定上の最大（何か + 非同期 swkbd = 2）の 2倍確保。

    template<typename TData>
    struct RecordEntry
    {
        AccountPlayEvent::Time  time;   //!< イベント発生時の時刻
        int64_t                 steadyClockValue; // イベント発生時の単調増加時計の時刻
        TData                   data;   //!< レコード Entity

        NN_IMPLICIT RecordEntry() NN_NOEXCEPT
        {
            Invalidate();
        }

        void Invalidate() NN_NOEXCEPT
        {
            time.Invalidate();
            data.Invalidate();
            steadyClockValue = 0;
        }

        inline bool IsValid() const NN_NOEXCEPT
        {
            return time.IsValid() && data.IsValid();
        }

        void ApplyTime(const PlayEvent& source) NN_NOEXCEPT
        {
            time.Apply(source);
            steadyClockValue = source.steadyTime;
        }
    };

    struct Condition
    {
        typedef AccountPlayEvent::PackedId64    PackedId64;
        typedef AccountPlayEvent::AppletId8     AppletId8;

        struct Applet : public AccountPlayEvent::Applet
        {
            AppletEventType     type;   //!< イベント発生時のイベント種別

            NN_IMPLICIT Applet() NN_NOEXCEPT : AccountPlayEvent::Applet() {}

            explicit Applet(const AppletEventData& source) NN_NOEXCEPT
            {
                Apply(source);
            }

            NN_FORCEINLINE void Apply(const AppletEventData& source) NN_NOEXCEPT
            {
                type = source.eventType;
                AccountPlayEvent::Applet::Apply(source);
            }

            NN_FORCEINLINE bool IsInFocused() const NN_NOEXCEPT
            {
                return (AppletEventType::InFocus == type);
            }

            NN_FORCEINLINE bool IsValid() const NN_NOEXCEPT
            {
                return (PackedId64::InvalidValue != programId);
            }

            NN_FORCEINLINE bool Equals(const AppletEventData& source) const NN_NOEXCEPT
            {
                return (programId.hi == source.programIdHi && programId.low == source.programIdLow && appletId == source.appletId);
            }
        };

        struct UserAccount : public AccountPlayEvent::Applet
        {
            account::Uid        userId;         //!< ユーザーアカウントID.

            NN_IMPLICIT UserAccount() NN_NOEXCEPT : AccountPlayEvent::Applet(), userId(account::InvalidUid) {}

            void Invalidate() NN_NOEXCEPT
            {
                userId = account::InvalidUid;
                AccountPlayEvent::Applet::Invalidate();
            }

            inline bool IsValid() const NN_NOEXCEPT
            {
                return static_cast<bool>(userId);
            }
        };

        struct DockMode
        {
            pdm::OperationMode  value;

            NN_IMPLICIT DockMode() NN_NOEXCEPT : value(pdm::OperationMode::Handheld) {}

            void Invalidate() NN_NOEXCEPT
            {
                value = pdm::OperationMode::Handheld;
            }

            inline bool IsValid() const NN_NOEXCEPT
            {
                return true;
            }
        };

        // アプレットイベントに際してオープン中アカウントに対してイベント送信だから、applet は共通化。
        // nsa と account がユーザー単位でインスタンス化。
        RecordEntry<UserAccount>        open[account::UserCountMax];    //!< Open時は != InvalidUid.
        RecordEntry<Applet>             applet[NestAppletCapacity];     //!< フォーカス状態アプレット.
        RecordEntry<DockMode>           dock;
        RecordEntry<Applet>             application;                    //!< 起動状態アプリケーション.

        void Invalidate() NN_NOEXCEPT;
        bool ApplyActiveApplication(AccountPlayEvent::Applet* pOutValue, const AccountPlayEvent::RecordCause& cause) NN_NOEXCEPT;
        bool ReceiveOpenAccount(const PlayEvent& source) NN_NOEXCEPT;
        bool ReceiveCloseAccount(const PlayEvent& source, RecordEntry<UserAccount>* pOutRecordPreOpenedEntry) NN_NOEXCEPT;
        bool ReceiveApplet(const PlayEvent& source, RecordEntry<Applet>* pOutRecordPreFocusedEntry) NN_NOEXCEPT;
    };

protected:
    AccountPlayEventFactory() NN_NOEXCEPT {}
    void ImportEventUnsafe(const PlayEvent& source) NN_NOEXCEPT;

    virtual void OnReceivePowerOn() NN_NOEXCEPT = 0;
    virtual void OnReceivePowerOff() NN_NOEXCEPT = 0;
    virtual void OnReceiveEventCreated(const account::Uid& uid, AccountPlayEvent& newEvent) NN_NOEXCEPT = 0;

    Condition& GetCondition() NN_NOEXCEPT;

private:
    void ImportEventForApplet(const PlayEvent& source) NN_NOEXCEPT;
    void ImportEventForUserAccount(const PlayEvent& source) NN_NOEXCEPT;
    void ImportEventForPowerStateChange(const PlayEvent& source) NN_NOEXCEPT;
    void ImportEventForOperationModeChange(const PlayEvent& source) NN_NOEXCEPT;

    bool MakeEventAccountOpenPeriod(AccountPlayEvent* pOutEvent, const RecordEntry<Condition::UserAccount>& openedEntry, const RecordCause& recordCause, int64_t closedSteadyClockValue) NN_NOEXCEPT;
    void PublishEventAtAccountClose(const PlayEvent& occurEvent, const RecordEntry<Condition::UserAccount>& openedEntry, const RecordCause& recordCause, const account::Uid& uid) NN_NOEXCEPT;

    Condition   m_Condition;
};

//!---------------------------------------------------------------------------
inline bool operator ==(const AccountPlayEventFactory::Condition::Applet& lhs, const AccountPlayEventFactory::Condition::Applet& rhs) NN_NOEXCEPT
{
    return (lhs.programId == rhs.programId && lhs.appletId == rhs.appletId);
}

inline bool operator !=(const AccountPlayEventFactory::Condition::Applet& lhs, const AccountPlayEventFactory::Condition::Applet& rhs) NN_NOEXCEPT
{
    return !(lhs == rhs);
}





/**
 * @brief   ユーザーアカウントデータベースサービスプロバイダ
 */
class AccountPlayEventProvider : protected AccountPlayEventFactory
{
    NN_DISALLOW_COPY(AccountPlayEventProvider);
    NN_DISALLOW_MOVE(AccountPlayEventProvider);

public:
    typedef int8_t BufferIndex;
    static const int BufferCapacity = account::UserCountMax;
    static const BufferIndex BufferIndexCountMax = static_cast<BufferIndex>(BufferCapacity);

    /**
     * @brief   シングルトンインスタンスを取得します。
     */
    static AccountPlayEventProvider& GetInstance() NN_NOEXCEPT
    {
        static AccountPlayEventProvider instance;
        return instance;
    }

    /**
     * @brief   ユーザーアカウントデータベースサービスプロバイダを初期化します。
     *
     * @details @ref pBootUsers で指定された初期登録済ユーザーアカウントのデータベースストリームを @ref AccountPlayEventBuffer::Activate() します。
     */
    bool Initialize(const account::Uid* pBootUsers, int userCount, const MigrationState initialMigrationState) NN_NOEXCEPT;

    /**
     * @brief   ユーザーアカウント登録状態の変動に併せて、@ref AccountPlayEventBuffer の再構築を行います。
     *
     * @details 既に管理中だが、@ref pNewUsers に存在しないユーザーアカウントは、@ref AccountPlayEventBuffer::Deactivate() されます。@n
     *          サービスプロバイダが管理していない @ref pNewUsers に存在するユーザーアカウントは、@n
     *          未使用の @ref AccountPlayEventBuffer を割り当て、@ref AccountPlayEventBuffer::Activate() されます。
     */
    void UpdateUserRegistry(const account::Uid* pNewUsers, int userCount) NN_NOEXCEPT;

    /**
     * @brief   デバイスDBのイベント一つをアカウントDBにインポートします。
     *
     * @details インポートされたイベントは、@ref AccountPlayEventFactory でシャドウコンディションを構築しながら、@n
     *          記録単位( @ref AccountPlayEvent )の作成が可能になった時点で、アカウントDBへ追加要求が行われます。
     */
    void Import(const PlayEvent& source) NN_NOEXCEPT;

    /**
     * @brief   指定ユーザーアカウントのイベント記録/参照サービスを再開します。
     */
    Result Resume(const account::Uid& uid) NN_NOEXCEPT;

    /**
     * @brief   指定ユーザーアカウントのイベント記録/参照サービスを一時停止します。
     */
    Result Suspend(const account::Uid& uid) NN_NOEXCEPT;

    /**
     * @brief       @ref PlayStatistics 情報を問い合わせます。
     *
     * @param[out]  pOutValue       問い合わせ結果出力バッファ
     * @param[in]   id              問い合わせ対象のアプリケーションID
     * @param[in]   pReadTemporary  データベース読み込み用一時記憶メモリ領域の先頭アドレス
     * @param[in]   byteOfTemporary データベース読み込み用一時記憶メモリ領域の容量( byte 単位 )
     * @param[in]   uid             問い合わせ対象のユーザーアカウントID
     *
     * @retresult
     *      @handleresult   {nn::ResultSuccess}
     *      @handleresult   {nn::pdm::ResultUserNotExist}
     *      @handleresult   {nn::pdm::ResultServiceNotAvailable}
     * @endresult
     *
     * @pre
     *              - pOutValue != nullptr
     *
     */
    Result Query(PlayStatistics* pOutValue, const ncm::ApplicationId& id,
        Bit8* const pReadTemporary, uint32_t byteOfTemporary,
        const account::Uid& uid) NN_NOEXCEPT;

    /**
     * @brief       @ref PlayStatistics 情報を問い合わせます。
     *              ユーザーアカウントに紐づくアプリケーションを全て照合します。
     *
     * @param[out]  pOutCount       問い合わせ結果の有効検出数
     * @param[out]  pOutValues      問い合わせ結果出力バッファ
     * @param[in]   receiveCapacity 問い合わせ結果出力バッファ容量( PlayStatistics単位 )
     * @param[in]   pReadTemporary  データベース読み込み用一時記憶メモリ領域の先頭アドレス
     * @param[in]   byteOfTemporary データベース読み込み用一時記憶メモリ領域の容量( byte 単位 )
     * @param[in]   uid             問い合わせ対象のユーザーアカウントID
     *
     * @retresult
     *      @handleresult   {nn::ResultSuccess}
     *      @handleresult   {nn::pdm::ResultUserNotExist}
     *      @handleresult   {nn::pdm::ResultServiceNotAvailable}
     * @endresult
     *
     * @pre
     *              - pOutCount != nullptr
     *              - pOutValues != nullptr
     *              - pOutValues > receiveCapacity * sizeof(PlayStatistics)
     *              - receiveCapacity > 0
     *
     */
    Result Query(uint32_t* pOutCount, PlayStatistics* pOutValues, uint32_t receiveCapacity,
        Bit8* const pReadTemporary, uint32_t byteOfTemporary,
        const account::Uid& uid) NN_NOEXCEPT;

    /**
     * @brief       @ref AccountEvent 情報を問い合わせます。
     *
     * @param[out]  pOutCount           問い合わせ結果の有効検出数
     * @param[out]  pOutValues          問い合わせ結果出力バッファ
     * @param[in]   receiveCapacity     問い合わせ結果出力バッファ容量( AccountEvent単位 )
     * @param[in]   eventIndexOffset    取得を開始するイベント全体の先頭からのオフセット。取得対象に含まれます。
     * @param[in]   pReadTemporary      データベース読み込み用一時記憶メモリ領域の先頭アドレス
     * @param[in]   byteOfTemporary     データベース読み込み用一時記憶メモリ領域の容量( byte 単位 )
     * @param[in]   uid                 問い合わせ対象のユーザーアカウントID
     *
     * @retresult
     *      @handleresult   {nn::ResultSuccess}
     *      @handleresult   {nn::pdm::ResultUserNotExist}
     *      @handleresult   {nn::pdm::ResultServiceNotAvailable}
     * @endresult
     *
     * @pre
     *              - pOutCount != nullptr
     *              - pOutValues != nullptr
     *              - pOutValues > receiveCapacity * sizeof(AccountEvent)
     *              - receiveCapacity > 0
     *
     * @details     システムが記録しているイベント全体を eventIndexOffset 番目から走査し、アカウントのオープン・クローズのイベントを出力します。@n
     *              システムはアカウントのオープン・クローズ以外のイベントも記録しており、eventIndexOffset はそれらを含めた全てのイベントの列の中での値です。@n
     *              出力のイベントのインデックスも同様に、全てのイベントを通してのインデックス値であるため、連続的な値にはなりません。
     */
    Result Query(uint32_t* pOutCount, AccountEvent* pOutValues, uint32_t receiveCapacity, uint32_t eventIndexOffset,
        Bit8* const pReadTemporary, uint32_t byteOfTemporary,
        const account::Uid& uid) NN_NOEXCEPT;


    /**
     * @brief       最近遊んだアプリケーションのアプリケーションIDを問い合わせます。
     * @param[out]  pOutCount           問い合わせ結果の有効検出数
     * @param[out]  pOutValues          問い合わせ結果出力バッファ
     * @param[in]   receiveCapacity     問い合わせ結果出力バッファ容量( ncm::ApplicationId 単位 )
     * @param[in]   pReadTemporary      データベース読み込み用一時記憶メモリ領域の先頭アドレス
     * @param[in]   byteOfTemporary     データベース読み込み用一時記憶メモリ領域の容量( byte 単位 )
     * @param[in]   uid                 問い合わせ対象のユーザーアカウントID
     *
     * @retresult
     *      @handleresult   {nn::ResultSuccess}
     *      @handleresult   {nn::pdm::ResultUserNotExist}
     *      @handleresult   {nn::pdm::ResultServiceNotAvailable}
     * @endresult
     *
     * @pre
     *              - pOutCount != nullptr
     *              - pOutValues != nullptr
     *              - receiveCapacity > 0
     */
    Result QueryRecentlyPlayedApplication(uint32_t* pOutCount, ncm::ApplicationId* pOutValues, uint32_t receiveCapacity,
        Bit8* const pReadTemporary, uint32_t byteOfTemporary,
        const account::Uid& uid) NN_NOEXCEPT;

    /**
     * @brief       いずれかのユーザーの最近遊んだアプリケーションが更新されたときにシグナルされるシステムイベントを取得します。
     * @return      いずれかのユーザーの最近遊んだアプリケーションが更新されたときにシグナルされるシステムイベントのポインタ
     */
    os::SystemEvent* GetRecentlyPlayedApplicationUpdateEvent() NN_NOEXCEPT;


    /**
     * @brief       全てのバッファに対してフラッシュを実行します。@n
     *
     * @param[in]   excessRate  100 分率で表現された超過率です。ゼロ指定は常に書き込みます。@
     *                          指定率を超過時にのみ Flush を実施します。
     */
    void Flush(const uint32_t excessRate = 0) NN_NOEXCEPT;

    /**
     * @brief   指定ユーザーアカウントのデータベース保存対象ストレージファイルを削除します。
     *
     * @details デバッグ用です。
     */
    Result Discard(const account::Uid& uid) NN_NOEXCEPT;

    /**
     * @brief   各データベースの保存対象ストレージファイルを削除します。
     *
     * @details デバッグ用です。
     */
    void Discard() NN_NOEXCEPT;

    /**
     * @brief   管理している @ref AccountPlayEventBuffer へのインスタンスを取得します。
     *
     * @details 非スレッドセーフであり、外部からのアクセスを保証するものではないので運用には注意してください。
     */
    NN_FORCEINLINE const AccountPlayEventBuffer& GetBuffer(const BufferIndex index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(0 <= index && index < BufferIndexCountMax);
        return m_Buffers[index];
    }


    //!------------------------------------------------------------------------
    //! @name Debug インタフェース
    //@{

    Result QueryAvailableRange(uint32_t* pOutCount, uint32_t* pOutStartIndex, uint32_t* pOutLastIndex, const account::Uid& uid) NN_NOEXCEPT;

    Result QueryRaw(uint32_t* pOutCount, AccountPlayEvent* pOutValues, uint32_t receiveCapacity, uint32_t eventIndexOffset, const account::Uid& uid) NN_NOEXCEPT;

    //@}
    //!------------------------------------------------------------------------

    //!------------------------------------------------------------------------
    //! @name Migration ( デバイスDB → アカウントDB 変換 )関連
    //@{

    bool CheckRequestForForcedMigration() NN_NOEXCEPT;

    void InitializeMigration(MigrationRecorder* pRecorders, const uint8_t recorderCount) NN_NOEXCEPT;

    void PreapreMigration() NN_NOEXCEPT;

    void ImportMigration(const PlayEvent& source) NN_NOEXCEPT;

    void FinishMigration() NN_NOEXCEPT;

    void FinalizeMigration() NN_NOEXCEPT;

    //@}
    //!------------------------------------------------------------------------

private:
    class FactoryReceiver
    {
    public:
        virtual void OnReceivePowerOn(AccountPlayEventProvider* pOwner) NN_NOEXCEPT
        {
            pOwner->HandlePowerOn();
        }
        virtual void OnReceivePowerOff(AccountPlayEventProvider* pOwner) NN_NOEXCEPT
        {
            pOwner->HandlePowerOff();
        }
        virtual void OnReceiveEventCreated(AccountPlayEventProvider* pOwner, const account::Uid& uid, AccountPlayEvent& newEvent) NN_NOEXCEPT
        {
            pOwner->HandleEventCreated(uid, newEvent);
        }
    };

    class MigrationHandle : public FactoryReceiver
    {
    public:
        virtual void OnReceivePowerOn(AccountPlayEventProvider* pOwner) NN_NOEXCEPT NN_OVERRIDE
        {
            pOwner->HandleMigrationPowerOn(this);
        }
        virtual void OnReceivePowerOff(AccountPlayEventProvider* pOwner) NN_NOEXCEPT NN_OVERRIDE
        {
            pOwner->HandleMigrationPowerOff(this);
        }
        virtual void OnReceiveEventCreated(AccountPlayEventProvider* pOwner, const account::Uid& uid, AccountPlayEvent& newEvent) NN_NOEXCEPT NN_OVERRIDE
        {
            pOwner->HandleMigrationEventCreated(this, uid, newEvent);
        }

        NN_FORCEINLINE MigrationRecorder* GetRecorders() NN_NOEXCEPT
        {
            return m_pRecorders;
        }

        NN_FORCEINLINE const uint8_t GetRecorderCount() const NN_NOEXCEPT
        {
            return m_RecorderCount;
        }

        NN_FORCEINLINE void Reset(MigrationRecorder* pRecorders, const uint8_t recorderCount) NN_NOEXCEPT
        {
            m_pRecorders = pRecorders;
            m_RecorderCount = recorderCount;
        }

        NN_FORCEINLINE void ApplyForceExecution(const bool enabled) NN_NOEXCEPT
        {
            m_IsForceExecution = enabled;
        }

        NN_FORCEINLINE const bool IsForceExecution() const NN_NOEXCEPT
        {
            return m_IsForceExecution;
        }

        MigrationHandle() NN_NOEXCEPT
            : m_pRecorders(nullptr)
            , m_RecorderCount(0)
            , m_IsForceExecution(false) {}

    private:
        MigrationRecorder*  m_pRecorders;       //!< レコーダーインスタンス配列。( 1 or 8 )
        uint8_t             m_RecorderCount;    //!< 有効なレコーダーインスタンス数。
        bool                m_IsForceExecution; //!< 強制マイグレーション実施要求。( fwdbg )
    };

    AccountPlayEventProvider() NN_NOEXCEPT
        : m_ActiveFactoryReceiver(&m_FactoryReceiver)
        , m_RecentlyPlayedApplicationUpdateEvent(os::EventClearMode_ManualClear, true) {}

    AccountPlayEventBuffer* FindIdleBufferUnsafe() NN_NOEXCEPT;
    AccountPlayEventBuffer* FindBufferUnsafe(const account::Uid& uid) NN_NOEXCEPT;

    virtual void OnReceivePowerOn() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnReceivePowerOff() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnReceiveEventCreated(const account::Uid& uid, AccountPlayEvent& newEvent) NN_NOEXCEPT NN_OVERRIDE;

    void HandlePowerOn() NN_NOEXCEPT;
    void HandlePowerOff() NN_NOEXCEPT;
    void HandleEventCreated(const account::Uid& uid, AccountPlayEvent& newEvent) NN_NOEXCEPT;

    void HandleMigrationPowerOn(MigrationHandle* pHandle) NN_NOEXCEPT;
    void HandleMigrationPowerOff(MigrationHandle* pHandle) NN_NOEXCEPT;
    void HandleMigrationEventCreated(MigrationHandle* pHandle, const account::Uid& uid, AccountPlayEvent& newEvent) NN_NOEXCEPT;

    void AdjustPlayStatisticsWithCurrentCondition(PlayStatistics* pStatistics, const ncm::ApplicationId& appId, const account::Uid& uid) NN_NOEXCEPT;

    NN_FORCEINLINE const bool CanImportNormally() const NN_NOEXCEPT
    {
        return (&m_FactoryReceiver == m_ActiveFactoryReceiver);
    }

    mutable os::SdkRecursiveMutex   m_BufferUpdateLock;
    AccountPlayEventBuffer          m_Buffers[BufferCapacity];
    FactoryReceiver*                m_ActiveFactoryReceiver;
    FactoryReceiver                 m_FactoryReceiver;
    MigrationHandle                 m_MigrationHandle;

    os::SystemEvent                 m_RecentlyPlayedApplicationUpdateEvent;
};



}}}
