﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief PCV ライブラリの型や定数の定義
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_TypedStorage.h>

#include "pcv_Modules.h"
#include "pcv_PowerControlTargets.h"
#include "pcv_PowerDomains.h"

namespace nn {
namespace pcv {

/**
* @brief クロックレートの型です。
*
* @details 単位は Hz です。
*/
typedef uint32_t ClockHz;

/**
* @brief Microvolts values
*
* @details Used with the regulator/power domain APIs.
*/
typedef int32_t MicroVolt;

/**
* @brief Temperature in millicelius
*
* @details Used with the temperatures APIs.
*/
typedef int32_t MilliC;

/**
* @brief モジュールの状態を格納する構造体です。
*
* @details GetState 関数で使用します。
*/
struct ModuleState {
    ClockHz clockFrequency; //!< クロックレートが格納されます
    bool    clockEnabled;   //!< true が入っていた場合 enabled です。enabled の場合、当該モジュールのクロック供給が有効になっています。
    bool    powerEnabled;   //!< true が入っていた場合 enabled です。enabled の場合、当該モジュールに電力が供給されています。
    bool    resetAsserted;  //!< true が入っていた場合 asserted です。asserted の場合、当該モジュールのクロック供給回路がリセット状態になっています。
    ClockHz minVClockRate;  //!< The minimum voltage for this module is the minimum voltage needed for minVClockRate
};

/**
* @brief 電源ドメインの状態を格納する構造体です。
*
* @details GetPowerDomainStateTable 関数で使用します。
*/
struct PowerDomainState {
    bool    enabled;    //!< true が入っていた場合 enabled GetVoltageEnabled() で取得できる値と同じです。
    int32_t voltage;    //!< 現在設定されている電圧値です。GetVoltageValue() で取得できる値と同じです。
};

/**
* @brief クロックレートリストの形式を表す列挙型クラスです。
*
* @details GetPossibleClockRates 関数で使用します。
*/
enum ClockRatesListType {
    ClockRatesListType_Invalid, //!< 無効なリストです。
    ClockRatesListType_Discrete,//!< 正確な値のリストが返ったことを表します。SetClockRate 実行時には正確な値を引数にしてください。
    ClockRatesListType_Range,   //!< 値の範囲が返ったことを表します。SetClockRate 実行時には正確な値をとらせる必要はありません。上限クロックレート以下かつ指定クロックレート以上で設定可能な最小のクロックレートとなります。指定クロックレート以上に設定不可能な場合は、指定クロックレート以下で設定可能な最大のクロックレートとなります。
};

/**
* @brief The min and max of a temperature threshold range
*
* @details Used with GetTemperatureThresholds API
*/
struct TemperatureThreshold {
    MilliC minMilliC;           //!< minimum temperature of the threshold range
    MilliC maxMilliC;           //!< maximum temperature of the threshold range
};

/**
* @brief The maximum possible number of clock rates that GetPossibleClockRates API can return.
*
* @details Can be used to size the return buffer's size used with GetPossibleClockRates API.
*/
const int MaxNumClockRates = 32;   // Going forward, use this one not MaxRatesFromVcc. Also need to update to Raptor's exported const/value, not determine the value here

/**
* @brief MaxNumTemperatureThresholds
*
* @details  The maximum possible number thresholds that GetTemperatureThresholds API can return.
*/
const int MaxNumTemperatureThresholds = 8;  // When available, need to update to Raptor's exported const/value, not determine the value here

} // pcv
} // nn
