﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/pctl/pctl_TypesWatcher.h>
#include <nn/pctl/pctl_ResultWatcher.h>
#include <nn/pctl/pctl_PairingInfo.h>

namespace nn { namespace pctl {

/**
 * @name 見守り機能サーバー連携確立処理関数
 * @{
 */

/**
 * @brief 見守り用アカウントと連携(ペアリング)を行うためのリクエストを発行します。
 * @param[out] pInfo 連携先の相手情報が含まれる構造体へのポインター
 * @param[in] code 連携コード
 * @return 処理の結果を表す nn::Result 値。通信状況によってこれ以外の Result 値が返ることがあります。
 * @retval nn::ResultSuccess 連携処理中状態に入りました。
 * @retval nn::pctl::ResultUnknownPairingCode 連携コードが誤っているか無効です。
 * @retval nn::pctl::ResultPairingFailed 連携処理に失敗しました。
 * @retval nn::pctl::ResultCanceled 処理がキャンセルされました。
 * @retval nn::pctl::ResultConnectionNotAccepted 通信の利用要求が提出されていないか切断されています。
 * @pre
 *  - pInfo != nullptr
 *  - code != nullptr
 *  - 事前に通信の利用要求が出されている
 *  - 本体の起動モードがメンテナンスモードではない
 * @post
 *  - 戻り値が nn::ResultSuccess である場合、連携処理中状態になり、相手側の情報が pInfo に入る
 *
 * @details
 * 本関数は内部で通信処理を行うためブロックします。
 * また、適切な通信処理を行うために
 * 事前に通信の利用要求をシステムに出す必要があります。
 *
 * 戻り値が nn::ResultSuccess および nn::pctl::ResultCanceled ではない場合は、
 * エラーハンドリングとしてエラービューアアプレットを利用することができます。
 *
 * 本関数が true を返した場合は、pInfo の情報を元に
 * ユーザーに確認を行い、連携処理を進める場合は @ref AuthorizePairing を
 * 呼び出す必要があります。
 *
 * 本関数が true を返した場合、pInfo から有効なデータを得ることができますが、
 * 状態が変化するとデータの取得ができなくなるため、
 * 必要なデータを取得したら pInfo を使用しないようにしてください。
 * データが取得できなくなる条件については @ref PairingInfo の説明をご覧ください。
 *
 * 通信処理を中断したい場合は @ref CancelNetworkRequest 関数を呼び出してください。
 * (中断された場合、戻り値は nn::pctl::ResultCanceled となります。)
 */
nn::Result RequestPairing(PairingInfo* pInfo, const char* code) NN_NOEXCEPT;

/**
 * @brief 連携先の情報が問題ないことをサーバーに通知し、連携処理を完了させます。
 * @param[in, out] pInfo 連携先の相手情報が含まれる構造体へのポインター。
 * @return 処理の結果を表す nn::Result 値。通信状況によってこれ以外の Result 値が返ることがあります。
 * @retval nn::ResultSuccess 連携処理が完了しました。
 * @retval nn::pctl::ResultPairingFailed 連携処理に失敗しました。
 * @retval nn::pctl::ResultCanceled 処理がキャンセルされました。
 * @retval nn::pctl::ResultConnectionNotAccepted 通信の利用要求が提出されていないか切断されています。
 * @pre
 *  - pInfo != nullptr
 *  - pInfo のインスタンスが有効な状態であり、連携処理中状態を保持している
 *  - 事前に通信の利用要求が出されている
 *  - 本体の起動モードがメンテナンスモードではない
 * @post
 *  - pInfo は連携済み状態を保持している
 *
 * @details
 * 本関数は内部で通信処理を行うためブロックします。
 * また、適切な通信処理を行うために
 * 事前に通信の利用要求をシステムに出す必要があります。
 *
 * 戻り値が nn::ResultSuccess および nn::pctl::ResultCanceled ではない場合は、
 * エラーハンドリングとしてエラービューアアプレットを利用することができます。
 *
 * 本関数は @ref RequestPairing の呼び出し後に受け取った pInfo の値が
 * 問題なく、連携処理を完了させたい場合に呼び出します。
 * 内容に問題があるなどして連携処理を取り止めたい場合は
 * 特別な処理を行う必要はありません。
 *
 * 本関数が true を返した場合、pInfo から有効なデータを得ることができますが、
 * 状態が変化するとデータの取得ができなくなるため、
 * 必要なデータを取得したら pInfo を使用しないようにしてください。
 * データが取得できなくなる条件については @ref PairingInfo の説明をご覧ください。
 *
 * 通信処理を中断したい場合は @ref CancelNetworkRequest 関数を呼び出してください。
 * (中断された場合、戻り値は nn::pctl::ResultCanceled となります。)
 */
nn::Result AuthorizePairing(PairingInfo* pInfo) NN_NOEXCEPT;

/**
 * @}
 */

}}
