﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/pctl/pctl_TypesForAuthentication.h>
#include <nn/pctl/pctl_ResultForAuthentication.h>

#include <nn/os/os_SystemEvent.h>

namespace nn { namespace pctl {

/**
 * @name 本体機能向け解除コード・マスターキー処理関連関数
 * @{
 */

/**
 * @brief ペアコンの解除コード(PINコード)を用いて、ペアコン制限の一時的な解除状態を設定します。
 * @param[in] code 解除コード(入力値)
 * @return nn::Result の値
 * @retval nn::ResultSuccess 解除コードの検証に成功し、一時的な解除状態に設定しました。
 * @retval nn::pctl::ResultWrongPinCode 入力された解除コードが正しくありません。
 * @pre
 *  - code != nullptr
 *  - IsRestrictionEnabled() == true
 * @post
 *  - IsRestrictionTemporaryUnlocked() == true
 */
nn::Result UnlockRestrictionTemporarily(const char* code) NN_NOEXCEPT;

/**
 * @brief ペアコンの解除コード(PINコード)を用いて、汎用設定利用・変更の
 *        許可が必要なシステムシーンへの遷移の制限を解除します。
 * @param[in] code 解除コード(入力値)
 * @return nn::Result の値
 * @retval nn::ResultSuccess 解除コードの検証に成功し、一時的な解除状態に設定しました。
 * @retval nn::pctl::ResultWrongPinCode 入力された解除コードが正しくありません。
 * @pre
 *  - code != nullptr
 *  - IsRestrictionEnabled() == true
 */
nn::Result UnlockSystemSettingsRestriction(const char* code) NN_NOEXCEPT;

/**
 * @brief ペアコンの解除コード(PINコード)を設定します。
 * @param[in] code 新しい解除コード
 * @pre
 *  - code が nullptr ではない場合
 *    - code の文字数が4文字以上8文字以下である
 * @post
 *  - code が示す値を @ref UnlockRestrictionTemporarily や
 *    @ref UnlockSystemSettingsRestriction に指定すると、その場合に限り true が返る
 *
 * @details
 * code に nullptr を設定すると解除コード未設定状態、すなわち
 * ペアコンの制限が無効である状態にすることができます。
 * この方法で制限を無効にした場合でも、各種制限値は
 * 制限を無効にする前の状態を維持します。
 * 各種制限値を含めてリセットする場合は @ref DeleteSettings を利用してください。
 */
void SetPinCode(const char* code) NN_NOEXCEPT;

/**
 * @brief 現在設定されているペアコンの解除コード(PINコード)の桁数を返します。
 * @return 解除コードの桁数(末尾のNULL文字を除く)
 *
 * @details
 * 解除コードが未設定である場合は 0 が返ります。
 */
int GetPinCodeLength() NN_NOEXCEPT;

/**
 * @brief 現在設定されているペアコンの解除コード(PINコード)を返します。
 * @param[out] outCode 解除コードを受け取るバッファー
 * @param[in] maxLengthWithNull outCode が受け取れる解除コードの最大長(NULL文字分を含む)
 * @return 出力できた場合、outCode に出力された解除コードの桁数(末尾のNULL文字を除く)。
 *         出力できなかった場合は 0。
 * @pre
 *  - outCode != nullptr
 *  - maxLengthWithNull > 0
 *
 * @details
 * 実際の長さは @ref GetPinCodeLength で取得できます。
 * この戻り値 + 1 した値を maxLengthWithNull に指定することで、
 * 解除コード全体を取得することができます。
 *
 * 関数が成功した場合、outCode は必ずNULL文字込みのデータが返ります。
 *
 * 解除コードが未設定である場合は 0 が返ります。
 */
int GetPinCode(char* outCode, int maxLengthWithNull) NN_NOEXCEPT;

/**
 * @brief ペアコン解除コードの入力を連続で間違えたことをサーバーに通知します。
 *
 * @details
 * サーバーへの通知は非同期に行われるため、本関数は処理を待たずに制御を返します。
 *
 * 本体がメンテナンスモードで起動されている場合、本関数は何も行いません。
 */
void NotifyWrongPinCodeInputManyTimes() NN_NOEXCEPT;

/**
 * @brief マスターキーの発行に利用するお問い合わせ番号を生成します。
 * @param[out] pOutCodeData お問い合わせ番号データを受け取るポインター
 * @pre
 *  - pOutCodeData != nullptr
 *
 * @details
 * お問い合わせ番号には有効期限があります。有効期限を過ぎた場合、
 * 本関数が返すデータは無効なデータとなります。
 * (そのデータを @ref CheckMasterKey に指定すると、常にマスターキーが
 * 正しくないという判定が行われます。)@n
 * また、短い期間で本関数を複数回呼び出した場合、得られるデータは
 * 同じデータとなります。
 *
 * 本関数が返すデータは本体の再起動やシャットダウンなどによって無効化されます
 * (スリープでは無効化しません)。
 *
 * @platformbegin{Windows}
 * Windows 環境では暫定実装として、常に固定のお問い合わせ番号データを返します。
 * @platformend
 */
void GenerateInquiryCode(InquiryCode* pOutCodeData) NN_NOEXCEPT;

/**
 * @brief お問い合わせ番号を元に生成されたマスターキーが正しいかどうかを判定します。
 * @param[in] codeData @ref GenerateInquiryCode で生成されたお問い合わせ番号のデータ
 * @param[in] masterKey お問い合わせ番号を元に生成されたマスターキー
 * @return masterKey が正しければ true
 * @pre
 *  - masterKey != nullptr
 *
 * @details
 * codeData が指すデータが無効である場合(有効期限切れも含みます)、
 * 本関数は false を返します。
 *
 * 本関数による判定処理は時間がかかる可能性があります。
 *
 * @platformbegin{Windows}
 * Windows 環境では暫定実装として、codeData.codeString と masterKey それぞれの
 * 先頭6文字が文字列として一致するかどうかの結果(一致する場合は true)を返します。
 * @platformend
 */
bool CheckMasterKey(const InquiryCode& codeData, const char* masterKey) NN_NOEXCEPT;

/**
 * @brief ペアコン設定の同期で解除コードが変更された際に通知を受け取るためのイベントオブジェクトを返します。
 * @return イベントオブジェクトインスタンスへのポインター。
 *         本体の起動モードがメンテナンスモードであるなど、
 *         通知が全く発行されない状況である場合は nullptr が返ります。
 * @post
 *  - 戻り値が nullptr でない場合、戻り値のインスタンスは初期化された状態であり
 *    @ref nn::os::SystemEvent::Wait などでシグナルを待つことが可能である
 *
 * @details
 * サーバー側でペアコン設定のうち解除コードが更新されて本体に反映された直後のタイミングで、
 * 本関数が返すイベントオブジェクトがシグナル状態となります。
 * @ref nn::pctl::GetSynchronizationEvent が返すイベント経由でも通知が行われますが、
 * 本関数のイベントは解除コードの変更に関する通知に特化しており、
 * 別の設定が変更されても解除コードの変更がない場合は通知が行われません。@n
 * なお、ユーザーが能動的に本体上で設定変更を行った場合は通知が行われません。
 *
 * 本関数が返すイベントオブジェクトはシグナルを自動でクリアするモードとなります。
 * (シグナル状態のイベントオブジェクトに対して @ref nn::os::SystemEvent::Wait
 * などを呼び出すとシグナル状態がリセットされます。)@n
 * なお、内部的には同一のイベントオブジェクトを参照しており、
 * 複数のプロセスで同時に利用されることを想定していないため、
 * 同時にイベントのハンドルが行われた場合などの挙動は未定義です。
 *
 * 本関数が返すオブジェクトのインスタンスは、同一プロセスにおいては
 * 常に同一のインスタンスとなります。(異なるプロセスでは
 * 状態管理が独立した別のイベントオブジェクトを返します。)
 * グローバルに存在するインスタンスであるため、delete 演算子や
 * メモリ解放などの関数にインスタンスへのポインターを渡さないでください。
 */
nn::os::SystemEvent* GetPinCodeChangedEvent() NN_NOEXCEPT;

/**
 * @}
 */

}}
