﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/pctl/pctl_TypesWatcher.h>
#include <nn/time/time_LocationName.h>
#include <nn/time/time_PosixTime.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

/**
 * @brief イベントの最大保存件数です。
 */
static const int MaxEventCount = 3000;

/**
 * @brief 何秒おきにイベント送信を行うかを示す定数です。(秒単位)
 */
static const int16_t TimeSpanForPostEvents = 10 * 60;

/**
 * @brief 強制電源断時に最大何秒の時間経過とするかを示す定数です。(秒単位)
 */
static const int16_t MaxElapsedTimeSpanOnUnexpectedShutdown = TimeSpanForPostEvents;

/**
 * @brief イベントの種類を示す定数です。
 * @details 計算のために、サーバーには送信しないもののローカルには保存するイベントも含まれます。
 */
enum class EventType : uint8_t
{
    Invalid = 0,   // 無効なデータ
    DidDeviceLaunch,
    DidWakeup,
    DidSleep,
    DidApplicationLaunch,           // Payload::ApplicationLaunch を利用
    DidApplicationTerminate,        // Payload::ApplicationTerminate を利用
    Idle,
    DidUnlock,
    DidLock,
    DidWrongUnlockCode,
    DidComeOnline,
    DidGoOffline,
    DidAddNewManagedApplication,    // Payload::AddManagedApplication を利用
    DidRemoveManagedApplication,    // Payload::RemoveManagedApplication を利用
    DidInterruptPlaying,
    DidApplicationPlay,             // Payload::PlayedApplication を利用
    DidAlarmMakeInvisible,
    DidAlarmMakeVisible,
    DidApplicationDownloadStart,    // Payload::StartedApplicationDownload を利用
    DidReachLimitTime,
    DidApplicationSuspend,          // Payload::SuspendedApplication を利用
    DidApplicationResume,           // Payload::ResumedApplication を利用
    DidUserOpen,                    // Payload::OpenedUser を利用
    DidUserClose,                   // Payload::ClosedUser を利用
    DidShutdown,
    DidPlayTimerStart,
    DidPlayTimerStop,
    DidDeviceActivate,
    DidLocationNameChange,
    DidUnexpectedShutdownOccur,
    DidLimitedApplicationLaunch,    // Payload::LimitedApplicationLaunch を使用
    EventTypeMaxCount               // (計算用)
};

/**
* @brief イベントデータのバージョン番号(Sys1～2向け)です。
*/
static const uint8_t EventDataVersionInitial = 0;
/**
 * @brief イベントデータのバージョン番号(Sys3向け)です。
 */
static const uint8_t EventDataVersionForSys3 = 1;

/**
 * @brief イベントデータのバージョン番号(書き込みで利用するもの)です。
 */
static const uint8_t EventDataVersionCurrent = EventDataVersionForSys3;
/**
 * @brief プレイ状態の計算に使用できるイベントの最低バージョンです。
 */
static const uint8_t EventDataVersionPlayStateSupported = EventDataVersionForSys3;

/**
 * @brief セーブデータに保存するイベントデータです。
 */
struct EventData
{
    EventType eventType;
    bool isDaylightSavingTime;
    NN_PADDING2;
    int32_t timezoneOffsetSeconds;
    nn::time::PosixTime timestamp; // in UTC
    nn::time::LocationName timezoneName;
    NN_PADDING4;
    int64_t _unused;               // (未使用、旧: 連続稼働時間)
    uint8_t dataVersion;           // (初期システムバージョンでは 0 埋めされている)
    uint8_t reservedForFutureUse[15];
    union Payload
    {
        struct ApplicationLaunch
        {
            nn::ncm::ApplicationId applicationId;
        } applicationLaunch;
        struct ApplicationTerminate
        {
            nn::ncm::ApplicationId applicationId;
        } applicationTerminate;
        struct AddManagedApplication
        {
            nn::ncm::ApplicationId applicationId;
        } addManagedApplication;
        struct RemoveManagedApplication
        {
            nn::ncm::ApplicationId applicationId;
        } removeManagedApplication;
        struct PlayedApplication
        {
            nn::ncm::ApplicationId applicationId;
        } playedApplication;
        struct StartedApplicationDownload
        {
            nn::ncm::ApplicationId applicationId;
        } startedApplicationDownload;
        struct SuspendedApplication
        {
            nn::ncm::ApplicationId applicationId;
        } suspendedApplication;
        struct ResumedApplication
        {
            nn::ncm::ApplicationId applicationId;
        } resumedApplication;
        struct OpenedUser
        {
            nn::account::Uid uid;
        } openedUser;
        struct ClosedUser
        {
            nn::account::Uid uid;
        } closedUser;
        struct LimitedApplicationLaunch
        {
            nn::ncm::ApplicationId applicationId;
        } limitedApplicationLaunch;

        uint8_t reserved[16];
    } payload;
};
NN_STATIC_ASSERT(sizeof(nn::time::PosixTime) == 8); // データ構造上このサイズを期待しているため確認
NN_STATIC_ASSERT(sizeof(EventData) == 96); // 念のための確認

}}}}}
