﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/account/account_Types.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/time/time_CalendarTime.h>
#include <nn/time/time_PosixTime.h>
#include <nn/util/util_BitFlagSet.h>

#include <type_traits>

namespace nn { namespace pctl { namespace detail { namespace service {

struct PlayedApplicationStateFlags;
struct PlayedApplicationUserStateFlags;
struct PlayStateFlags;
struct PlayedApplicationState;
struct PlayedApplicationUserState;
struct PlayState;

namespace watcher
{
    struct EventData;
}

////////////////////////////////////////////////////////////////////////////////

enum class TypeNumber : int16_t
{
    Invalid = 0,
    PlayedApplicationState = 1,
    PlayedApplicationUserState = 2
};

struct PlayStateDataBlock
{
    TypeNumber typeNumber;
    uint8_t data[54];

    bool IsAnyDataStored() const NN_NOEXCEPT
    {
        return typeNumber != TypeNumber::Invalid;
    }
    void ClearData() NN_NOEXCEPT
    {
        typeNumber = TypeNumber::Invalid;
    }
};

////////////////////////////////////////////////////////////////////////////////

/**
 * @brief アプリケーションの状態を示すフラグ型です。
 */
typedef nn::util::BitFlagSet<32, PlayedApplicationStateFlags> PlayedApplicationStateFlagSet;

/**
 * @brief アプリケーションの状態を示すフラグの定義です。
 */
struct PlayedApplicationStateFlags
{
    //! @brief アプリケーションが実行中状態であることを示すフラグです。
    typedef PlayedApplicationStateFlagSet::Flag<0> Running;
    //! @brief アプリケーションが中断状態であることを示すフラグです。Running がセットされている場合のみセットされる場合があります。
    typedef PlayedApplicationStateFlagSet::Flag<1> Suspended;
    //! @brief ユーザーが非選択状態であることを示すフラグです。
    typedef PlayedApplicationStateFlagSet::Flag<2> Anonymous;
};

/**
 * @brief アプリケーションが実行されたことに関する状態を持つ構造体です。
 * @details この状態は日付の変更があったタイミングでリセットされます。
 */
struct PlayedApplicationState
{
    TypeNumber typeNumber;
    int16_t _padding1[1];
    /**
     * @brief アプリケーションの状態に関するフラグです。
     */
    PlayedApplicationStateFlagSet flags;
    /**
     * @brief アプリケーションが起動された(仮想)時刻です。
     * @details アプリケーションの中断・再開が行われたなどの場合にこの時刻が補正されます。
     */
    nn::time::PosixTime timeStarted;
    /**
     * @brief アプリケーションの累計実行時間です。
     */
    nn::TimeSpanType elapsedTime;
    /**
     * @brief 対象アプリケーションのアプリケーションIDです。
     */
    nn::ncm::ApplicationId applicationId;
    /**
     * @brief ユーザー非選択状態が開始した時刻です。
     */
    nn::time::PosixTime timeAnonymousStarted;
    /**
     * @brief ユーザー非選択状態の累計時間です。
     */
    nn::TimeSpanType elapsedAnonymousTime;
    uint8_t _padding2[8];

    /**
     * @brief data が指す内容がこの構造体の内容になるかどうかを返します。
     */
    static bool IsTypeMatch(const PlayStateDataBlock* data) NN_NOEXCEPT
    {
        return data->typeNumber == TypeNumber::PlayedApplicationState;
    }

    /**
     * @brief data が指す種類がこの構造体の内容になる場合、この構造体の型に変換して返します。
     * @return 種類が一致する場合は変換された data の値、それ以外は nullptr
     */
    static const PlayedApplicationState* FromDataBlock(const PlayStateDataBlock* data) NN_NOEXCEPT
    {
        return data->typeNumber == TypeNumber::PlayedApplicationState ?
            reinterpret_cast<const PlayedApplicationState*>(data) : nullptr;
    }

    /**
     * @brief data が指す種類がこの構造体の内容になる場合、この構造体の型に変換して返します。
     * @return 種類が一致する場合は変換された data の値、それ以外は nullptr
     */
    static PlayedApplicationState* FromDataBlock(PlayStateDataBlock* data) NN_NOEXCEPT
    {
        return data->typeNumber == TypeNumber::PlayedApplicationState ?
            reinterpret_cast<PlayedApplicationState*>(data) : nullptr;
    }

    /**
     * @brief data が指す内容をこの構造体の内容で初期化します。
     * @return 初期化済みの、data を変換したポインター
     */
    static PlayedApplicationState* InitializeData(PlayStateDataBlock* data) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<PlayedApplicationState*>(data);
        p->InitializeData();
        return p;
    }

    /**
     * @brief この構造体の内容を初期化します。
     */
    void InitializeData() NN_NOEXCEPT
    {
        this->typeNumber = TypeNumber::PlayedApplicationState;
        std::memset(this->_padding1, 0, sizeof(this->_padding1));
        this->flags.Reset();
        this->timeStarted = nn::time::PosixTime{ 0 };
        this->elapsedTime = nn::TimeSpan(0);
        this->applicationId = nn::ncm::ApplicationId::GetInvalidId();
        this->timeAnonymousStarted = nn::time::PosixTime{ 0 };
        this->elapsedAnonymousTime = nn::TimeSpan(0);
        std::memset(this->_padding2, 0, sizeof(this->_padding2));
    }

    /**
     * @brief 有効なデータを持っているかどうかを返します。
     */
    bool IsValidData() const NN_NOEXCEPT
    {
        return this->typeNumber == TypeNumber::PlayedApplicationState;
    }

    /**
     * @brief このインスタンスが示すデータが指定したアプリケーションIDのデータかどうかを返します。
     */
    bool IsEqualApplicationId(const nn::ncm::ApplicationId& appId) const NN_NOEXCEPT
    {
        return this->applicationId == appId;
    }

    /**
     * @brief 保持しているデータを無効化します。
     * @post IsValidData() == false
     */
    void InvalidateData() NN_NOEXCEPT
    {
        reinterpret_cast<PlayStateDataBlock*>(this)->ClearData();
    }

    /**
     * @brief アプリケーションが実行状態であれば true を返します。
     * @pre IsValidData() == true
     */
    bool IsRunning() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        return this->flags.Test<PlayedApplicationStateFlags::Running>();
    }

    /**
     * @brief アプリケーションの状態を実行状態として設定します。
     * @pre applicationId != nn::ncm::ApplicationId::GetInvalidId()
     * @post IsValidData() == true
     * @details
     * 実行状態と同時にユーザー非選択状態に設定されます。ユーザーが選択されている場合は直後に
     * SetNonAnonymousState を呼び出すことで対応できます。
     */
    void SetStarted(const nn::ncm::ApplicationId& appId, const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ncm::ApplicationId::GetInvalidId());
        if (this->applicationId != appId)
        {
            this->elapsedTime = nn::TimeSpan();
            this->flags.Reset();
        }
        if (!IsRunning())
        {
            this->applicationId = appId;
            this->timeStarted = timeAt;
            this->flags.Set<PlayedApplicationStateFlags::Running>(true);
            this->flags.Set<PlayedApplicationStateFlags::Suspended>(false);
            // 便宜上「ユーザー非選択状態が始まった」とする
            // (アプリケーション開始時は「非選択状態」とする)
            SetAnonymousState(timeAt);
        }
    }

    /**
     * @brief アプリケーションの状態を停止状態に設定します。
     * @pre IsValidData() == true
     */
    void SetStopped(const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        if (IsRunning())
        {
            // 便宜上「ユーザー非選択状態が終わった」とする
            SetNonAnonymousState(timeAt);
            // 中断している場合は稼働時間に含めない
            if (!IsSuspended())
            {
                auto elapsed = (timeAt - this->timeStarted);
                // 終了時刻が過去になる場合は経過時間 0 とする
                if (elapsed < nn::TimeSpan(0))
                {
                    elapsed = nn::TimeSpan(0);
                }
                this->elapsedTime += elapsed;
            }
            this->flags.Set<PlayedApplicationStateFlags::Running>(false);
        }
    }

    /**
     * @brief アプリケーションが中断状態であれば true を返します。
     * @pre IsValidData() == true
     */
    bool IsSuspended() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        return flags.Test<PlayedApplicationStateFlags::Suspended>();
    }

    /**
     * @brief アプリケーションの状態を中断状態に設定します。
     * @return 状態変更があれば true
     * @pre IsValidData() == true
     * @details IsRunning() == false や IsSuspended() == true の場合は何もしません。
     */
    bool SetSuspended(const nn::time::PosixTime& timeSuspended) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        if (!IsRunning() || IsSuspended())
        {
            return false;
        }

        auto elapsed = (timeSuspended - this->timeStarted);
        // 中断時刻が過去になる場合は経過時間 0 とする
        if (elapsed < nn::TimeSpan(0))
        {
            elapsed = nn::TimeSpan(0);
        }
        this->elapsedTime += elapsed;
        this->timeStarted = timeSuspended;
        if (IsAnonymous())
        {
            elapsed = (timeSuspended - this->timeAnonymousStarted);
            // 中断時刻が過去になる場合は経過時間 0 とする
            if (elapsed < nn::TimeSpan(0))
            {
                elapsed = nn::TimeSpan(0);
            }
            this->elapsedAnonymousTime += elapsed;
            this->timeAnonymousStarted = timeSuspended;
        }
        this->flags.Set<PlayedApplicationStateFlags::Suspended>(true);
        return true;
    }

    /**
     * @brief アプリケーションの状態を非中断状態に設定します。
     * @return 状態変更があれば true
     * @pre IsValidData() == true
     * @details IsRunning() == false や IsSuspended() == false の場合は何もしません。
     */
    bool SetResumed(const nn::time::PosixTime& timeResumed) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        if (!IsRunning() || !IsSuspended())
        {
            return false;
        }

        this->timeStarted = timeResumed;
        if (IsAnonymous())
        {
            this->timeAnonymousStarted = timeResumed;
        }
        this->flags.Set<PlayedApplicationStateFlags::Suspended>(false);
        return true;
    }

    /**
     * @brief ユーザーが非選択状態であれば true を返します。
     * @pre IsValidData() == true
     */
    bool IsAnonymous() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        return this->flags.Test<PlayedApplicationStateFlags::Anonymous>();
    }

    /**
     * @brief ユーザーが非選択状態であること設定します。
     * @pre IsValidData() == true
     * @details IsRunning() == false や IsAnonymous() == true の場合は何もしません。
     */
    void SetAnonymousState(const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        if (IsRunning() && !IsAnonymous())
        {
            this->timeAnonymousStarted = timeAt;
            this->flags.Set<PlayedApplicationStateFlags::Anonymous>(true);
        }
    }

    /**
     * @brief ユーザーが1人以上選択された状態であること設定します。
     * @pre IsValidData() == true
     * @details IsRunning() == false や IsAnonymous() == false の場合は何もしません。
     */
    void SetNonAnonymousState(const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        if (IsRunning() && IsAnonymous())
        {
            if (!IsSuspended())
            {
                auto elapsed = (timeAt - this->timeAnonymousStarted);
                // 経過時間がマイナスになる場合は 0 とする
                if (elapsed < nn::TimeSpan(0))
                {
                    elapsed = nn::TimeSpan(0);
                }
                this->elapsedAnonymousTime += elapsed;
            }
            this->timeAnonymousStarted = timeAt;
            this->flags.Set<PlayedApplicationStateFlags::Anonymous>(false);
        }
    }

    /**
     * @brief アプリケーションの稼働時間を返します。
     * @pre IsValidData() == true
     * @details
     * アプリケーションが実行中であれば、開始時刻と timeNow を考慮した値を返します。
     */
    nn::TimeSpan GetRunningTime(const nn::time::PosixTime& timeNow) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        return (IsRunning() && !IsSuspended() ? (timeNow - this->timeStarted) : nn::TimeSpan()) + nn::TimeSpan(this->elapsedTime);
    }

    /**
     * @brief アプリケーションのユーザー非選択状態におけるプレイ時間を返します。
     * @pre IsValidData() == true
     * @details
     * アプリケーションが実行中で、かつユーザー非選択状態であれば、開始時刻と timeNow を考慮した値を返します。
     */
    nn::TimeSpan GetAnonymousPlayingTime(const nn::time::PosixTime& timeNow) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        return (IsRunning() && !IsSuspended() && IsAnonymous() ?
            (timeNow - this->timeAnonymousStarted) : nn::TimeSpan()) + nn::TimeSpan(this->elapsedAnonymousTime);
    }

    /**
     * @brief スリープなどで除外される時間を適用して補正します。
     * @pre IsValidData() == true
     */
    void AdjustIgnoreTime(const nn::time::PosixTime& ignoreStartTime, const nn::time::PosixTime& ignoreEndTime) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        if (!IsRunning())
        {
            return;
        }
        if (ignoreStartTime < this->timeStarted)
        {
            // 開始前から除外が始まっていた場合は除外区間終了時刻を開始時刻に設定
            this->timeStarted = ignoreEndTime;
        }
        else
        {
            // 除外する時間分だけ開始時刻を調整
            this->timeStarted += ignoreEndTime - ignoreStartTime;
        }
        if (IsAnonymous())
        {
            if (ignoreStartTime < this->timeAnonymousStarted)
            {
                // 開始前から除外が始まっていた場合は除外区間終了時刻を開始時刻に設定
                this->timeAnonymousStarted = ignoreEndTime;
            }
            else
            {
                // 除外する時間分だけ開始時刻を調整
                this->timeAnonymousStarted += ignoreEndTime - ignoreStartTime;
            }
        }
    }

    /**
     * @brief アプリケーションの稼働開始時刻をリセットします。
     * @pre IsValidData() == true
     * @details 経過時間もリセットされます。
     */
    void ResetStartTime(const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        if (IsRunning())
        {
            if (!IsSuspended())
            {
                this->timeStarted = timeAt;
                if (IsAnonymous())
                {
                    this->timeAnonymousStarted = timeAt;
                }
            }
            this->elapsedTime = nn::TimeSpan(0);
            this->elapsedAnonymousTime = nn::TimeSpan(0);
        }
    }
};
NN_STATIC_ASSERT(sizeof(PlayedApplicationState) == sizeof(PlayStateDataBlock));

////////////////////////////////////////////////////////////////////////////////

/**
 * @brief アプリケーションのユーザー別の状態を示すフラグ型です。
 */
typedef nn::util::BitFlagSet<32, PlayedApplicationUserStateFlags> PlayedApplicationUserStateFlagSet;

/**
 * @brief アプリケーションのユーザー別の状態を示すフラグの定義です。
 */
struct PlayedApplicationUserStateFlags
{
    //! @brief ユーザーがオープン状態であることを示すフラグです。
    typedef PlayedApplicationUserStateFlagSet::Flag<0> Opened;
    //! @brief アプリケーションが中断状態であることを示すフラグです。Opened がセットされている場合のみセットされる場合があります。
    typedef PlayedApplicationUserStateFlagSet::Flag<1> Suspended;
};

/**
 * @brief アプリケーションのユーザー別の状態を保持します。
 * @details
 * - この情報はアプリケーションごとに個別に保持されます。また、アプリケーションの起動終了に関係ない
 *  「全体の状態」としてさらに別途状態が保持されます。
 * - Open されていないユーザーの状態は主に日付を跨いだタイミング(PlayState::ResetPlayedStatus)でリセットされます。
 */
struct PlayedApplicationUserState
{
    TypeNumber typeNumber;
    int16_t _padding1[1];
    /**
     * @brief ユーザーのプレイ状態に関するフラグです。
     */
    PlayedApplicationUserStateFlagSet flags;
    /**
     * @brief ユーザーがプレイを開始した(仮想)時刻です。
     * @details アプリケーションの中断・再開が行われたなどの場合にこの時刻が補正されます。
     */
    nn::time::PosixTime timeStarted;
    /**
     * @brief ユーザーの累計プレイ時間です。
     */
    nn::TimeSpanType elapsedTime;
    /**
     * @brief ユーザーがプレイしたアプリケーションのIDです。
     * @details nn::ncm::ApplicationId::GetInvalidId() である場合は全体の状態管理を表します。
     */
    nn::ncm::ApplicationId applicationId;
    /**
     * @brief ユーザーの識別子です。
     */
    nn::account::Uid uid;
    uint8_t _padding2[8];

    /**
    * @brief data が指す内容がこの構造体の内容になるかどうかを返します。
    */
    static bool IsTypeMatch(PlayStateDataBlock* data) NN_NOEXCEPT
    {
        return data->typeNumber == TypeNumber::PlayedApplicationUserState;
    }

    /**
     * @brief data が指す種類がこの構造体の内容になる場合、この構造体の型に変換して返します。
     * @return 種類が一致する場合は変換された data の値、それ以外は nullptr
     */
    static const PlayedApplicationUserState* FromDataBlock(const PlayStateDataBlock* data) NN_NOEXCEPT
    {
        return data->typeNumber == TypeNumber::PlayedApplicationUserState ?
            reinterpret_cast<const PlayedApplicationUserState*>(data) : nullptr;
    }

    /**
     * @brief data が指す種類がこの構造体の内容になる場合、この構造体の型に変換して返します。
     * @return 種類が一致する場合は変換された data の値、それ以外は nullptr
     */
    static PlayedApplicationUserState* FromDataBlock(PlayStateDataBlock* data) NN_NOEXCEPT
    {
        return data->typeNumber == TypeNumber::PlayedApplicationUserState ?
            reinterpret_cast<PlayedApplicationUserState*>(data) : nullptr;
    }

    /**
     * @brief data が指す内容をこの構造体の内容で初期化します。
     * @return 初期化済みの、data を変換したポインター
     */
    static PlayedApplicationUserState* InitializeData(PlayStateDataBlock* data) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<PlayedApplicationUserState*>(data);
        p->InitializeData();
        return p;
    }

    /**
     * @brief この構造体の内容を初期化します。
     */
    void InitializeData() NN_NOEXCEPT
    {
        this->typeNumber = TypeNumber::PlayedApplicationUserState;
        std::memset(this->_padding1, 0, sizeof(this->_padding1));
        this->flags.Reset();
        this->timeStarted = nn::time::PosixTime{ 0 };
        this->elapsedTime = nn::TimeSpan(0);
        this->applicationId = nn::ncm::ApplicationId::GetInvalidId();
        this->uid = nn::account::InvalidUid;
        std::memset(this->_padding2, 0, sizeof(this->_padding2));
    }

    /**
     * @brief 有効なデータを持っているかどうかを返します。
     */
    bool IsValidData() const NN_NOEXCEPT
    {
        return this->typeNumber == TypeNumber::PlayedApplicationUserState;
    }

    /**
     * @brief 保持しているデータを無効化します。
     * @post IsValidData() == false
     */
    void InvalidateData() NN_NOEXCEPT
    {
        reinterpret_cast<PlayStateDataBlock*>(this)->ClearData();
    }

    /**
     * @brief 保持しているデータが指すUidが指定のものと同じかどうかを返します。
     */
    bool IsEqualUid(const nn::account::Uid& u) const NN_NOEXCEPT
    {
        return this->uid == u;
    }

    /**
     * @brief 保持しているデータが全体の状態を指すデータかどうかを返します。
     */
    bool IsGlobalStatusData() const NN_NOEXCEPT
    {
        return this->applicationId == nn::ncm::ApplicationId::GetInvalidId();
    }

    /**
     * @brief 保持しているデータが指すアプリケーションIDが指定のものと同じかどうかを返します。
     */
    bool IsEqualApplicationId(const nn::ncm::ApplicationId& appId) const NN_NOEXCEPT
    {
        return this->applicationId == appId;
    }

    /**
     * @brief ユーザーがオープン状態かどうかを返します。
     */
    bool IsOpened() const NN_NOEXCEPT
    {
        return this->flags.Test<PlayedApplicationUserStateFlags::Opened>();
    }

    /**
     * @brief ユーザーの状態をオープン状態に設定します。
     * @details 同一の uid であった場合、applicationId はそのままの状態を保持します。
     */
    void SetOpened(const nn::account::Uid& u, const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(u, nn::account::InvalidUid);
        if (this->uid != u)
        {
            this->elapsedTime = nn::TimeSpan();
            this->flags.Reset();
        }
        if (!IsOpened())
        {
            this->uid = u;
            this->timeStarted = timeAt;
            this->flags.Set<PlayedApplicationUserStateFlags::Opened>(true);
            this->flags.Set<PlayedApplicationUserStateFlags::Suspended>(false);
        }
    }

    /**
     * @brief ユーザーの状態を非オープン状態に設定します。
     */
    void SetClosed(const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        if (IsOpened())
        {
            // 中断している場合は稼働時間に含めない
            if (!IsSuspended())
            {
                auto elapsed = (timeAt - this->timeStarted);
                // 終了時刻が過去になる場合は経過時間 0 とする
                if (elapsed < nn::TimeSpan(0))
                {
                    elapsed = nn::TimeSpan(0);
                }
                this->elapsedTime += elapsed;
            }
            this->flags.Set<PlayedApplicationUserStateFlags::Opened>(false);
        }
    }

    /**
     * @brief アプリケーションが中断状態であれば true を返します。
     * @pre IsValidData() == true
     */
    bool IsSuspended() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        return this->flags.Test<PlayedApplicationUserStateFlags::Suspended>();
    }

    /**
     * @brief アプリケーションの状態を中断状態に設定します。
     * @pre IsValidData() == true
     * @details IsRunning() == false や IsSuspended() == true、IsGlobalStatusData() == true の場合は何もしません。
     */
    void SetSuspended(const nn::time::PosixTime& timeSuspended) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        if (IsOpened() && !IsSuspended() && !IsGlobalStatusData())
        {
            auto elapsed = (timeSuspended - this->timeStarted);
            // 中断時刻が過去になる場合は経過時間 0 とする
            if (elapsed < nn::TimeSpan(0))
            {
                elapsed = nn::TimeSpan(0);
            }
            this->elapsedTime += elapsed;
            this->timeStarted = timeSuspended;
            this->flags.Set<PlayedApplicationUserStateFlags::Suspended>(true);
        }
    }

    /**
     * @brief アプリケーションの状態を非中断状態に設定します。
     * @pre IsValidData() == true && IsGlobalStatusData() == false
     * @details IsRunning() == false や IsSuspended() == false、IsGlobalStatusData() == true の場合は何もしません。
     */
    void SetResumed(const nn::time::PosixTime& timeResumed) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        if (IsOpened() && IsSuspended() && !IsGlobalStatusData())
        {
            this->timeStarted = timeResumed;
            this->flags.Set<PlayedApplicationUserStateFlags::Suspended>(false);
        }
    }

    /**
     * @brief ユーザーのプレイ時間を返します。
     * @pre IsValidData() == true
     * @details
     * ユーザーが Open 状態であれば、Open した時刻と timeNow を考慮した値を返します。
     */
    nn::TimeSpan GetPlayingTime(const nn::time::PosixTime& timeNow) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        return (IsOpened() && !IsSuspended() ? (timeNow - this->timeStarted) : nn::TimeSpan()) + nn::TimeSpan(this->elapsedTime);
    }

    /**
     * @brief スリープなどで除外される時間を適用して補正します。
     * @pre IsValidData() == true
     */
    void AdjustIgnoreTime(const nn::time::PosixTime& ignoreStartTime, const nn::time::PosixTime& ignoreEndTime) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        if (!IsOpened())
        {
            return;
        }
        if (ignoreStartTime < this->timeStarted)
        {
            // 開始前から除外が始まっていた場合は除外区間終了時刻を開始時刻に設定
            this->timeStarted = ignoreEndTime;
        }
        else
        {
            // 除外する時間分だけ開始時刻を調整
            this->timeStarted += ignoreEndTime - ignoreStartTime;
        }
    }

    /**
    * @brief アプリケーションの稼働開始時刻をリセットします。
    * @pre IsValidData() == true
    * @details 経過時間もリセットされます。
    */
    void ResetStartTime(const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValidData());
        if (IsOpened())
        {
            if (!IsSuspended())
            {
                this->timeStarted = timeAt;
            }
            this->elapsedTime = nn::TimeSpan(0);
        }
    }
};
NN_STATIC_ASSERT(sizeof(PlayedApplicationUserState) == sizeof(PlayStateDataBlock));

////////////////////////////////////////////////////////////////////////////////

/**
 * @brief PlayStateDataBlock ベースのデータを列挙するためのテンプレートクラスです。
 * @tparam TBase PlayStateDataBlockEnumerator の親クラス(後述)
 * @tparam TDataBlockType PlayStateDataBlock ベースのデータ型
 * @details
 * TBase は「bool IsTargetDataBlock(PlayStateDataBlock* data) const」を実装する必要があります。
 * このメソッドは列挙しようとしているデータが実際の列挙対象のデータかどうかを判定する際に用いられます。
 */
template <class TBase, class TDataBlockType>
class PlayStateDataBlockEnumerator : public TBase
{
public:
    PlayStateDataBlockEnumerator() NN_NOEXCEPT :
        TBase(),
        m_Parent(nullptr),
        m_Current(nullptr)
    {
    }

    NN_IMPLICIT PlayStateDataBlockEnumerator(const PlayStateDataBlockEnumerator<TBase, TDataBlockType>& enumerator) NN_NOEXCEPT :
        TBase(enumerator),
        m_Parent(enumerator.m_Parent),
        m_Current(enumerator.m_Current)
    {
    }
    explicit PlayStateDataBlockEnumerator(PlayState* parent) NN_NOEXCEPT :
        m_Parent(parent),
        m_Current(nullptr)
    {
    }
    PlayStateDataBlockEnumerator<TBase, TDataBlockType>& operator = (const PlayStateDataBlockEnumerator<TBase, TDataBlockType>& enumerator) NN_NOEXCEPT
    {
        TBase::operator = (enumerator);
        m_Parent = enumerator.m_Parent;
        m_Current = enumerator.m_Current;
        return *this;
    }

public:
    bool HasCurrent() const NN_NOEXCEPT
    {
        return m_Current != nullptr;
    }
    TDataBlockType* GetCurrent() const NN_NOEXCEPT
    {
        return reinterpret_cast<TDataBlockType*>(m_Current);
    }
    bool Next() NN_NOEXCEPT;
    void MoveToFirst() NN_NOEXCEPT;

private:
    bool IsTargetDataBlock(PlayStateDataBlock* data) const NN_NOEXCEPT
    {
        return TBase::IsTargetDataBlock(data);
    }

private:
    PlayState* m_Parent;
    PlayStateDataBlock* m_Current;
};

/**
 * @brief PlayedApplicationState を列挙するために必要な処理を持つクラス
 */
class PlayStatePlayedApplicationEnumeratorBase
{
protected:
    PlayStatePlayedApplicationEnumeratorBase() NN_NOEXCEPT {}
    NN_IMPLICIT PlayStatePlayedApplicationEnumeratorBase(const PlayStatePlayedApplicationEnumeratorBase& ) NN_NOEXCEPT {}

    bool IsTargetDataBlock(PlayStateDataBlock* data) const NN_NOEXCEPT
    {
        return PlayedApplicationState::IsTypeMatch(data);
    }
};
/**
 * @brief アプリケーションデータを列挙するためのクラスです。
 */
typedef PlayStateDataBlockEnumerator<PlayStatePlayedApplicationEnumeratorBase, PlayedApplicationState> PlayStatePlayedApplicationEnumerator;

/**
 * @brief PlayedApplicationUserState を列挙するために必要な処理を持つクラス
 */
class PlayStatePlayedUserEnumeratorBase
{
public:
    /**
     * @brief 列挙の対象とする PlayedApplicationUserState が持つアプリケーションIDを指定します。
     * @details 既定では IsGlobalStatusData() == true となるデータが列挙されます。
     */
    void SetTargetApplicationId(const nn::ncm::ApplicationId& applicationId) NN_NOEXCEPT
    {
        m_TargetApplicationId = applicationId;
    }

    PlayStatePlayedUserEnumeratorBase& operator = (const PlayStatePlayedUserEnumeratorBase& base) NN_NOEXCEPT
    {
        m_TargetApplicationId = base.m_TargetApplicationId;
        return *this;
    }

protected:
    PlayStatePlayedUserEnumeratorBase() NN_NOEXCEPT :
        m_TargetApplicationId(nn::ncm::ApplicationId::GetInvalidId())
    {
    }

    NN_IMPLICIT PlayStatePlayedUserEnumeratorBase(const PlayStatePlayedUserEnumeratorBase& e) NN_NOEXCEPT :
        m_TargetApplicationId(e.m_TargetApplicationId)
    {
    }

    bool IsTargetDataBlock(PlayStateDataBlock* data) const NN_NOEXCEPT
    {
        auto p = PlayedApplicationUserState::FromDataBlock(data); // (PlayedApplicationUserState::IsTypeMatch 判定を含む)
        return p != nullptr && p->IsEqualApplicationId(m_TargetApplicationId);
    }

private:
    nn::ncm::ApplicationId m_TargetApplicationId;
};
/**
 * @brief アプリケーションごとのユーザーデータを列挙するためのクラスです。
 */
typedef PlayStateDataBlockEnumerator<PlayStatePlayedUserEnumeratorBase, PlayedApplicationUserState> PlayStatePlayedUserEnumerator;

////////////////////////////////////////////////////////////////////////////////

/**
 * @brief プレイ状態を示すフラグ型です。
 */
typedef nn::util::BitFlagSet<32, PlayStateFlags> PlayStateFlagSet;

/**
 * @brief プレイ状態を示すフラグの定義です。
 */
struct PlayStateFlags
{
    //! @brief 本体稼働中(電源ON、スリープ中含む)であることを示すフラグです。
    typedef PlayStateFlagSet::Flag<0> DeviceRunning;
    //! @brief プレイタイマーが有効である(稼働している)ことを示すフラグです。
    typedef PlayStateFlagSet::Flag<1> PlayTimerEnabled;
    //! @brief スリープ中であることを示すフラグです。
    typedef PlayStateFlagSet::Flag<2> Sleeping;
    //! @brief 一時解除状態であることを示すフラグです。Sleeping がONのときは発生しません。
    typedef PlayStateFlagSet::Flag<3> TemporaryUnlocked;
};

/**
 * @brief 実行状態を管理するデータです。
 * @details 本構造体のインスタンスはそのままファイルへの入出力に使われることを想定しています。
 */
struct PlayState
{
    /**
     * @brief PlayedApplicationState を保持する最大数です。
     */
    static const int MaxPlayedApplicationStateCount = 32;
    static const int MaxDataBlockCount = 288; // 32タイトル * (1 + 8 ユーザー) 程度

    /**
     * @brief 最後に状態更新がなされた(その状態が発生した)時点での日付です。
     * @details 原則として UpdatePlayStateFromEvent でのイベントの日付ですが、日付変更を検知した場合は一時的にその日付になります。
     */
    nn::time::CalendarTime lastDate;
    // [暗黙的paddingあり(int8_t 分)]
    /**
     * @brief 最後に状態更新がなされた(その状態が発生した)時点での時刻です。
     * @details 原則として UpdatePlayStateFromEvent でのイベントの時刻ですが、日付変更を検知した場合は一時的にその時刻になります。
     */
    nn::time::PosixTime lastTimestamp;
    int64_t lastElapsedTimeSeconds;

    /**
     * @brief アプリケーションを起動していない/中断している期間の開始時刻です。
     */
    nn::time::PosixTime miscTimeStart;
    /**
     * @brief アプリケーションを起動していない/中断している期間のこれまでの累計時間です。
     */
    nn::TimeSpanType miscElapsedTime;

    /**
     * @brief スリープや一時解除などの時間を計上しない期間の開始時刻です。
     */
    nn::time::PosixTime ignoreTimeStart;
    /**
     * @brief スリープや一時解除などの時間を計上しない期間のこれまでの累計時間です。
     */
    nn::TimeSpanType ignoreElapsedTime;

    /**
     * @brief 一時解除が行われた時刻です。
     */
    nn::time::PosixTime temporaryUnlockedTimeStart;
    /**
     * @brief 一時解除していた期間のこれまでの累計時間です。
     */
    nn::TimeSpanType temporaryUnlockedElapsedTime;

    /**
     * @brief プレイタイマーが最後に停止した時刻です。
     * @details デバイス起動時にはその時刻が設定されます。
     */
    nn::time::PosixTime playTimerStoppedTime;

    PlayStateFlagSet playStateFlags;
    int32_t _padding;

    /**
     * @brief 今までに実行した/実行しているアプリケーションの状態を保持するデータの配列です。
     */
    //PlayedApplicationState playedApplications[MaxPlayedApplicationStateCount];
    PlayStateDataBlock data[MaxDataBlockCount];

    /**
     * @brief 全データを初期化します。
     */
    void InitializeData() NN_NOEXCEPT;

    /**
     * @brief data 内の各アプリケーションの実行状態データ、および各ユーザーの Open 状態データをすべてリセットします。
     * @details
     * - アプリケーションの状態データについては以下の処理が行われます。
     *   - 実行されていないことを示すデータについては無効状態に設定
     *   - 実行中を示すデータについては稼働時間をリセットして開始時刻を指定した時刻に再設定
     * - ユーザーごとの状態データについては以下の処理が行われます。
     *   - Open されていないことを示すデータについては無効状態に設定
     *   - Open されているを示すデータについては稼働時間をリセットして開始時刻を指定した時刻に再設定
     * 定期的なタイミングなどでデータをリフレッシュする際に利用します。
     */
    void ResetPlayedStatus(const nn::time::PosixTime& timeNow) NN_NOEXCEPT;

    /**
     * @brief アプリケーションの状態を保持しているデータを
     *        指定したアプリケーションIDに基づいて返します。
     * @return PlayedApplicationState のポインター、または見つからない場合 nullptr
     */
    const PlayedApplicationState* FindApplicationStateData(const nn::ncm::ApplicationId& applicationId) const NN_NOEXCEPT;

    /**
     * @brief アプリケーションの状態を保持しているデータを
     *        指定したアプリケーションIDに基づいて返します。
     * @return PlayedApplicationState のポインター、または見つからない場合 nullptr
     */
    PlayedApplicationState* FindApplicationStateData(const nn::ncm::ApplicationId& applicationId) NN_NOEXCEPT
    {
        typedef std::remove_pointer<decltype(this)>::type ThisType;
        // const 版と同じ処理になるので型変換で対処
        return const_cast<PlayedApplicationState*>(
            const_cast<const ThisType*>(this)->FindApplicationStateData(applicationId)
            );
    }

    /**
     * @brief アプリケーションの状態を保持するために利用可能なデータを、
     *        指定したアプリケーションIDに基づき PlayedApplicationState に変換して返します。
     * @return PlayedApplicationState のポインター、または見つからない(記録できない)場合 nullptr
     * @details
     * 既にデータが最大数記録されている場合、本メソッドは記録不可として nullptr を返します。
     */
    PlayedApplicationState* FindApplicationStateDataForRegister(const nn::ncm::ApplicationId& applicationId) NN_NOEXCEPT;

    /**
     * @brief アプリケーションに関わらない全体のユーザーの状態を保持しているデータを
     *        指定した Uid に基づいて返します。
     * @return PlayedApplicationUserState のポインター、または見つからない場合 nullptr
     * @post
     * - 戻り値が非 nullptr である場合、そのインスタンスに対する IsGlobalStatusData() == true
     */
    const PlayedApplicationUserState* FindApplicationUserGlobalStateData(const nn::account::Uid& uid) const NN_NOEXCEPT;

    /**
     * @brief アプリケーションに関わらない全体のユーザーの状態を保持しているデータを
     *        指定した Uid に基づいて返します。
     * @return PlayedApplicationUserState のポインター、または見つからない場合 nullptr
     * @post
     * - 戻り値が非 nullptr である場合、そのインスタンスに対する IsGlobalStatusData() == true
     */
    PlayedApplicationUserState* FindApplicationUserGlobalStateData(const nn::account::Uid& uid) NN_NOEXCEPT
    {
        typedef std::remove_pointer<decltype(this)>::type ThisType;
        // const 版と同じ処理になるので型変換で対処
        return const_cast<PlayedApplicationUserState*>(
            const_cast<const ThisType*>(this)->FindApplicationUserGlobalStateData(uid)
            );
    }

    /**
     * @brief アプリケーションに関わらない全体のユーザーの状態を保持するために利用可能なデータを、
     *        指定した Uid に基づき PlayedApplicationUserState に変換して返します。
     * @return PlayedApplicationUserState のポインター、または見つからない(記録できない)場合 nullptr
     * @post
     * - 戻り値が非 nullptr である場合、そのインスタンスに対する IsGlobalStatusData() == true
     * @details
     * 既にデータが最大数記録されている場合、本メソッドは記録不可として nullptr を返します。
     */
    PlayedApplicationUserState* FindApplicationUserGlobalStateDataForRegister(const nn::account::Uid& uid) NN_NOEXCEPT;

    /**
     * @brief アプリケーションごとのユーザーの状態を保持するために利用可能なデータを、
     *        指定した Uid に基づき PlayedApplicationUserState に変換して返します。
     * @return PlayedApplicationUserState のポインター、または見つからない(記録できない)場合 nullptr
     * @details
     * 既にデータが最大数記録されている場合、本メソッドは記録不可として nullptr を返します。
     */
    PlayedApplicationUserState* FindApplicationUserStateDataForRegister(const nn::account::Uid& uid, const nn::ncm::ApplicationId& applicationId) NN_NOEXCEPT;

    /**
     * @brief アプリケーションの状態を実行状態として設定します。
     * @param[in] applicationId アプリケーションを指す ApplicationId
     * @param[in] timeStarted 実行状態になった時刻
     * @pre applicationId != nn::ncm::ApplicationId::GetInvalidId()
     * @details
     * 内部で FindApplicationStateDataForRegister(applicationId)->SetStarted(applicationId, timeStarted) を行います。
     * (ただし FindApplicationStateDataForRegister() == nullptr の場合は何も行いません。)
     */
    void SetApplicationStarted(const nn::ncm::ApplicationId& applicationId, const nn::time::PosixTime& timeStarted) NN_NOEXCEPT;

    /**
     * @brief アプリケーションの状態を停止状態として設定します。
     * @pre applicationId != nn::ncm::ApplicationId::GetInvalidId()
     * @details
     * 内部で FindApplicationStateData(applicationId)->SetStopped(timeStarted) を行います。
     * (ただし FindApplicationStateData() == nullptr や、戻り値が指定した applicationId のものと異なるデータの場合は何も行いません。)
     */
    void SetApplicationStopped(const nn::ncm::ApplicationId& applicationId, const nn::time::PosixTime& timeStopped) NN_NOEXCEPT;

    /**
     * @brief アプリケーションの状態を中断状態に設定します。
     * @pre applicationId != nn::ncm::ApplicationId::GetInvalidId()
     * @details
     * 内部で FindApplicationStateData(applicationId)->SetStopped(timeStarted) を行います。
     * (ただし FindApplicationStateData() == nullptr や、戻り値が指定した applicationId のものと異なるデータの場合は何も行いません。)
     */
    void SetApplicationSuspended(const nn::ncm::ApplicationId& applicationId, const nn::time::PosixTime& timeSuspended) NN_NOEXCEPT;

    /**
     * @brief アプリケーションの状態を非中断状態に設定します。
     * @pre applicationId != nn::ncm::ApplicationId::GetInvalidId()
     * @details
     * 内部で FindApplicationStateData(applicationId)->SetStopped(timeStarted) を行います。
     * (ただし FindApplicationStateData() == nullptr や、戻り値が指定した applicationId のものと異なるデータの場合は何も行いません。)
     */
    void SetApplicationResumed(const nn::ncm::ApplicationId& applicationId, const nn::time::PosixTime& timeResumed) NN_NOEXCEPT;

    /**
     * @brief 実行状態になっているすべてのアプリケーションを停止状態に設定します。
     */
    void SetAllApplicationStopped(const nn::time::PosixTime& timeStopped) NN_NOEXCEPT;

    /**
     * @brief プレイタイマー稼動中かどうかを返します。
     */
    bool IsPlayTimerEnabled() const NN_NOEXCEPT
    {
        return playStateFlags.Test<PlayStateFlags::PlayTimerEnabled>();
    }

    /**
     * @brief プレイタイマー稼動状態を変更します。
     */
    void SetPlayTimerEnabled(bool enabled, const nn::time::PosixTime& timeAt) NN_NOEXCEPT;

    /**
     * @brief スリープ中かどうかを返します。
     */
    bool IsSleeping() const NN_NOEXCEPT
    {
        return playStateFlags.Test<PlayStateFlags::Sleeping>();
    }

    /**
    * @brief 一時解除状態を変更します。
    * @pre sleeping != IsSleeping() のとき、IsTemporaryUnlocked() == false
    */
    void SetSleeping(bool sleeping, const nn::time::PosixTime& timeAt) NN_NOEXCEPT;

    /**
     * @brief 一時解除状態かどうかを返します。
     */
    bool IsTemporaryUnlocked() const NN_NOEXCEPT
    {
        return playStateFlags.Test<PlayStateFlags::TemporaryUnlocked>();
    }

    /**
     * @brief 一時解除状態を変更します。
     * @pre temporaryUnlocked != IsTemporaryUnlocked() のとき、IsSleeping() == false
     */
    void SetTemporaryUnlocked(bool temporaryUnlocked, const nn::time::PosixTime& timeAt) NN_NOEXCEPT;

    /**
     * @brief デバイス稼働状態かどうかを返します。
     */
    bool IsDeviceRunning() const NN_NOEXCEPT
    {
        return playStateFlags.Test<PlayStateFlags::DeviceRunning>();
    }

    /**
     * @brief デバイス稼働状態を変更します。
     */
    void SetDeviceRunning(bool isRunning) NN_NOEXCEPT
    {
        if (isRunning == IsDeviceRunning())
        {
            return;
        }
        playStateFlags.Set<PlayStateFlags::DeviceRunning>(isRunning);
    }

    /**
     * @brief デバイス稼働状態を変更し、それに伴う追加処理を行います。
     * @details 例として、isRunning == false の場合はアプリケーション停止状態へ移行などを行います。
     */
    void ProcessDeviceRunningStatus(bool isRunning, const nn::time::PosixTime& timeAt, bool isDayChanged) NN_NOEXCEPT;

    /**
     * @brief 時間計測を無視する期間に入っているかどうかを返します。
     */
    bool IsIgnorableTime() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!(IsSleeping() && IsTemporaryUnlocked()));
        return !IsPlayTimerEnabled() || IsTemporaryUnlocked() || IsSleeping();
    }

    /**
     * @brief アプリケーションが実行していない/中断している期間の累計時間を返します。
     * @details
     * アプリケーションが実行していない/中断している期間が継続中の場合は timeNow を考慮した値を返します。
     */
    nn::TimeSpan GetTemporaryUnlockedTime(const nn::time::PosixTime& timeNow) const NN_NOEXCEPT
    {
        // 一時解除中ではない場合はこれまでの経過時間のみ返す
        if (!IsTemporaryUnlocked())
        {
            return nn::TimeSpan(this->temporaryUnlockedElapsedTime);
        }
        else
        {
            nn::time::PosixTime timeNowActual = timeNow;
            // プレイタイマー停止中は停止された時刻を「現在時刻」とすることで、それまでの時間が得られる
            if (!IsPlayTimerEnabled())
            {
                timeNowActual = playTimerStoppedTime;
            }
            return (timeNowActual - this->temporaryUnlockedTimeStart) + nn::TimeSpan(this->temporaryUnlockedElapsedTime);
        }
    }

    /**
     * @brief ユーザーの状態を Open 状態として設定します。
     * @param[in] uid ユーザーを指す Uid
     * @param[in] timeAt Open 状態になった時刻
     * @pre uid != nn::account::InvalidUid
     */
    void SetUserOpened(const nn::account::Uid& uid, const nn::time::PosixTime& timeAt) NN_NOEXCEPT;

    /**
     * @brief ユーザーの状態を Close 状態として設定します。
     * @param[in] uid ユーザーを指す Uid
     * @param[in] timeAt 非 Open 状態になった時刻
     * @pre uid != nn::account::InvalidUid
     */
    void SetUserClosed(const nn::account::Uid& uid, const nn::time::PosixTime& timeAt) NN_NOEXCEPT;

    /**
     * @brief 全ユーザーの状態を Close 状態として設定します。
     * @param[in] timeAt 非 Open 状態になった時刻
     */
    void SetAllUserClosed(const nn::time::PosixTime& timeAt) NN_NOEXCEPT;

    /**
     * @brief アプリケーションが実行していない/中断している期間が開始した状態を設定します。
     */
    void SetMiscTimeStarted(const nn::time::PosixTime& timeAt) NN_NOEXCEPT;

    /**
     * @brief アプリケーションが実行していない/中断している期間が終了した状態を設定します。
     */
    void SetMiscTimeFinished(const nn::time::PosixTime& timeAt) NN_NOEXCEPT;

    /**
     * @brief アプリケーションが実行していない/中断している期間の累計時間を返します。
     * @details
     * アプリケーションが実行していない/中断している期間が継続中の場合は timeNow を考慮した値を返します。
     */
    nn::TimeSpan GetMiscTime(const nn::time::PosixTime& timeNow) const NN_NOEXCEPT
    {
        // アプリケーション起動中やデバイスが起動していない場合はこれまでの経過時間のみ返す
        if (!IsDeviceRunning() || IsAnyApplicationRunning())
        {
            return nn::TimeSpan(this->miscElapsedTime);
        }
        else
        {
            nn::time::PosixTime timeNowActual = timeNow;
            // 無効期間中はそれを開始した時刻を「現在時刻」とすることで、それまでの時間が得られる
            if (IsIgnorableTime())
            {
                timeNowActual = ignoreTimeStart;
            }
            return (timeNowActual - this->miscTimeStart) + nn::TimeSpan(this->miscElapsedTime);
        }
    }

    /**
     * @brief アプリケーションが実行中かどうかを返します。
     * @details 「実行中」には中断状態のものを含みません。
     */
    bool IsAnyApplicationRunning() const NN_NOEXCEPT;

    /**
     * @brief スリープなどで除外される時間を適用して補正します。
     */
    void AdjustIgnoreTime(const nn::time::PosixTime& ignoreStartTime, const nn::time::PosixTime& ignoreEndTime) NN_NOEXCEPT;

    /**
     * @brief ある時点でのアプリケーションの稼働時間を返します。
     * @details 本インスタンスが保持している状態は、簡単のため timeAt の時点までの状態が適用されているものとみなします。
     */
    nn::TimeSpan GetApplicationRunningTime(const PlayedApplicationState& state, const nn::time::PosixTime& timeAt) const NN_NOEXCEPT;

    /**
     * @brief ある時点でのアプリケーションにおけるユーザー非選択状態でのプレイ時間を返します。
     * @details 本インスタンスが保持している状態は、簡単のため timeAt の時点までの状態が適用されているものとみなします。
     */
    nn::TimeSpan GetApplicationAnonymousPlayingTime(const PlayedApplicationState& state, const nn::time::PosixTime& timeAt) const NN_NOEXCEPT;

    /**
     * @brief ある時点でのユーザーのプレイ時間を返します。
     * @details 本インスタンスが保持している状態は、簡単のため timeAt の時点までの状態が適用されているものとみなします。
     */
    nn::TimeSpan GetUserPlayingTime(const PlayedApplicationUserState& state, const nn::time::PosixTime& timeAt) const NN_NOEXCEPT;

    /**
     * @brief イベントデータをもとに状態を更新します。
     * @return 更新が行われた場合は true、行われなかった場合(変更なし)は false
     */
    bool UpdatePlayStateFromEvent(const watcher::EventData& eventData) NN_NOEXCEPT;

    /**
     * @brief アプリケーションデータを列挙するためのデータを返します。
     */
    PlayStatePlayedApplicationEnumerator GetPlayedApplicationEnumerator() NN_NOEXCEPT
    {
        PlayStatePlayedApplicationEnumerator e(this);
        e.MoveToFirst();
        return e;
    }

    /**
     * @brief アプリケーションごとのユーザーデータを列挙するためのデータを返します。
     */
    PlayStatePlayedUserEnumerator GetPlayedUserEnumerator(nn::ncm::ApplicationId targetApplicationId) NN_NOEXCEPT
    {
        PlayStatePlayedUserEnumerator e(this);
        e.SetTargetApplicationId(targetApplicationId);
        e.MoveToFirst();
        return e;
    }

private:
    /**
     * @brief 時間計測を無視する期間かどうかに影響するフラグの書き換えを行います。
     * @tparam TBitFlag 書き換えるフラグの型(PlayStateFlags::*)
     * @tparam BoolValueToIgnorableTime flag がどちらの値の時が無視する期間になるかどうか
     * @param[in] flag セットするフラグのON/OFF値
     * @param[in] timeAt 発生時刻
     */
    template <class TBitFlag, bool BoolValueToIgnorableTime>
    inline void SetFlagForIgnorableTime(bool flag, const nn::time::PosixTime& timeAt) NN_NOEXCEPT
    {
        if (flag == BoolValueToIgnorableTime)
        {
            // フラグの書き換えによってはじめて無視する期間に入る場合はセットする
            // (playStateFlags を更新する前にチェックする)
            if (!IsIgnorableTime())
            {
                ignoreTimeStart = timeAt;
            }
            playStateFlags.Set<TBitFlag>(flag);
        }
        else
        {
            playStateFlags.Set<TBitFlag>(flag);
            // フラグの書き換えによって無視する期間にならなく場合は時間調整する
            // (playStateFlags を更新してからチェックする)
            if (!IsIgnorableTime())
            {
                AdjustIgnoreTime(ignoreTimeStart, timeAt);
                ignoreElapsedTime += (timeAt - ignoreTimeStart);
            }
        }
    }

    /**
     * @brief 指定時刻を利用してプレイ時間(elapsedTime)の更新を行います。
     */
    bool UpdateElapsedTime(const nn::time::PosixTime& timeAt) NN_NOEXCEPT;
};
NN_STATIC_ASSERT(std::is_pod<PlayState>::value);

/**
 * @brief 計算用に用いる一時的な PlayState データのブロックを取得します。
 * @details WatcherEventManager::Initialize および UploadDeviceEvents 内でのみ使用します。
 */
PlayState* GetTempPlayStateData() NN_NOEXCEPT;

/**
 * @brief PlayState へのデータのアクセスに対して排他制御を行うクラスです。
 * @details このクラスはコピーできません。
 */
class PlayStateHolder
{
    NN_DISALLOW_COPY(PlayStateHolder);
private:
    PlayStateHolder() NN_NOEXCEPT; // disallow
    PlayStateHolder(PlayState* pState, nn::os::SdkMutexType* pMutex) NN_NOEXCEPT :
        m_pState(pState), m_pMutex(pMutex)
    {
        NN_SDK_REQUIRES_NOT_NULL(pMutex);
        m_pMutex->lock();
    }

public:
    NN_IMPLICIT PlayStateHolder(PlayStateHolder&& holder) NN_NOEXCEPT :
        m_pState(std::move(holder.m_pState)),
        m_pMutex(std::move(holder.m_pMutex))
    {
        holder.m_pState = nullptr;
        holder.m_pMutex = nullptr;
    }
    ~PlayStateHolder() NN_NOEXCEPT
    {
        if (m_pMutex != nullptr)
        {
            m_pMutex->unlock();
        }
    }
    PlayState* Get() const NN_NOEXCEPT
    {
        return m_pState;
    }
    NN_IMPLICIT operator PlayState*() const NN_NOEXCEPT
    {
        return m_pState;
    }
    PlayState* operator ->() const NN_NOEXCEPT
    {
        return m_pState;
    }

    /**
     * @brief PlayStateHolder のインスタンスを取得します。
     * @details PlayStateHolder のインスタンスが有効な間は排他制御されます(他の Acquire 呼び出しをブロックします)。
     */
    static PlayStateHolder Acquire() NN_NOEXCEPT;

private:
    PlayState* m_pState;
    nn::os::SdkMutexType* m_pMutex;
};

////////////////////////////////////////////////////////////////////////////////

template <class TBase, class TDataBlockType>
bool PlayStateDataBlockEnumerator<TBase, TDataBlockType>::Next() NN_NOEXCEPT
{
    if (m_Current == nullptr)
    {
        return false;
    }
    for (++m_Current; m_Current < m_Parent->data + PlayState::MaxDataBlockCount; ++m_Current)
    {
        if (IsTargetDataBlock(m_Current))
        {
            return true;
        }
    }
    m_Current = nullptr;
    return false;
}

template <class TBase, class TDataBlockType>
void PlayStateDataBlockEnumerator<TBase, TDataBlockType>::MoveToFirst() NN_NOEXCEPT
{
    m_Current = m_Parent->data;
    while (NN_STATIC_CONDITION(true))
    {
        // 末尾まで到達したら nullptr を設定して抜ける
        if (m_Current >= m_Parent->data + PlayState::MaxDataBlockCount)
        {
            m_Current = nullptr;
            break;
        }
        // 対象が見つかったら抜ける
        if (IsTargetDataBlock(m_Current))
        {
            break;
        }
        ++m_Current;
    }
}

}}}}
