﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/pctl/detail/ipc/pctl_IParentalControlService.sfdl.h>
#include <nn/pctl/detail/service/pctl_ServiceMain.h>
#include <nn/pctl/detail/service/common/pctl_AsyncContext.h>

#include <nn/os/os_SystemEvent.h>

namespace nn { namespace pctl { namespace detail { namespace service {

// in pctl_ServiceWatcher.h
class PairingInfoDataHolder;

/**
 * @brief 各APIの実装を持つサービスインターフェイス(IParentalControlService)の実装
 */
class ParentalControlServiceImpl
{
public:
    /**
     * @brief コンストラクターです。
     * @param[in] capability 利用可能な機能を示す値(@ref ipc::Capability の各値の組み合わせ)
     */
    explicit ParentalControlServiceImpl(int capability) NN_NOEXCEPT;
    ~ParentalControlServiceImpl() NN_NOEXCEPT;

    nn::Result InternalInitialize(nn::Bit64 processId) NN_NOEXCEPT;

    nn::Result Initialize() NN_NOEXCEPT;
    nn::Result CheckFreeCommunicationPermission() NN_NOEXCEPT;
    nn::Result ConfirmLaunchApplicationPermission(nn::ncm::ApplicationId applicationId, const nn::sf::InArray<std::int8_t>& ratingAge, bool freeCommunicationFlag) NN_NOEXCEPT;
    nn::Result ConfirmResumeApplicationPermission(nn::ncm::ApplicationId applicationId, const nn::sf::InArray<std::int8_t>& ratingAge, bool freeCommunicationFlag) NN_NOEXCEPT;
    nn::Result ConfirmSnsPostPermission() NN_NOEXCEPT;
    nn::Result ConfirmSystemSettingsPermission() NN_NOEXCEPT;
    nn::Result IsRestrictionTemporaryUnlocked(nn::sf::Out<bool> outValue) NN_NOEXCEPT;
    nn::Result RevertRestrictionTemporaryUnlocked() NN_NOEXCEPT;
    nn::Result EnterRestrictedSystemSettings() NN_NOEXCEPT;
    nn::Result LeaveRestrictedSystemSettings() NN_NOEXCEPT;
    nn::Result IsRestrictedSystemSettingsEntered(nn::sf::Out<bool> outValue) NN_NOEXCEPT;
    nn::Result RevertRestrictedSystemSettingsEntered() NN_NOEXCEPT;
    nn::Result GetRestrictedFeatures(nn::sf::Out<int> outFeatures) NN_NOEXCEPT;
    nn::Result ConfirmPlayableApplicationVideo(nn::ncm::ApplicationId applicationId, const nn::sf::InArray<std::int8_t>& ratingAge) NN_NOEXCEPT;
    nn::Result IsRestrictionEnabled(nn::sf::Out<bool> outValue) NN_NOEXCEPT;
    nn::Result GetSafetyLevel(nn::sf::Out<int> outSafetyLevel) NN_NOEXCEPT;
    nn::Result SetSafetyLevel(int level) NN_NOEXCEPT;
    nn::Result GetSafetyLevelSettings(nn::sf::Out<nn::pctl::SafetyLevelSettings> pSettings, int level) NN_NOEXCEPT;
    nn::Result GetCurrentSettings(nn::sf::Out<nn::pctl::SafetyLevelSettings> pSettings) NN_NOEXCEPT;
    nn::Result SetCustomSafetyLevelSettings(nn::pctl::SafetyLevelSettings pSettings) NN_NOEXCEPT;
    nn::Result GetDefaultRatingOrganization(nn::sf::Out<int> outOrganization) NN_NOEXCEPT;
    nn::Result SetDefaultRatingOrganization(int organization) NN_NOEXCEPT;
    nn::Result DeleteSettings() NN_NOEXCEPT;
    nn::Result GetFreeCommunicationApplicationListCount(nn::sf::Out<int> outCount) NN_NOEXCEPT;
    nn::Result GetFreeCommunicationApplicationList(nn::sf::Out<int> outCount, const nn::sf::OutArray<nn::pctl::FreeCommunicationApplicationInfo>& pOutTitleInfoArray, int offset) NN_NOEXCEPT;
    nn::Result UpdateFreeCommunicationApplicationList(const nn::sf::InArray<nn::pctl::FreeCommunicationApplicationInfo>& pTitleInfoArray) NN_NOEXCEPT;
    nn::Result AddToFreeCommunicationApplicationList(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;
    nn::Result DisableFeaturesForReset() NN_NOEXCEPT;
    nn::Result NotifyApplicationDownloadStarted(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;
    // ForDebug
    nn::Result DeleteFromFreeCommunicationApplicationListForDebug(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;
    nn::Result ClearFreeCommunicationApplicationListForDebug() NN_NOEXCEPT;
    nn::Result GetExemptApplicationListCountForDebug(nn::sf::Out<int> outCount) NN_NOEXCEPT;
    nn::Result GetExemptApplicationListForDebug(nn::sf::Out<int> outCount, const nn::sf::OutArray<nn::pctl::ExemptApplicationInfo>& pOutTitleInfoArray, int offset) NN_NOEXCEPT;
    nn::Result UpdateExemptApplicationListForDebug(const nn::sf::InArray<nn::pctl::ExemptApplicationInfo>& pTitleInfoArray) NN_NOEXCEPT;
    nn::Result AddToExemptApplicationListForDebug(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;
    nn::Result DeleteFromExemptApplicationListForDebug(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;
    nn::Result ClearExemptApplicationListForDebug() NN_NOEXCEPT;

    nn::Result ConfirmStereoVisionPermission() NN_NOEXCEPT;
    nn::Result ResetConfirmedStereoVisionPermission() NN_NOEXCEPT;
    nn::Result IsStereoVisionPermitted(nn::sf::Out<bool> outValue) NN_NOEXCEPT;
    nn::Result ConfirmStereoVisionRestrictionConfigurable() NN_NOEXCEPT;
    nn::Result GetStereoVisionRestriction(nn::sf::Out<bool> outRestricted) NN_NOEXCEPT;
    nn::Result SetStereoVisionRestriction(bool restricted) NN_NOEXCEPT;

    nn::Result UnlockRestrictionTemporarily(const nn::sf::InArray<char>& code) NN_NOEXCEPT;
    nn::Result UnlockSystemSettingsRestriction(const nn::sf::InArray<char>& code) NN_NOEXCEPT;
    nn::Result SetPinCode(const nn::sf::InArray<char>& code) NN_NOEXCEPT;
    nn::Result GenerateInquiryCode(nn::sf::Out<nn::pctl::InquiryCode> pOutCodeData) NN_NOEXCEPT;
    nn::Result CheckMasterKey(nn::sf::Out<bool> outValue, const nn::pctl::InquiryCode& codeData, const nn::sf::InArray<char>& masterKey) NN_NOEXCEPT;
    nn::Result GetPinCodeLength(nn::sf::Out<int> outValue) NN_NOEXCEPT;
    nn::Result GetPinCodeChangedEvent(nn::sf::Out<nn::sf::NativeHandle> pEvent) NN_NOEXCEPT;
    nn::Result GetPinCode(nn::sf::Out<int> outLength, const nn::sf::OutArray<char>& outCode) NN_NOEXCEPT;

    nn::Result IsPairingActive(nn::sf::Out<bool> outValue) NN_NOEXCEPT;
    nn::Result GetSettingsLastUpdated(nn::sf::Out<nn::time::PosixTime> outValue) NN_NOEXCEPT;
    nn::Result DeletePairing() NN_NOEXCEPT;
    nn::Result GetPairingAccountInfo(nn::sf::Out<nn::pctl::detail::PairingAccountInfoBase> pAccountInfo, const nn::pctl::detail::PairingInfoBase& pairingInfo) NN_NOEXCEPT;
    nn::Result GetAccountNickname(nn::sf::Out<std::uint32_t> pActualSize, const nn::sf::OutArray<char>& pNickname, const nn::pctl::detail::PairingAccountInfoBase& accountInfo) NN_NOEXCEPT;
    nn::Result GetAccountState(nn::sf::Out<int> outState, const nn::pctl::detail::PairingAccountInfoBase& accountInfo) NN_NOEXCEPT;
    nn::Result GetSynchronizationEvent(nn::sf::Out<nn::sf::NativeHandle> pEvent) NN_NOEXCEPT;
    nn::Result StartPlayTimer() NN_NOEXCEPT;
    nn::Result StopPlayTimer() NN_NOEXCEPT;
    nn::Result IsPlayTimerEnabled(nn::sf::Out<bool> outValue) NN_NOEXCEPT;
    nn::Result GetPlayTimerRemainingTime(nn::sf::Out<nn::TimeSpanType> outTime) NN_NOEXCEPT;
    nn::Result IsRestrictedByPlayTimer(nn::sf::Out<bool> outValue) NN_NOEXCEPT;
    nn::Result GetPlayTimerSettings(nn::sf::Out<nn::pctl::PlayTimerSettings> pSettings) NN_NOEXCEPT;
    nn::Result GetPlayTimerEventToRequestSuspension(nn::sf::Out<nn::sf::NativeHandle> pEvent) NN_NOEXCEPT;
    nn::Result IsPlayTimerAlarmDisabled(nn::sf::Out<bool> outValue) NN_NOEXCEPT;
    nn::Result SetPlayTimerSettingsForDebug(const nn::pctl::PlayTimerSettings& settings) NN_NOEXCEPT;
    nn::Result SetPlayTimerAlarmDisabledForDebug(bool isDisabled) NN_NOEXCEPT;
    nn::Result GetPlayTimerSpentTimeForTest(nn::sf::Out<nn::TimeSpanType> outTime) NN_NOEXCEPT;

    nn::Result NotifyWrongPinCodeInputManyTimes() NN_NOEXCEPT;
    nn::Result CancelNetworkRequest() NN_NOEXCEPT;
    nn::Result GetUnlinkedEvent(nn::sf::Out<nn::sf::NativeHandle> pEvent) NN_NOEXCEPT;
    nn::Result ClearUnlinkedEvent() NN_NOEXCEPT;

    nn::Result DisableAllFeatures(nn::sf::Out<bool> outIsAlreadyDisabled) NN_NOEXCEPT;
    nn::Result PostEnableAllFeatures(nn::sf::Out<bool> outIsAlreadyEnabled) NN_NOEXCEPT;
    nn::Result IsAllFeaturesDisabled(nn::sf::Out<bool> outValue, nn::sf::Out<bool> outIsEnabledOnNextBoot) NN_NOEXCEPT;

    nn::Result RequestPairingAsync(nn::sf::Out<nn::pctl::detail::AsyncData> pAsyncData, nn::sf::Out<nn::sf::NativeHandle> pEvent, const nn::sf::InArray<char>& code) NN_NOEXCEPT;
    nn::Result FinishRequestPairing(nn::sf::Out<nn::pctl::detail::PairingInfoBase> pInfo, const nn::pctl::detail::AsyncData& asyncData) NN_NOEXCEPT;
    nn::Result AuthorizePairingAsync(nn::sf::Out<nn::pctl::detail::AsyncData> pAsyncData, nn::sf::Out<nn::sf::NativeHandle> pEvent, const nn::pctl::detail::PairingInfoBase& pairingInfo) NN_NOEXCEPT;
    nn::Result FinishAuthorizePairing(nn::sf::Out<nn::pctl::detail::PairingInfoBase> pUpdatedInfo, const nn::pctl::detail::AsyncData& asyncData) NN_NOEXCEPT;
    nn::Result RetrievePairingInfoAsync(nn::sf::Out<nn::pctl::detail::AsyncData> pAsyncData, nn::sf::Out<nn::sf::NativeHandle> pEvent) NN_NOEXCEPT;
    nn::Result FinishRetrievePairingInfo(nn::sf::Out<nn::pctl::detail::PairingInfoBase> pInfo, const nn::pctl::detail::AsyncData& asyncData) NN_NOEXCEPT;
    nn::Result UnlinkPairingAsync(nn::sf::Out<nn::pctl::detail::AsyncData> pAsyncData, nn::sf::Out<nn::sf::NativeHandle> pEvent, bool force) NN_NOEXCEPT;
    nn::Result FinishUnlinkPairing(const nn::pctl::detail::AsyncData& asyncData, bool force) NN_NOEXCEPT;
    nn::Result GetAccountMiiImageAsync(nn::sf::Out<nn::pctl::detail::AsyncData> pAsyncData, nn::sf::Out<nn::sf::NativeHandle> pEvent, nn::sf::Out<std::uint32_t> pActualSize, const nn::sf::OutBuffer& pImage, const nn::pctl::detail::PairingAccountInfoBase& accountInfo) NN_NOEXCEPT;
    nn::Result FinishGetAccountMiiImage(nn::sf::Out<std::uint32_t> pActualSize, const nn::sf::OutBuffer& pImage, const nn::pctl::detail::AsyncData& asyncData) NN_NOEXCEPT;
    nn::Result GetAccountMiiImageContentTypeAsync(nn::sf::Out<nn::pctl::detail::AsyncData> pAsyncData, nn::sf::Out<nn::sf::NativeHandle> pEvent, nn::sf::Out<std::uint32_t> pActualLength, const nn::sf::OutArray<char>& pContentType, const nn::pctl::detail::PairingAccountInfoBase& accountInfo) NN_NOEXCEPT;
    nn::Result FinishGetAccountMiiImageContentType(nn::sf::Out<std::uint32_t> pActualLength, const nn::sf::OutArray<char>& pContentType, const nn::pctl::detail::AsyncData& asyncData) NN_NOEXCEPT;
    nn::Result SynchronizeParentalControlSettingsAsync(nn::sf::Out<nn::pctl::detail::AsyncData> pAsyncData, nn::sf::Out<nn::sf::NativeHandle> pEvent) NN_NOEXCEPT;
    nn::Result RequestUpdateExemptionListAsync(nn::sf::Out<nn::pctl::detail::AsyncData> pAsyncData, nn::sf::Out<nn::sf::NativeHandle> pEvent, nn::ncm::ApplicationId applicationId, bool isExempt) NN_NOEXCEPT;
    nn::Result RequestPostEvents(nn::sf::Out<int> outCount, nn::sf::OutArray<nn::pctl::detail::service::watcher::EventData> pEventData) NN_NOEXCEPT;

    nn::Result FinishSynchronizeParentalControlSettingsWithLastUpdated(nn::sf::Out<nn::time::PosixTime> outLastUpdated, nn::pctl::detail::AsyncData asyncData) NN_NOEXCEPT;

    // deprecated
    nn::Result FinishSynchronizeParentalControlSettings(const nn::pctl::detail::AsyncData& asyncData) NN_NOEXCEPT;
    nn::Result ConfirmPlayableApplicationVideoOld(const nn::sf::InArray<std::int8_t>& ratingAge) NN_NOEXCEPT;

private:
    // 1セッション(= ParentalControlServiceImpl インスタンス)に対し並列に動作させる最大の AsyncContext の数
    static const int MaxAsyncContextConcurrencyPerSession = 2;
    // 非同期処理の管理構造体
    struct AsyncDataInProgress
    {
        struct ArrayData
        {
            common::AsyncContext* pContext;
            PairingInfoDataHolder* pDataHolder;
            int32_t id;
            int32_t reserved;
        };
        // 現在動作している AsyncContext
        ArrayData asyncContextInProgressArray[MaxAsyncContextConcurrencyPerSession];
        // 次に配列にセットする際の対象のインデックス
        // (既に配列の対象位置に AsyncContext が含まれている場合はキャンセルさせる)
        int nextTargetIndex;
    };

    // (pDataHolder は nullptr 可)
    void PushAsyncData(AsyncData* outData, common::AsyncContext* pContext, PairingInfoDataHolder* pDataHolder) NN_NOEXCEPT;
    common::AsyncContext* PopAsyncData(PairingInfoDataHolder** ppDataHolder, const AsyncData& data) NN_NOEXCEPT;

    bool DisableAllFeaturesImpl(bool isRecoveryMode) NN_NOEXCEPT;

    int m_capability;
    PctlApplicationInfo m_TargetApplicationInfo;

    AsyncDataInProgress m_AsyncDataInProgress;
};

}}}}
