﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_Optional.h>
#include <atomic>

namespace nn { namespace pctl { namespace detail { namespace service { namespace json {

enum WriteType : uint8_t
{
    WriteType_Key = 0,
    WriteType_String,
    WriteType_Int64,
    WriteType_Uint64,
    WriteType_Double,
    WriteType_Boolean,
    WriteType_ObjectBegin,
    WriteType_ObjectEnd,
    WriteType_ArrayBegin,
    WriteType_ArrayEnd,
    WriteType_ObjectAny,
    WriteType_ObjectRepeat,

    WriteType_ArrayRepeatValues = 0x40,
    WriteType_OptionalKey = 0x80,
    WriteType_TypeMask = 0x1F
};

/**
 * @brief 決められたJSONデータを出力するために用いるデータ定義用構造体です。
 *        NN_DETAIL_PCTL_JSON_BEGIN_WRITE ～ NN_DETAIL_PCTL_JSON_END_WRITE を用いて定義します。
 */
struct WriteDataDefinition
{
    void* data;
    uint8_t type;
};

/**
 * @brief キーと値を出力するかどうかを判定する際に呼び出されるコールバック関数の型です。
 * @param[out] outKeyName 出力するキー名を受け取るポインター。nullptr を出力するとキーと値が出力されません。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目に当たるかを表す値(開始値は 0)
 * @return 処理が続行可能なら true、エラー等で処理を中断する場合は false
 */
typedef bool (* OutputKeyNameFunction)(const char** outKeyName, void* param, int index);
/**
 * @brief 文字列値を出力するコールバック関数の型です。
 * @param[out] outLength 出力する文字列の長さ(終端のNULL文字を除く)。nn::util::nullopt をセットすると null 値として扱われます。
 * @param[out] outValue 出力する文字列を受け取るポインター。outValue 自体が nullptr の場合は outLength にのみ出力してください。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目に当たるかを表す値(開始値は 0)
 * @return 正しく出力できたら true、それ以外であれば false
 */
typedef bool (* OutputStringValueFunction)(nn::util::optional<size_t>* outLength, const char** outValue, void* param, int index);
/**
 * @brief int64_t 値を出力するコールバック関数の型です。
 * @param[out] outValue 出力する値を受け取るポインター。nn::util::nullopt をセットすると null 値として扱われます。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目に当たるかを表す値(開始値は 0)
 * @return 正しく出力できたら true、それ以外であれば false
 */
typedef bool (* OutputInt64ValueFunction)(nn::util::optional<int64_t>* outValue, void* param, int index);
/**
 * @brief uint64_t 値を出力するコールバック関数の型です。
 * @param[out] outValue 出力する値を受け取るポインター。nn::util::nullopt をセットすると null 値として扱われます。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目に当たるかを表す値(開始値は 0)
 * @return 正しく出力できたら true、それ以外であれば false
 */
typedef bool (* OutputUint64ValueFunction)(nn::util::optional<uint64_t>* outValue, void* param, int index);
/**
 * @brief double 値を出力するコールバック関数の型です。
 * @param[out] outValue 出力する値を受け取るポインター。nn::util::nullopt をセットすると null 値として扱われます。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目に当たるかを表す値(開始値は 0)
 * @return 正しく出力できたら true、それ以外であれば false
 */
typedef bool (* OutputDoubleValueFunction)(nn::util::optional<double>* outValue, void* param, int index);
/**
 * @brief bool 値を出力するコールバック関数の型です。
 * @param[out] outValue 出力する値を受け取るポインター。nn::util::nullopt をセットすると null 値として扱われます。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目に当たるかを表す値(開始値は 0)
 * @return 正しく出力できたら true、それ以外であれば false
 */
typedef bool (* OutputBooleanValueFunction)(nn::util::optional<bool>* outValue, void* param, int index);

/**
 * @brief オブジェクト値を出力する際に呼び出されるコールバック関数の型です。
 * @param[out] outWriteNull オブジェクトの代わりに null を出力するかどうか
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目に当たるかを表す値(開始値は 0)
 * @return 正しく出力できたら true、それ以外であれば false
 */
typedef bool (* ReadyForOutputObjectFunction)(bool* outWriteNull, void* param, int index);
/**
 * @brief 可変オブジェクト値を出力する際に呼び出されるコールバック関数の型です。
 * @param[out] outDefinitionLength outDefinitions に返された配列の長さを受け取るポインター。nn::util::nullopt を返すとオブジェクトの代わりに null 値が出力されます。
 * @param[out] outDefinitions 出力するデータの定義を示す配列を受け取るポインター
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目に当たるかを表す値(開始値は 0)
 * @return 正しく出力できたら true、それ以外であれば false
 */
typedef bool (* ReadyForOutputAnyObjectFunction)(nn::util::optional<int>* outDefinitionLength, const WriteDataDefinition** outDefinitions, void* param, int index);
/**
 * @brief 可変オブジェクト値を出力する際に呼び出されるコールバック関数の型です。キーが必要になるたびに呼び出されます。
 * @param[out] outProcessItem 次の要素を出力するかどうかを受け取るポインター。
 *     itemIndex が 0 のときに nn::util::nullopt をセットすると配列値ではなく null 値を出力します。
 * @param[out] outKeyName 次に出力するキーと値の組み合わせにおけるキー名
 * @param[out] outDefinitionLength outDefinitions に返された配列の長さを受け取るポインター。
 * @param[out] outDefinition 次に出力するキーと値の組み合わせにおける値の内容を定義したデータ配列へのポインター。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目に当たるかを表す値(開始値は 0)
 * @param[in] itemIndex 何個目のキー・値の組み合わせを出力するかを表す値(開始値は 0)
 * @return 正しく出力できたら true、それ以外であれば false
 */
typedef bool (* ReadyForOutputObjectRepeatFunction)(nn::util::optional<bool>* outHasItem, const char** outKeyName, int* outDefinitionLength, const WriteDataDefinition** outDefinition, void* param, int index, int itemIndex);
/**
 * @brief 配列値を出力する際に呼び出されるコールバック関数の型です。
 * @param[out] outWriteNull 配列の代わりに null を出力するかどうか
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目に当たるかを表す値(開始値は 0)
 * @return 正しく出力できたら true、それ以外であれば false
 */
typedef bool (* ReadyForOutputArrayFunction)(bool* outWriteNull, void* param, int index);
/**
 * @brief 配列値を出力する際に呼び出されるコールバック関数の型です。要素が必要になるたびに呼び出されます。
 * @param[out] outProcessItem 次の要素を出力するかどうかを受け取るポインター。
 *     itemIndex が 0 のときに nn::util::nullopt をセットすると配列値ではなく null 値を出力します。
 * @param[in] param 解析開始時に指定される任意のデータ
 * @param[in] index 同じ階層内で何番目に当たるかを表す値(開始値は 0)
 * @param[in] itemIndex 配列内の何番目の要素を出力するかを表す値(開始値は 0)
 * @return 正しく出力できたら true、それ以外であれば false
 */
typedef bool (* ReadyForOutputArrayRepeatFunction)(nn::util::optional<bool>* outHasItem, void* param, int index, int itemIndex);

// コールバック関数の型から値の型を得るテンプレートクラス
template <typename FunctionType> class TypeFromOutputFunction;
template <> class TypeFromOutputFunction<OutputInt64ValueFunction>
{
public:
    typedef int64_t Type;
};
template <> class TypeFromOutputFunction<OutputUint64ValueFunction>
{
public:
    typedef uint64_t Type;
};
template <> class TypeFromOutputFunction<OutputDoubleValueFunction>
{
public:
    typedef double Type;
};
template <> class TypeFromOutputFunction<OutputBooleanValueFunction>
{
public:
    typedef bool Type;
};

/**
 * @brief 決められたJSONデータを出力する際に用いるクラスです。
 *
 * @details
 * 主に以下の点で RapidJSON の Writer と異なります。
 *  - 任意のデータが出力できない(オブジェクトにおける任意キーの出力は一部未対応)
 *  - バッファリングと中断処理に対応
 */
class JsonStructuredWriter
{
public:
    static const int MaxDepth = 32;

    enum class WriteResult : int
    {
        Error = -1,
        NoMoreData = 0,
        Succeeded = 1,
        NeedMoreBuffer = 2
    };

    JsonStructuredWriter(void* param, const WriteDataDefinition* dataArray, int dataCount) NN_NOEXCEPT :
        m_Param(param),
        m_pDataArray(dataArray),
        m_DataCount(dataCount),
        m_CurrentDepth(0),
        m_pCacheBuffer(nullptr),
        m_CacheBufferSize(0),
        m_CacheCurrentOffset(0),
        m_CacheCurrentSize(0)
    {
        InitializeHandler();
    }

    template <size_t N>
    JsonStructuredWriter(void* param, const WriteDataDefinition (& dataArray)[N]) NN_NOEXCEPT :
        m_Param(param),
        m_pDataArray(dataArray),
        m_DataCount(static_cast<int>(N)),
        m_CurrentDepth(0),
        m_pCacheBuffer(nullptr),
        m_CacheBufferSize(0),
        m_CacheCurrentOffset(0),
        m_CacheCurrentSize(0)
    {
        NN_STATIC_ASSERT(N <= INT_MAX);
        InitializeHandler();
    }

    /**
     * @brief Writer の状態を初期状態にします。
     */
    void InitializeHandler() NN_NOEXCEPT
    {
        auto& info = m_DepthInfo[0];
        info.subArray = m_pDataArray;
        info.subArrayLength = m_DataCount;
        info.currentPosition = 0;
        info.index = 0;
        info.isObject = false;
        info.isCommaNecessary = false;
        info.isNextValue = false;
        info.repeatFunction = nullptr;
    }

    /**
     * @brief buffer にデータを書き込みます。
     *
     * @details
     * FillData を繰り返し呼び出すことでデータの書き込みを進めることができます。
     * 既に CalculateTotalSize を呼び出し済みである場合など、最初から
     * 書き込みを行いたい場合は、InitializeHandler を呼び出して
     * Writer を初期状態にする必要があります。
     *
     * なお、出力する値とそれを表現するために必要なデータ(囲み文字など)のサイズが
     * length を超える場合は、既定では出力ができません。
     * その場合、SetCacheBuffer を使って一旦キャッシュに書き出すように設定することで
     * length より大きいデータも分割して出力できるようになります。
     */
    WriteResult FillData(size_t* outLengthWritten, void* buffer, size_t length) NN_NOEXCEPT;
    /**
     * @brief 出力データの合計サイズを計算します。
     *
     * @details
     * CalculateTotalSize 完了時、Writer はデータ末尾まで書き込みを行った状態になります。
     * 続けて FillData を実施する場合は、その前に InitializeHandler を呼び出して
     * Writer をもう一度初期状態にする必要があります。
     */
    WriteResult CalculateTotalSize(size_t* outLength) NN_NOEXCEPT;
    /**
     * @brief データ出力時に値のデータを一旦キャッシュに書き込むようにバッファーを設定します。
     *
     * @details
     * 本メソッドでバッファーをセットした場合、FillData 呼び出し時は
     * 必ずこのバッファーに一旦値が出力されてから出力が行われます。
     * その際、バッファーにセットされたデータが一度に出力できない場合
     * 分割して出力が行われます。
     *
     * pCacheBuffer に nullptr を設定するか、cacheBufferSize に 0 を設定すると
     * キャッシュを使わない設定となります(既定の動作)。
     *
     * キャッシュのバッファーは本クラスのインスタンスが有効な間のみ使用されますが、
     * 自動では解放処理をしないため、呼び出し元で管理する必要があります。
     *
     * キャッシュバッファーを同時に使いまわした場合の動作は未定義です。
     * また、FillData 呼び出しを繰り返している途中でバッファーを
     * 変更した場合の動作も未定義です。
     */
    void SetCacheBuffer(void* pCacheBuffer, size_t cacheBufferSize) NN_NOEXCEPT;

private:
    struct DepthInfo
    {
        const WriteDataDefinition* subArray;
        int subArrayLength;
        int currentPosition;
        int index;
        bool isObject;
        bool isCommaNecessary;
        bool isNextValue;
        NN_PADDING1;
        union
        {
            ReadyForOutputArrayRepeatFunction repeatFunction;
            ReadyForOutputObjectRepeatFunction repeatObjectFunction;
            void* repeatFunctionPtr;
        };
    };

    WriteResult FillDataWithCache(size_t* outLengthWritten, void* buffer, size_t length) NN_NOEXCEPT;
    WriteResult FillDataWithoutCache(size_t* outLengthWritten, void* buffer, size_t length) NN_NOEXCEPT;

    WriteResult WriteCurrentData(size_t* outLength, void* buffer, size_t bufferLength) NN_NOEXCEPT;
    WriteResult WriteData(size_t* outLength, bool* outHasItem, uint8_t type, void* data, void* buffer, size_t bufferLength) NN_NOEXCEPT;
    WriteResult WriteKey(size_t* outLength, const char* keyName, void* buffer, size_t bufferLength) NN_NOEXCEPT;
    WriteResult WriteKeyOptional(size_t* outLength, OutputKeyNameFunction func, void* buffer, size_t bufferLength) NN_NOEXCEPT;
    WriteResult WriteString(size_t* outLength, OutputStringValueFunction func, void* buffer, size_t bufferLength) NN_NOEXCEPT;
    template <typename FunctionType>
    WriteResult WriteSimpleValue(size_t* outLength, FunctionType func, void* buffer, size_t bufferLength) NN_NOEXCEPT;
    WriteResult WriteObjectBegin(size_t* outLength, ReadyForOutputObjectFunction func, void* buffer, size_t bufferLength) NN_NOEXCEPT;
    WriteResult WriteObjectAnyBegin(size_t* outLength, ReadyForOutputAnyObjectFunction func, void* buffer, size_t bufferLength) NN_NOEXCEPT;
    WriteResult WriteObjectRepeatFirst(size_t* outLength, ReadyForOutputObjectRepeatFunction func, void* buffer, size_t bufferLength) NN_NOEXCEPT;
    WriteResult WriteObjectRepeatNext(size_t* outLength, ReadyForOutputObjectRepeatFunction func, void* buffer, size_t bufferLength) NN_NOEXCEPT;
    WriteResult WriteObjectEnd(size_t* outLength, void* buffer, size_t bufferLength) NN_NOEXCEPT;
    WriteResult WriteArrayBegin(size_t* outLength, bool* outHasItem, void* func, bool isRepeat, void* buffer, size_t bufferLength) NN_NOEXCEPT;
    WriteResult WriteArrayEnd(size_t* outLength, void* buffer, size_t bufferLength) NN_NOEXCEPT;

    // startPosition にあるデータに対してその次のチェック位置を計算します。
    // startPosition のデータがObject/Arrayの場合はその終了位置も計算します。
    int CalculateNextPosition(int startPosition) const NN_NOEXCEPT;
    // expectEndType を検索して DpethInfo を更新します。
    // 戻り値が false の場合は見つからなかったことを示します。
    bool PushDepthInfo(const DepthInfo& currentInfo, void* repeatFunctionPtr, WriteType expectEndType) NN_NOEXCEPT;

    void* m_Param;
    const WriteDataDefinition* m_pDataArray;
    int m_DataCount;
    int m_CurrentDepth;
    DepthInfo m_DepthInfo[MaxDepth];

    void* m_pCacheBuffer;
    size_t m_CacheBufferSize;
    size_t m_CacheCurrentOffset;
    size_t m_CacheCurrentSize;
};

}}}}}

/**
 * @brief JSONの期待フォーマット定義を開始します。
 * @param specifier 'static' や 'const' などの指定
 * @param dataName 変数名
 */
#define NN_DETAIL_PCTL_JSON_BEGIN_WRITE(specifier, dataName) \
    specifier ::nn::pctl::detail::service::json::WriteDataDefinition dataName[] = {
/**
 * @brief JSONの期待フォーマット定義を終了します。
 */
#define NN_DETAIL_PCTL_JSON_END_WRITE() \
    };
/**
 * @brief 指定した種類を期待します。
 */
#define NN_DETAIL_PCTL_JSON_WRITE(type, data) \
    { (data), static_cast<uint8_t>((type)) },
/**
 * @brief 指定したキー名を出力します。
 */
#define NN_DETAIL_PCTL_JSON_WRITE_KEY(name) \
    NN_DETAIL_PCTL_JSON_WRITE(::nn::pctl::detail::service::json::WriteType::WriteType_Key, const_cast<char*>((name)))
/**
 * @brief 指定したキー名を出力します。キーと付随する値を省略するかどうかを制御することができます。
 */
#define NN_DETAIL_PCTL_JSON_WRITE_KEY_OPTIONAL(func) \
    NN_DETAIL_PCTL_JSON_WRITE( \
        ::nn::pctl::detail::service::json::WriteType::WriteType_Key | ::nn::pctl::detail::service::json::WriteType::WriteType_OptionalKey, \
        reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::OutputKeyNameFunction>((func))) \
    )
/**
 * @brief 文字列値を出力します。出力時に指定の関数を呼び出します。
 */
#define NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(func) \
    NN_DETAIL_PCTL_JSON_WRITE( \
        ::nn::pctl::detail::service::json::WriteType::WriteType_String, \
        reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::OutputStringValueFunction>((func))) \
    )
/**
 * @brief 数値を出力します。出力時に符号付き64ビット整数として指定の関数を呼び出します。
 */
#define NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(func) \
    NN_DETAIL_PCTL_JSON_WRITE( \
        ::nn::pctl::detail::service::json::WriteType::WriteType_Int64, \
        reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::OutputInt64ValueFunction>((func))) \
    )
/**
 * @brief 数値を出力します。出力時に符号なし64ビット整数として指定の関数を呼び出します。
 */
#define NN_DETAIL_PCTL_JSON_WRITE_VALUE_UINT64(func) \
    NN_DETAIL_PCTL_JSON_WRITE( \
        ::nn::pctl::detail::service::json::WriteType::WriteType_Uint64, \
        reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::OutputUint64ValueFunction>((func))) \
    )
/**
 * @brief 数値を出力します。出力時に倍精度浮動小数点の値として指定の関数を呼び出します。
 */
#define NN_DETAIL_PCTL_JSON_WRITE_VALUE_DOUBLE(func) \
    NN_DETAIL_PCTL_JSON_WRITE( \
        ::nn::pctl::detail::service::json::WriteType::WriteType_Double, \
        reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::OutputDoubleValueFunction>((func))) \
    )
/**
 * @brief bool値を出力します。出力時に指定の関数を呼び出します。
 */
#define NN_DETAIL_PCTL_JSON_WRITE_VALUE_BOOLEAN(func) \
    NN_DETAIL_PCTL_JSON_WRITE( \
        ::nn::pctl::detail::service::json::WriteType::WriteType_Boolean, \
        reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::OutputBooleanValueFunction>((func))) \
    )
/**
 * @brief オブジェクトの開始を出力します。func は nullptr を指定することができます。
 */
#define NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(func) \
    NN_DETAIL_PCTL_JSON_WRITE( \
        ::nn::pctl::detail::service::json::WriteType::WriteType_ObjectBegin, \
        reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::ReadyForOutputObjectFunction>((func))) \
    )
/**
 * @brief オブジェクトを出力します。具体的なオブジェクトの中身はコールバックによって都度決定されます。
 * @details
 * NN_DETAIL_PCTL_JSON_WRITE_OBJECT_ANY の後に NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END は使用できません。
 * (NN_DETAIL_PCTL_JSON_WRITE_OBJECT_ANY は単体で使用します。)
 */
#define NN_DETAIL_PCTL_JSON_WRITE_OBJECT_ANY(func) \
    NN_DETAIL_PCTL_JSON_WRITE( \
        ::nn::pctl::detail::service::json::WriteType::WriteType_ObjectAny, \
        reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::ReadyForOutputAnyObjectFunction>((func))) \
    )
/**
 * @brief オブジェクトを出力します。NN_DETAIL_PCTL_JSON_WRITE_OBJECT_ANY と異なり、データが必要になるたびにコールバックが呼び出されます。
 * @details
 * NN_DETAIL_PCTL_JSON_WRITE_OBJECT_REPEAT の後に NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END は使用できません。
 * (NN_DETAIL_PCTL_JSON_WRITE_OBJECT_REPEAT は単体で使用します。)
 */
#define NN_DETAIL_PCTL_JSON_WRITE_OBJECT_REPEAT(func) \
    NN_DETAIL_PCTL_JSON_WRITE( \
        ::nn::pctl::detail::service::json::WriteType::WriteType_ObjectRepeat, \
        reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::ReadyForOutputObjectRepeatFunction>((func))) \
    )
/**
 * @brief オブジェクトの終了を出力します。
 */
#define NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END() \
    NN_DETAIL_PCTL_JSON_WRITE(::nn::pctl::detail::service::json::WriteType::WriteType_ObjectEnd, nullptr)
/**
 * @brief 配列の開始を出力します。配列に含まれる値は1種類固定(配列内の先頭データを利用)で任意の要素数を受け付けます。
 */
#define NN_DETAIL_PCTL_JSON_WRITE_ARRAY_BEGIN(func) \
    NN_DETAIL_PCTL_JSON_WRITE( \
        ::nn::pctl::detail::service::json::WriteType::WriteType_ArrayBegin | ::nn::pctl::detail::service::json::WriteType::WriteType_ArrayRepeatValues, \
        reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::ReadyForOutputArrayRepeatFunction>((func))) \
    )
/**
 * @brief 配列の開始を出力します。固定長の配列を受け付けます。func は nullptr を指定することができます。
 */
#define NN_DETAIL_PCTL_JSON_WRITE_ARRAY_BEGIN_FIXED(func) \
    NN_DETAIL_PCTL_JSON_WRITE( \
        ::nn::pctl::detail::service::json::WriteType::WriteType_ArrayBegin, \
        reinterpret_cast<void*>(static_cast<::nn::pctl::detail::service::json::ReadyForOutputArrayFunction>((func))) \
    )
/**
 * @brief 配列の終了を出力します。
 */
#define NN_DETAIL_PCTL_JSON_WRITE_ARRAY_END() \
    NN_DETAIL_PCTL_JSON_WRITE(::nn::pctl::detail::service::json::WriteType::WriteType_ArrayEnd, nullptr)
