﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/pctl/detail/service/common/pctl_Cancelable.h>
#include <nn/pctl/detail/service/json/pctl_Json.h>
#include <nn/pctl/detail/service/json/pctl_JsonDataHandler.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace json {

/*!
    @brief      JsonDataHandler をラップするハンドラークラスです。
*/
class JsonDataHandlerProxy
{
public:
    /*!
        @brief      コンストラクタです。
    */
    JsonDataHandlerProxy() NN_NOEXCEPT;

    /*!
        @brief      イベントハンドラを設定します。

        @param[in]  handler イベントハンドラ。

        @details
                    独自のイベントハンドラを設定することで、入力する JSON データに応じたハンドリングを行うことが可能です。
    */
    void SetHandler(JsonDataHandler* handler) NN_NOEXCEPT;

    /*!
        @brief      キャンセル可能オブジェクトを設定します。

        @param[in]  cancelable  キャンセル可能オブジェクト。

        @details
                    キャンセル可能オブジェクトを設定することで、入力処理を中断することが可能です。
    */
    void SetCancelable(const common::Cancelable* cancelable) NN_NOEXCEPT;

    /*!
        @brief      入力ストリームに指定した文字列値の格納バッファに対するオーバーフロー検出器を設定します。

        @param[in]  overflowDetector    オーバーフロー検出器。

        @details
                    オーバーフロー検出器を設定することで、キー名や文字列値のオーバーフローを検出が可能になります。
    */
    void SetOverflowDetector(const std::atomic<bool>* overflowDetector) NN_NOEXCEPT;

public:
    /*!
        @brief      オブジェクトの開始イベントです。

        @return     処理を継続するかどうか。

        @details
                    本関数は、RapidJSON 用関数です。
    */
    bool StartObject() NN_NOEXCEPT;

    /*!
        @brief      オブジェクトの終了イベントです。

        @param[in]  numObjects  オブジェクト数。

        @return     処理を継続するかどうか。

        @details
                    本関数は、RapidJSON 用関数です。
    */
    bool EndObject(RAPIDJSON_NAMESPACE::SizeType numObjects) NN_NOEXCEPT;

    /*!
        @brief      配列の開始イベントです。

        @return     処理を継続するかどうか。

        @details
                    本関数は、RapidJSON 用関数です。
    */
    bool StartArray() NN_NOEXCEPT;

    /*!
        @brief      配列の終了イベントです。

        @param[in]  numElements 要素数。

        @return     処理を継続するかどうか。

        @details
                    本関数は、RapidJSON 用関数です。
    */
    bool EndArray(RAPIDJSON_NAMESPACE::SizeType numElements) NN_NOEXCEPT;

    /*!
        @brief      キーの出現イベントです。

        @param[in]  key     キー。
        @param[in]  length  文字列長。

        @return     処理を継続するかどうか。

        @details
                    本関数は、RapidJSON 用関数です。
    */
    bool Key(const char* key, RAPIDJSON_NAMESPACE::SizeType length, bool) NN_NOEXCEPT;

    /*!
        @brief      値（null）の出現イベントです。

        @return     処理を継続するかどうか。
    */
    bool Null() NN_NOEXCEPT;

    /*!
        @brief      値（bool）の出現イベントです。

        @param[in]  value   値。

        @return     処理を継続するかどうか。
    */
    bool Bool(bool value) NN_NOEXCEPT;

    /*!
        @brief      値（int32_t）の出現イベントです。

        @param[in]  value   値。

        @return     処理を継続するかどうか。
    */
    bool Int(int32_t value) NN_NOEXCEPT;

    /*!
        @brief      値（uint32_t）の出現イベントです。

        @param[in]  value   値。

        @return     処理を継続するかどうか。
    */
    bool Uint(uint32_t value) NN_NOEXCEPT;

    /*!
        @brief      値（int64_t）の出現イベントです。

        @param[in]  value   値。

        @return     処理を継続するかどうか。
    */
    bool Int64(int64_t value) NN_NOEXCEPT;

    /*!
        @brief      値（uint64_t）の出現イベントです。

        @param[in]  value   値。

        @return     処理を継続するかどうか。
    */
    bool Uint64(uint64_t value) NN_NOEXCEPT;

    /*!
        @brief      値（double）の出現イベントです。

        @param[in]  value   値。

        @return     処理を継続するかどうか。
    */
    bool Double(double value) NN_NOEXCEPT;

    /*!
        @brief      値（string）の出現イベントです。

        @param[in]  value   値。
        @param[in]  length  文字列長。

        @return     処理を継続するかどうか。
    */
    bool String(const char* value, RAPIDJSON_NAMESPACE::SizeType length, bool) NN_NOEXCEPT;

protected:
    /*!
        @brief      キャンセルされたかどうかを判定します。

        @return     キャンセルされたかどうか。

        @details
                    @ref SetCancelable でキャンセル可能オブジェクトを設定していた場合、 Cancelable::IsCanceled を呼び出します。@n
                    キャンセル可能オブジェクトが設定されていない場合、常に false を返します。
    */
    bool IsCanceled() const NN_NOEXCEPT;

    /*!
        @brief      文字列値がオーバーフローを起こしているかどうかを判定します。

        @return     オーバーフローを起こしているかどうか。

        @details
                    @ref SetOverflowDetector でオーバーフロー検出器を設定していた場合、値を取得します。@n
                    オーバーフロー検出器が設定されていない場合、常に false を返します。
    */
    bool IsOverflowed() const NN_NOEXCEPT;

private:
    //
    JsonDataHandler* m_Handler;
    //
    const common::Cancelable* m_Cancelable;
    //
    const std::atomic<bool>* m_OverflowDetector;
};

}}}}}
