﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

// include/rapidjson/msinttypes が削除されているため、事前定義が必要。
#define RAPIDJSON_NO_INT64DEFINE

#define RAPIDJSON_NAMESPACE         nne::rapidjson::pctl
#define RAPIDJSON_NAMESPACE_BEGIN   namespace nne { namespace rapidjson { namespace pctl {
#define RAPIDJSON_NAMESPACE_END     }}}

#define RAPIDJSON_ASSERT(x) NN_SDK_ASSERT(x)

#define RAPIDJSON_HAS_CXX11_RVALUE_REFS 1 // NOLINT(preprocessor/const)
#define RAPIDJSON_HAS_CXX11_TYPETRAITS  1 // NOLINT(preprocessor/const)

#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(push)
#pragma warning(disable : 4244)
#pragma warning(disable : 4668)
#pragma warning(disable : 4702)
#endif

#include <rapidjson/reader.h>
#include <rapidjson/writer.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(pop)
#endif

#include <nn/pctl/detail/service/json/pctl_JsonOutputStream.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace json {

/*!
    @brief      メモリの動的確保を禁止したアロケータです。
*/
class NullAllocator
{
public:
    static const bool kNeedFree = false;

public:
    // Do not call.
    void* Malloc(size_t) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(false, "Memory allocation is not allowed.");
        return nullptr;
    }

    // Do not call.
    void* Realloc(void*, size_t, size_t) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(false, "Memory re-allocation is not allowed.");
        return nullptr;
    }

    static void Free(void*) NN_NOEXCEPT
    {
    }
};

/*!
    @brief      固定長のメモリを返すアロケータです。
*/
class FixedSizeAllocator
{
public:
    static const bool kNeedFree = false;

public:
    /*!
        @brief      コンストラクタです。
    */
    FixedSizeAllocator() NN_NOEXCEPT : m_IsAllocated(false)
    {
    }

    void* Malloc(size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(size <= sizeof (m_Buffer));
        NN_UNUSED(size);

        NN_SDK_ASSERT(!m_IsAllocated);

        m_IsAllocated = true;

        return m_Buffer;
    }

    void* Realloc(void* p, size_t oldSize, size_t newSize) NN_NOEXCEPT
    {
        NN_UNUSED(oldSize);

        if (p)
        {
            NN_SDK_ASSERT(p == m_Buffer);
            m_IsAllocated = false;
        }
        return Malloc(newSize);
    }

    static void Free(void*) NN_NOEXCEPT
    {
    }

private:
    //
    char m_Buffer[256];
    //
    bool m_IsAllocated;
};

/*!
    @brief      ライブラリで利用する JsonReader オブジェクトです。
*/
typedef RAPIDJSON_NAMESPACE::GenericReader
    <RAPIDJSON_NAMESPACE::UTF8<>, RAPIDJSON_NAMESPACE::UTF8<>, NullAllocator> JsonReader;

/*!
    @brief      ライブラリで利用する JsonWriter オブジェクトです。
*/
typedef RAPIDJSON_NAMESPACE::Writer
    <JsonMemoryOutputStream, RAPIDJSON_NAMESPACE::UTF8<>, RAPIDJSON_NAMESPACE::UTF8<>, FixedSizeAllocator> JsonWriter;

/*!
    @brief      JSON の整数型です。

    @details
                任意のサイズの整数型が格納されます。
*/
struct JsonInteger
{
    union
    {
        int64_t s;  //!< 64 ビット符号付き整数型。
        uint64_t u; //!< 64 ビット符号なし整数型。
    };
    bool isSigned;
};

}}}}}
